/*
 * Decompiled with CFR 0.152.
 */
package net.morilib.lisp.nano;

import java.math.BigInteger;
import net.morilib.lisp.nano.LispComplex;
import net.morilib.lisp.nano.LispDouble;
import net.morilib.lisp.nano.LispInteger;
import net.morilib.lisp.nano.LispRational;
import net.morilib.lisp.nano.LispReal;
import net.morilib.lisp.nano.LispUtils;

public final class LispMath {
    private static final LispComplex I = LispComplex.newComplex(0.0, 1.0);
    private static final LispComplex TWO_I = LispComplex.newComplex(0.0, 2.0);
    private static final LispComplex N_INF_PI_HALF_I = LispComplex.newComplex(Double.NEGATIVE_INFINITY, -1.5707963267948966);
    private static final Double M_ZERO = new Double(-0.0);
    private static final BigInteger LIMIT_NRT = BigInteger.valueOf(256L);
    private static final BigInteger LIMIT_INT = BigInteger.valueOf(Integer.MAX_VALUE);
    public static final LispDouble INEXACT_PI = new LispDouble(Math.PI);
    public static final LispDouble INEXACT_RAD_PER_DEG = new LispDouble(Math.PI / 180);

    private LispMath() {
    }

    private static LispComplex exptC(LispComplex nm1, LispComplex nm2) {
        double a1 = nm1.getRealDouble();
        double b1 = nm1.getImagDouble();
        double r1 = Math.hypot(a1, b1);
        double t1 = Math.atan2(b1, a1);
        double a2 = nm2.getRealDouble();
        double b2 = nm2.getImagDouble();
        double rr = a2 * Math.log(r1) - t1 * b2;
        double tr = b2 * Math.log(r1) + t1 * a2;
        return LispComplex.newComplex(Math.exp(rr) * Math.cos(tr), Math.exp(rr) * Math.sin(tr));
    }

    public static boolean isMinusZero(double r) {
        return r == 0.0 && M_ZERO.equals(new Double(r));
    }

    public static LispComplex expt(LispComplex nm1, LispComplex nm2) {
        if (nm1.isNaN() || nm2.isNaN()) {
            return LispDouble.NaN;
        }
        if (nm2.isZero()) {
            if (nm1.isExact() && nm2.isExact()) {
                return LispInteger.ONE;
            }
            return LispDouble.ONE;
        }
        if (nm1.isZero()) {
            if (nm1.isExact() && nm2.isExact()) {
                return LispInteger.ZERO;
            }
            return LispDouble.ZERO;
        }
        if (nm1.isOne()) {
            if (nm1.isExact() && nm2.isExact()) {
                return LispInteger.ONE;
            }
            return LispDouble.ONE;
        }
        if (nm2.isOne()) {
            if (nm1.isExact() && nm2.isExact()) {
                return nm1;
            }
            return nm1.toInexact();
        }
        if (!nm2.isReal()) {
            return LispMath.exptC(nm1, nm2);
        }
        if (!nm1.isReal()) {
            return LispMath.expt(nm1, nm2.getRealDouble());
        }
        if (!(nm1.isExact() && nm1.isRational() && nm2.isExact() && nm2.isRational())) {
            if (nm1.getRealDouble() > 0.0) {
                return new LispDouble(Math.pow(nm1.getRealDouble(), nm2.getRealDouble()));
            }
            return LispMath.expt(nm1, nm2.getRealDouble());
        }
        if (!nm2.isInteger()) {
            BigInteger d2;
            BigInteger n2;
            if (((LispReal)nm1).signum() < 0) {
                return LispMath.expt(nm1, nm2.getRealDouble());
            }
            if (((LispReal)nm2).signum() < 0) {
                n2 = ((LispReal)nm2).uminus().getNumerator();
                d2 = ((LispReal)nm2).uminus().getDenominator();
            } else {
                n2 = nm2.getNumerator();
                d2 = nm2.getDenominator();
            }
            if (d2.compareTo(LIMIT_NRT) > 0 || n2.compareTo(LIMIT_INT) > 0) {
                return new LispDouble(Math.pow(nm1.getRealDouble(), nm2.getRealDouble()));
            }
            return new LispDouble(Math.pow(nm1.getRealDouble(), nm2.getRealDouble()));
        }
        if (nm1.isInteger()) {
            LispReal rm2 = nm2.getReal();
            BigInteger b = rm2.getBigInteger();
            Integer b2 = LispUtils.toIntExact(b);
            BigInteger a = nm1.getReal().getBigInteger();
            if (b2 == null) {
                return new LispDouble(Math.pow(nm1.getRealDouble(), nm2.getRealDouble()));
            }
            if (b.signum() > 0) {
                return LispInteger.valueOf(a.pow(b2));
            }
            return LispRational.newRational(BigInteger.ONE, a.pow(-b2.intValue()));
        }
        if (nm1.isRational()) {
            LispReal rm2 = nm2.getReal();
            BigInteger b = rm2.getBigInteger();
            Integer b2 = LispUtils.toIntExact(b);
            BigInteger an = nm1.getReal().getNumerator();
            BigInteger ad = nm1.getReal().getDenominator();
            if (b2 == null) {
                return new LispDouble(Math.pow(nm1.getRealDouble(), nm2.getRealDouble()));
            }
            if (b.signum() > 0) {
                return LispRational.newRational(an.pow(b2), ad.pow(b2));
            }
            return LispRational.newRational(ad.pow(-b2.intValue()), an.pow(-b2.intValue()));
        }
        return LispMath.exptC(nm1, nm2);
    }

    public static LispComplex expt(LispComplex nm1, double r) {
        double a1 = nm1.getRealDouble();
        double b1 = nm1.getImagDouble();
        if (nm1.isNaN() || Double.isNaN(r)) {
            return LispDouble.NaN;
        }
        if (b1 == 0.0 && a1 == 1.0) {
            return LispDouble.ONE;
        }
        if (r == Double.POSITIVE_INFINITY) {
            double r1 = Math.hypot(a1, b1);
            if (r1 < 1.0 && r1 > -1.0) {
                return LispDouble.ZERO;
            }
            if (b1 == 0.0 && a1 > 1.0) {
                return LispDouble.POSITIVE_INFINITY;
            }
            return LispDouble.NaN;
        }
        if (r == Double.NEGATIVE_INFINITY) {
            double r1 = Math.hypot(a1, b1);
            if (r1 > 1.0 || r1 < -1.0) {
                return LispDouble.ZERO;
            }
            if (b1 == 0.0 && a1 < 1.0 && a1 > 0.0) {
                return LispDouble.POSITIVE_INFINITY;
            }
            return LispDouble.NaN;
        }
        if (b1 == 0.0 && a1 > 0.0) {
            return new LispDouble(Math.pow(a1, r));
        }
        double r1 = Math.hypot(a1, b1);
        double t1 = Math.atan2(b1, a1);
        double a2 = r;
        double rr = a2 * Math.log(r1);
        double tr = t1 * a2;
        return LispComplex.newComplex(Math.exp(rr) * Math.cos(tr), Math.exp(rr) * Math.sin(tr));
    }

    public static LispComplex exp(LispComplex n) {
        if (n.isReal()) {
            return new LispDouble(Math.exp(n.getRealDouble()));
        }
        double a = n.getRealDouble();
        double b = n.getImagDouble();
        return LispComplex.newComplex(Math.exp(a) * Math.cos(b), Math.exp(a) * Math.sin(b));
    }

    public static LispComplex log(LispComplex n) {
        if (n.getReal().isZero() && LispMath.isMinusZero(n.getImagDouble())) {
            return N_INF_PI_HALF_I;
        }
        if (n.isReal() && n.getReal().signum() > 0) {
            return new LispDouble(Math.log(n.getRealDouble()));
        }
        double a = n.getRealDouble();
        double b = n.getImagDouble();
        double r = Math.hypot(a, b);
        double t = Math.atan2(b, a);
        return LispComplex.newComplex(Math.log(r), t);
    }

    public static LispComplex sin(LispComplex n) {
        if (n.isReal()) {
            return new LispDouble(Math.sin(n.getRealDouble()));
        }
        if (n.getReal().isZero()) {
            double y = n.getImagDouble();
            double b = Math.sinh(y);
            return LispComplex.newComplex(0.0, b);
        }
        double x = n.getRealDouble();
        double y = n.getImagDouble();
        double a = Math.sin(x) * Math.cosh(y);
        double b = Math.cos(x) * Math.sinh(y);
        return LispComplex.newComplex(a, b);
    }

    public static LispComplex cos(LispComplex n) {
        if (n.isReal()) {
            return new LispDouble(Math.cos(n.getRealDouble()));
        }
        if (n.getReal().isZero()) {
            double y = n.getImagDouble();
            double a = Math.cosh(y);
            return LispComplex.newComplex(a, 0.0);
        }
        double x = n.getRealDouble();
        double y = n.getImagDouble();
        double a = Math.cos(x) * Math.cosh(y);
        double b = -(Math.sin(x) * Math.sinh(y));
        return LispComplex.newComplex(a, b);
    }

    public static LispComplex tan(LispComplex n) {
        if (n.isReal()) {
            return new LispDouble(Math.tan(n.getRealDouble()));
        }
        if (n.getReal().isZero()) {
            double y = n.getImagDouble();
            return LispComplex.newComplex(0.0, Math.tanh(y));
        }
        if (n.getImagDouble() == Double.POSITIVE_INFINITY) {
            double x = n.getRealDouble();
            LispComplex nm = LispComplex.newComplex(Math.tan(x), 1.0);
            LispComplex dn = LispComplex.newComplex(1.0, Math.tan(x));
            return (LispComplex)nm.div(dn);
        }
        if (n.getImagDouble() == Double.NEGATIVE_INFINITY) {
            double x = n.getRealDouble();
            LispComplex nm = LispComplex.newComplex(Math.tan(x), -1.0);
            LispComplex dn = LispComplex.newComplex(1.0, -Math.tan(x));
            return (LispComplex)nm.div(dn);
        }
        return (LispComplex)LispMath.sin(n).div(LispMath.cos(n));
    }

    public static LispComplex asin(LispComplex n) {
        double x = n.getRealDouble();
        double y = n.getImagDouble();
        if (n.isReal() && x >= -1.0 && x <= 1.0) {
            return new LispDouble(Math.asin(n.getRealDouble()));
        }
        if (Double.isInfinite(x) && Double.isInfinite(y)) {
            return LispDouble.NaN;
        }
        if (x == Double.POSITIVE_INFINITY) {
            if (y <= 0.0) {
                x = -x;
            }
            return LispComplex.newComplex(1.5707963267948966, x);
        }
        if (x == Double.NEGATIVE_INFINITY) {
            if (y >= 0.0) {
                x = -x;
            }
            return LispComplex.newComplex(-1.5707963267948966, x);
        }
        if (y == Double.POSITIVE_INFINITY) {
            return LispComplex.newComplex(0.0, y);
        }
        if (y == Double.NEGATIVE_INFINITY) {
            return LispComplex.newComplex(0.0, y);
        }
        LispComplex z1 = (LispComplex)I.mul(n);
        LispComplex z2 = LispMath.expt((LispComplex)LispDouble.ONE.sub(n.mul(n)), 0.5);
        LispComplex z3 = LispMath.log((LispComplex)z1.add(z2));
        return LispComplex.newComplex(z3.getImagDouble(), -z3.getRealDouble());
    }

    public static LispComplex acos(LispComplex n) {
        double x = n.getRealDouble();
        if (n.isReal() && x >= -1.0 && x <= 1.0) {
            return new LispDouble(Math.acos(n.getRealDouble()));
        }
        LispDouble pi2 = new LispDouble(1.5707963267948966);
        return (LispComplex)((LispComplex)pi2).sub(LispMath.asin(n));
    }

    public static LispComplex atan(LispComplex n) {
        if (n.isReal()) {
            return new LispDouble(Math.atan(n.getRealDouble()));
        }
        if (Double.isInfinite(n.getImagDouble())) {
            double im = n.getImagDouble();
            if (Double.isInfinite(n.getRealDouble())) {
                return LispDouble.NaN;
            }
            if (im > 0.0) {
                return new LispDouble(1.5707963267948966);
            }
            return new LispDouble(-1.5707963267948966);
        }
        if (n.getRealDouble() == 0.0 && n.getImagDouble() == 1.0) {
            return LispComplex.newComplex(0.0, Double.POSITIVE_INFINITY);
        }
        if (n.getRealDouble() == 0.0 && n.getImagDouble() == -1.0) {
            return LispComplex.newComplex(0.0, Double.NEGATIVE_INFINITY);
        }
        LispComplex z1 = (LispComplex)LispDouble.ONE.add(I.mul(n));
        LispComplex z2 = (LispComplex)LispDouble.ONE.sub(I.mul(n));
        LispComplex z3 = (LispComplex)LispMath.log(z1).sub(LispMath.log(z2));
        return (LispComplex)z3.div(TWO_I);
    }
}

