/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.nina;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import net.morilib.automata.NFA;
import net.morilib.automata.NFAEdges;
import net.morilib.automata.NFAState;
import net.morilib.automata.TextBound;
import net.morilib.range.Interval;
import net.morilib.range.Range;
import net.morilib.util.IntervalMap;

public class NinaNFA implements NFA<Object, NFAState, Void> {

	private final NFAEdges<Object> deadEdge = new NFAEdges<Object>() {

		@Override
		public Set<NFAState> goNext(Object alphabet) {
			return Collections.emptySet();
		}

		@Override
		public Set<NFAState> goNext(int alphabet) {
			return Collections.emptySet();
		}

		@Override
		public Set<NFAState> goNext(char alphabet) {
			return Collections.emptySet();
		}

		@Override
		public Set<NFAState> goNextEpsilon() {
			return Collections.emptySet();
		}

		@Override
		public Set<? extends Range> nextAlphabets() {
			return Collections.emptySet();
		}

		@Override
		public boolean isNextEpsilon() {
			return false;
		}

	};

	NinaState initial;
	Map<NFAState, NFA<Object, NFAState, Void>> edges;
	Map<NFAState, Void> accept;

	NinaNFA() {
		edges  = new HashMap<NFAState, NFA<Object, NFAState, Void>>();
		accept = new HashMap<NFAState, Void>();
	}

	void linkAlphabet(NFAState b, NFAState e, Object alpha) {
		NFA<Object, NFAState, Void> m;
		NinaSingleNFA n;

		if(b == null || e == null) {
			return;
		}

		if((m = edges.get(b)) == null) {
			m = new NinaSingleNFA(b);
			edges.put(b, m);
		}

		if(m instanceof NinaSingleNFA) {
			((NinaSingleNFA)m).map.put(alpha, e);
		} else if(m instanceof ConcurrentNFA) {
			n = new NinaSingleNFA(b);
			n.map.put(alpha, e);
			((ConcurrentNFA<Object, NFAState, Void>)m).add(n);
		} else {
			throw new ClassCastException();
		}
	}

	void linkAlphabet(NFAState b, NFAState e, Range rng) {
		NFA<Object, NFAState, Void> m;
		NinaSingleNFA n;

		if(b == null || e == null) {
			return;
		}

		if((m = edges.get(b)) == null) {
			m = new NinaSingleNFA(b);
			edges.put(b, m);
		}

		if(m instanceof NinaSingleNFA) {
			((NinaSingleNFA)m).map.put(rng, e);
		} else if(m instanceof ConcurrentNFA) {
			n = new NinaSingleNFA(b);
			n.map.put(rng, e);
			((ConcurrentNFA<Object, NFAState, Void>)m).add(n);
		} else {
			throw new ClassCastException();
		}
	}

	void linkNFA(NFAState b, NFAState e,
			NFA<Object, NFAState, Void> nfa) {
		NFA<Object, NFAState, Void> m;
		ConcurrentNFA<Object, NFAState, Void> p;

		if(b == null || e == null) {
			return;
		}

		if((m = edges.get(b)) == null) {
			p = new ConcurrentNFA<Object, NFAState, Void>(b);
		} else if(m instanceof NinaSingleNFA) {
			p = new ConcurrentNFA<Object, NFAState, Void>(b);
			p.add(m);
		} else if(m instanceof ConcurrentNFA) {
			p = (ConcurrentNFA<Object, NFAState, Void>)m;
		} else {
			throw new ClassCastException();
		}
		p.add(new NFAEpsilonLink<Object, Void>(nfa, e));
		edges.put(b, p);
	}

	@Override
	public boolean isState(NFAState o) {
		return edges.containsKey(o);
	}

	@Override
	public Set<NFAState> getStates(NFAState state, Object alphabet) {
		NFA<Object, NFAState, Void> m;
		Set<NFAState> s, p = new HashSet<NFAState>();

		if((m = edges.get(state)) == null) {
			// do nothing
		} else if((s = m.getStates(state, alphabet)) == null) {
			// do nothing
		} else {
			p.addAll(s);
		}

		for(NFAState t : edges.keySet()) {
			m = edges.get(t);
			p.addAll(m.getStates(state, alphabet));
		}
		return p;
	}

	@Override
	public Set<NFAState> getStates(NFAState state, Range rng) {
		NFA<Object, NFAState, Void> m;
		Set<NFAState> s, p = new HashSet<NFAState>();

		if((m = edges.get(state)) == null) {
			// do nothing
		} else if((s = m.getStates(state, rng)) == null) {
			// do nothing
		} else {
			p.addAll(s);
		}

		for(NFAState t : edges.keySet()) {
			m = edges.get(t);
			p.addAll(m.getStates(state, rng));
		}
		return p;
	}

	@Override
	public Set<NFAState> getStates(NFAState state,
			EnumSet<TextBound> bound) {
		return Collections.emptySet();
	}

	@Override
	public Set<NFAState> getStatesEpsilon(NFAState state) {
		NFA<Object, NFAState, Void> m;
		Set<NFAState> s, p = new HashSet<NFAState>();

		if((m = edges.get(state)) == null) {
			// do nothing
		} else if((s = m.getStatesEpsilon(state)) == null) {
			// do nothing
		} else {
			p.addAll(s);
		}

		for(NFAState t : edges.keySet()) {
			m = edges.get(t);
			p.addAll(m.getStatesEpsilon(state));
		}
		return p;
	}

	@Override
	public Set<NFAState> getStatesBound(NFAState state,
			EnumSet<TextBound> bound) {
		return Collections.emptySet();
	}

	@Override
	public Set<NFAState> getInitialStates() {
		return Collections.<NFAState>singleton(initial);
	}

	@Override
	public boolean isInitialState(NFAState o) {
		return initial.equals(o);
	}

	@Override
	public boolean isFinal(NFAState state) {
		return accept.containsKey(state);
	}

	@Override
	public boolean isFinalAny(Set<NFAState> states) {
		for(NFAState s : states) {
			if(accept.containsKey(s))  return true;
		}
		return false;
	}

	@Override
	public NFAEdges<Object> getEdges(NFAState state) {
		NFA<Object, NFAState, Void> m;

		if((m = edges.get(state)) == null) {
			return deadEdge;
		} else if(m instanceof NinaSingleNFA) {
			return (NinaSingleNFA)m;
		} else if(m instanceof ConcurrentNFA) {
			return ((ConcurrentNFA<Object, NFAState, Void>)m).getEdges(
					state);
		} else {
			throw new RuntimeException();
		}
	}

	@Override
	public Set<Interval> nextAlphabets(NFAState state) {
		IntervalMap<Void> m = new IntervalMap<Void>();
		List<Interval> p = new ArrayList<Interval>();
		NFA<Object, NFAState, Void> n;

		if((n = edges.get(state)) == null) {
			// do nothing
		} else {
			p.addAll(n.nextAlphabets(state));
		}

		for(NFAState t : edges.keySet()) {
			n = edges.get(t);
			p.addAll(n.nextAlphabets(state));
		}
		for(Interval v : p)  m.put(v, null);
		return m.keySet();
	}

	@Override
	public Iterable<Interval> nextAlphabets(Set<NFAState> states) {
		IntervalMap<Void> m = new IntervalMap<Void>();
		List<Interval> p = new ArrayList<Interval>();

		for(NFAState t : states)  p.addAll(nextAlphabets(t));
		for(Interval v : p)  m.put(v, null);
		return m.keySet();
	}

	@Override
	public Set<Object> nextDiscreteAlphabets(NFAState state) {
		NFA<Object, NFAState, Void> m;
		Set<Object> p = new HashSet<Object>();

		if((m = edges.get(state)) == null) {
			// do nothing
		} else {
			p.addAll(m.nextDiscreteAlphabets(state));
		}

		for(NFAState t : edges.keySet()) {
			m = edges.get(t);
			p.addAll(m.nextDiscreteAlphabets(state));
		}
		return p;
	}

	@Override
	public Iterable<Object> nextDiscreteAlphabets(
			Set<NFAState> states) {
		Set<Object> a = new HashSet<Object>();

		for(NFAState s : states) {
			a.addAll(nextDiscreteAlphabets(s));
		}
		return a;
	}

	@Override
	public Set<NFAState> getAcceptedStates() {
		return new HashSet<NFAState>(accept.keySet());
	}

	@Override
	public Set<Void> getMatchTag(NFAState state) {
		return Collections.emptySet();
	}

	@Override
	public Set<Void> getMatchTagEnd(NFAState state) {
		return Collections.emptySet();
	}

	@Override
	public Set<NFAState> getAccept(NFAState state) {
		return accept.containsKey(state) ?
				Collections.<NFAState>singleton(state) :
					Collections.<NFAState>emptySet();
	}

	@Override
	public boolean isAccepted(NFAState state) {
		return accept.containsKey(state);
	}

	public String toString() {
		StringWriter b = new StringWriter();
		PrintWriter p = new PrintWriter(b);

		p.format("Initial: %s\n", initial);
		p.format("accept: %s\n", accept.keySet());
		for(NFAState s : edges.keySet()) {
			p.format("State %s:\n", s);
			p.format("Edges:\n%s\n\n", edges.get(s).toString());
		}
		return b.toString();
	}

}
