# -*- coding: utf-8 -*-
#
#  Copyright (C) 2001, 2002 by Tamito KAJIYAMA
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#  Copyright (C) 2004-2013 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#  Copyright (C) 2003-2005 by Shun-ichi TAHARA <jado@flowernet.gr.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

import email
import os

from gi.repository import Gtk
from gi.repository import GObject

import ninix.home


range_scale = list(range(100, 30, -10)) + [200, 300, 1000]
range_script_speed = list(range(-1, 5)) + [6, 8] # -1: no wait

# default settings
DEFAULT_BALLOON_FONTS = 'Sans'

def get_default_surface_scale():
    return range_scale[0]

def get_default_script_speed():
    return range_script_speed[len(range_script_speed) // 2]


class Preferences(dict):

    def __init__(self, filename):
        dict.__init__(self)
        self.filename = filename
        self.__stack = {}

    def __setitem__(self, key, item):
        if key in self and key not in self.__stack:
            self.__stack[key] = self[key]
        dict.__setitem__(self, key, item)

    def commit(self):
        self.__stack = {}

    def revert(self):
        self.update(self.__stack)
        self.__stack = {}

    def load(self):
        self.clear()
        try:
            with open(self.filename) as f:
                prefs = email.message_from_file(f)
                for key, value in prefs.items():
                    self[key] = value
        except IOError:
            return

    def save(self):
        try:
            os.makedirs(os.path.dirname(self.filename))
        except OSError:
            pass
        with open(self.filename, 'w') as f:
            keys = sorted(self.keys())
            for key in keys:
                if key in self.__stack:
                    value = self.__stack[key]
                else:
                    value = self[key]
                f.write('{0}: {1}\n'.format(key, value))

    def get_with_type(self, name, conv, default):
        value = self.get(name)
        if value:
            if conv is None:
                return value
            try:
                return conv(value)
            except ValueError:
                pass
        return default


class PreferenceDialog:

    PREFS_TYPE = {'sakura_name': None, # XXX: backward compat
                  'sakura_dir': None,
                  'default_balloon': None,
                  'ignore_default': int,
                  'script_speed': int,
                  'surface_scale': int,
                  'balloon_scalling': int,
                  'balloon_fonts': None,
                  'allowembryo': int,
                  'check_collision': int,
                  'check_collision_name': int,
                  'use_pna': int,
                  'sink_after_talk': int,
                  'raise_before_talk': int,
                  'animation_quality': float,
                  }

    def __init__(self):
        self.request_parent = lambda *a: None # dummy
        self.dialog = Gtk.Dialog()
        self.dialog.connect('delete_event', lambda *a: True) # XXX
        self.dialog.set_title('Preferences')
        self.dialog.set_default_size(-1, 600)
        self.notebook = Gtk.Notebook()
        self.notebook.set_tab_pos(Gtk.PositionType.TOP)
        content_area = self.dialog.get_content_area()
        content_area.add(self.notebook)
        self.notebook.show()
        for name, constructor in [
            (_('Surface&Balloon'), self.make_page_surface_n_balloon),
            (_('Misc'),            self.make_page_misc),
            (_('Debug'),           self.make_page_debug),
            ]:
            self.notebook.append_page(constructor(),
                                      Gtk.Label(name))
        self.dialog.add_button(Gtk.STOCK_OK, Gtk.ResponseType.OK)
        self.dialog.add_button(Gtk.STOCK_APPLY, Gtk.ResponseType.APPLY)
        self.dialog.add_button(Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL)
        self.dialog.connect('response', self.response)

    def set_responsible(self, request_method):
        self.request_parent = request_method

    def load(self):
        filename = ninix.home.get_preferences()
        self.__prefs = Preferences(filename)
        self.save = self.__prefs.save
        self.__prefs.load()
        self.reset()
        self.update() # XXX
        self.request_parent('NOTIFY', 'notify_preference_changed')
 
    def reset(self): ### FIXME ###
        self.fontchooser.set_font_name(
            self.get('balloon_fonts', DEFAULT_BALLOON_FONTS))
        self.set_default_balloon(self.get('default_balloon'))
        self.ignore_button.set_active(
            bool(self.get('ignore_default', 0)))
        scale = self.get('surface_scale', get_default_surface_scale())
        self.surface_scale_combo.set_active(
            range_scale.index(get_default_surface_scale() if scale not in range_scale else scale))
        script_speed = self.get('script_speed', get_default_script_speed())
        self.script_speed_combo.set_active(
            range_script_speed.index(get_default_script_speed() if script_speed not in range_script_speed else script_speed))
        self.balloon_scalling_button.set_active(
            bool(self.get('balloon_scalling')))
        self.allowembryo_button.set_active(
            bool(self.get('allowembryo')))
        self.check_collision_button.set_active(
            bool(self.get('check_collision', 0)))
        self.check_collision_name_button.set_active(
            bool(self.get('check_collision_name', 0)))
        self.use_pna_button.set_active(
            bool(self.get('use_pna', 1)))
        self.sink_after_talk_button.set_active(
            bool(self.get('sink_after_talk')))
        self.raise_before_talk_button.set_active(
            bool(self.get('raise_before_talk')))
        self.animation_quality_adjustment.set_value(
            self.get('animation_quality', 1.0))

    def get(self, name, default=None):
        assert name in self.PREFS_TYPE
        return self.__prefs.get_with_type(name, self.PREFS_TYPE[name], default)

    def set_current_sakura(self, directory):
        key = 'sakura_name' # obsolete
        if key in self.__prefs:
            del self.__prefs[key]
        key = 'sakura_dir'
        if key in self.__prefs:
            del self.__prefs[key]
        self.__prefs[key] = directory

    def edit_preferences(self):
        self.show()

    def update(self, commit=False): ## FIXME
        self.__prefs['allowembryo'] = str(int(self.allowembryo_button.get_active()))
        self.__prefs['balloon_fonts'] = self.fontchooser.get_font_name()
        selected = self.balloon_treeview.get_selection().get_selected()
        if selected:
            model, listiter = selected
            directory = model.get_value(listiter, 1)
            self.__prefs['default_balloon'] = directory
        self.__prefs['ignore_default'] = str(int(self.ignore_button.get_active()))
        self.__prefs['surface_scale'] = str(int(range_scale[self.surface_scale_combo.get_active()]))
        self.__prefs['script_speed'] = str(int(range_script_speed[self.script_speed_combo.get_active()]))
        self.__prefs['balloon_scalling'] = str(int(self.balloon_scalling_button.get_active()))
        self.__prefs['check_collision'] = str(int(self.check_collision_button.get_active()))
        self.__prefs['check_collision_name'] = str(int(self.check_collision_name_button.get_active()))
        self.__prefs['use_pna'] = str(int(self.use_pna_button.get_active()))
        self.__prefs['sink_after_talk'] = str(int(self.sink_after_talk_button.get_active()))
        self.__prefs['raise_before_talk'] = str(int(self.raise_before_talk_button.get_active()))
        self.__prefs['animation_quality'] = str(float(self.animation_quality_adjustment.get_value()))
        if commit:
            self.__prefs.commit()

    def ok(self):
        self.hide()
        self.update(commit=True)
        self.request_parent('NOTIFY', 'notify_preference_changed')

    def apply(self):
        self.update()
        self.request_parent('NOTIFY', 'notify_preference_changed')

    def cancel(self):
        self.hide()
        self.__prefs.revert()
        self.reset()
        self.request_parent('NOTIFY', 'notify_preference_changed')

    def show(self):
        self.dialog.show()

    def response(self, widget, response):
        func = {Gtk.ResponseType.OK: self.ok,
                Gtk.ResponseType.CANCEL: self.cancel,
                Gtk.ResponseType.APPLY: self.apply,
                Gtk.ResponseType.DELETE_EVENT: self.cancel,
                }
        func[response]()
        return True

    def hide(self):
        self.dialog.hide()

    def make_page_surface_n_balloon(self):
        page = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=5)
        page.set_border_width(5)
        page.show()
        frame = Gtk.Frame(label=_('Surface Scaling'))
        page.pack_start(frame, False, True, 0)
        frame.show()
        box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=5)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        hbox = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=5)
        box.pack_start(hbox, False, True, 0)
        hbox.show()
        label = Gtk.Label(label=_('Default Setting'))
        hbox.pack_start(label, False, True, 0)
        label.show()
        self.surface_scale_combo = Gtk.ComboBoxText()
        for value in range_scale:
            self.surface_scale_combo.append_text('{0:4d}%'.format(value))
        hbox.pack_start(self.surface_scale_combo, False, True ,0)
        self.surface_scale_combo.show()
        button = Gtk.CheckButton(_('Scale Balloon'))
        self.balloon_scalling_button = button
        box.pack_start(button, False, True, 0)
        button.show()
        frame = Gtk.Frame(label=_('Default Balloon'))
        page.pack_start(frame, True, True, 0)
        frame.show()
        box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=5)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        scrolled = Gtk.ScrolledWindow()
        scrolled.set_vexpand(True)
        scrolled.set_policy(Gtk.PolicyType.NEVER, Gtk.PolicyType.ALWAYS)
        scrolled.set_shadow_type(Gtk.ShadowType.ETCHED_IN)
        box.pack_start(scrolled, True, True, 0)
        scrolled.show()
        treeview = Gtk.TreeView(None)
        column = Gtk.TreeViewColumn(
            _('Balloon Name'), Gtk.CellRendererText(), text=0)
        treeview.append_column(column)
        treeview.get_selection().set_mode(Gtk.SelectionMode.SINGLE)
        self.balloon_treeview = treeview
        scrolled.add(treeview)
        treeview.show()
        button = Gtk.CheckButton(_('Always Use This Balloon'))
        self.ignore_button = button
        box.pack_start(button, False, True, 0)
        button.show()
        frame = Gtk.Frame(label=_('Font(s) for balloons'))
        page.pack_start(frame, False, True, 0)
        frame.show()
        box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=5)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        self.fontchooser = Gtk.FontButton()
        self.fontchooser.set_show_size(False)        
        box.add(self.fontchooser)
        self.fontchooser.show()
        frame = Gtk.Frame(label=_('Translucency'))
        page.pack_start(frame, False, True, 0)
        frame.show()
        box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=5)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        button = Gtk.CheckButton(_('Use PNA file'))
        self.use_pna_button = button
        box.pack_start(button, False, True, 0)
        button.show()
        frame = Gtk.Frame(label=_('Animation'))
        page.pack_start(frame, False, True,0 )
        frame.show()
        box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=5)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        hbox = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=5)
        box.add(hbox)
        hbox.show()
        label = Gtk.Label(label=_('Quality'))
        hbox.pack_start(label, False, True, 0)
        label.show()        
        self.animation_quality_adjustment = Gtk.Adjustment(1.0, 0.4, 1.0, 0.1, 0.1)
        button = Gtk.SpinButton(adjustment=self.animation_quality_adjustment,
                                climb_rate=0.2, digits=1)
        hbox.pack_start(button, False, True, 0)
        button.show()
        hbox.show()
        return page        

    def make_page_misc(self):
        page = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=5)
        page.set_border_width(5)
        page.show()
        frame = Gtk.Frame(label=_('SSTP Setting'))
        page.pack_start(frame, False, True, 0)
        frame.show()
        button = Gtk.CheckButton(_('Allowembryo'))
        self.allowembryo_button = button
        frame.add(button)
        button.show()
        frame = Gtk.Frame(label=_('Script Wait'))
        page.pack_start(frame, False, True, 0)
        frame.show()
        hbox = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=5)
        frame.add(hbox)
        hbox.show()
        label = Gtk.Label(label=_('Default Setting'))
        hbox.pack_start(label, False, True, 0)
        label.show()
        self.script_speed_combo = Gtk.ComboBoxText()
        for index in range(len(range_script_speed)):
            if index == 0:
                label = _('None')
            elif index == 1:
                label = ''.join(('1 (', _('Fast'), ')'))
            elif index == len(range_script_speed) - 1:
                label = ''.join((str(index), ' (', _('Slow'), ')'))
            else:
                label = str(index)
            self.script_speed_combo.append_text(label)
        hbox.pack_start(self.script_speed_combo, False, True, 0)
        self.script_speed_combo.show()
        frame = Gtk.Frame(label=_('Raise & Lower'))
        page.pack_start(frame, False, True, 0)
        frame.show()
        box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=5)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        button = Gtk.CheckButton(_('Sink after Talk'))
        self.sink_after_talk_button = button
        box.pack_start(button, False, True, 0)
        button.show()
        button = Gtk.CheckButton(_('Raise before Talk'))
        self.raise_before_talk_button = button
        box.pack_start(button, False, True, 0)
        button.show()
        return page

    def make_page_debug(self):
        page = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=5)
        page.set_border_width(5)
        page.show()
        frame = Gtk.Frame(label=_('Surface Debugging'))
        page.pack_start(frame, False, True, 0)
        frame.show()
        box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=5)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        button = Gtk.CheckButton(_('Display Collision Area'))
        self.check_collision_button = button
        box.pack_start(button, False, True, 0)
        button.show()
        button = Gtk.CheckButton(_('Display Collision Area Name'))
        self.check_collision_name_button = button
        box.pack_start(button, False, True, 0)
        button.show()
        return page

    def set_default_balloon(self, directory):
        model = Gtk.ListStore(GObject.TYPE_STRING, GObject.TYPE_STRING)
        for name, directory in self.request_parent('GET', 'get_balloon_list'):
            listiter = model.append()
            model.set_value(listiter, 0, name)
            model.set_value(listiter, 1, directory)
        self.balloon_treeview.set_model(model)
        listiter = model.get_iter_first()
        while listiter is not None:
            value = model.get_value(listiter, 1)
            if value == directory or directory is None:
                self.balloon_treeview.get_selection().select_iter(listiter)
                break
            listiter = model.iter_next(listiter) 
        else:
            listiter = model.get_iter_first()
            assert listiter is not None
            self.balloon_treeview.get_selection().select_iter(listiter)
