# -*- coding: utf-8 -*-
#
#  misaka.py - a "美坂" compatible Shiori module for ninix
#  Copyright (C) 2002 by Tamito KAJIYAMA
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#  Copyright (C) 2002-2013 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

# TODO:
# error recovery
# - ${foo}
# - $(foo) ?

import os
import re
import io
import sys
import logging
import time
import random

try:
    import chardet.universaldetector
except:
    chardet = None

from ninix.home import get_normalized_path


class MisakaError(ValueError):
    pass


def lexical_error(path=None, position=None):
    if path is None and position is None:
        error = 'lexical error'
    elif path is None:
        at = 'line {0:d}, column {1:d}'.format(*position)
        error = 'lexical error at {0}'.format(at)
    elif position is None:
        error = 'lexical error in {0}'.format(path)
    else:
        at = 'line {0:d}, column {1:d}'.format(*position)
        error = 'lexical error at {0} in {1}'.format(at, path)
    raise MisakaError(error)

def syntax_error(message, path=None, position=None):
    if path is None and position is None:
        error = 'syntax error ({0})'.format(message)
    elif path is None:
        at = 'line {0:d}, column {1:d}'.format(*position)
        error = 'syntax error at {0} ({1})'.format(at, message)
    elif position is None:
        error = 'syntax error in {0} ({1})'.format(path, message)
    else:
        at = 'line {0:d}, column {1:d}'.format(*position)
        error = 'syntax error at {0} in {1} ({2})'.format(at, path, message)
    raise MisakaError(error)

def list_dict(top_dir):
    buf = []
    try:
        filelist, debug, error = read_misaka_ini(top_dir)
    except Exception as e:
        filelist = []
    for filename in filelist:
        buf.append(os.path.join(top_dir, filename))
    return buf

def read_misaka_ini(top_dir):
    path = os.path.join(top_dir, b'misaka.ini')
    with open(path) as f:
        filelist = []
        debug = 0
        error = 0
        while 1:
            line = f.readline()
            if not line:
                break
            line = line.strip()
            if not line or line.startswith('//'):
                continue
            if line == 'dictionaries':
                line = f.readline()
                if line.strip() != '{':
                    syntax_error('expected an open brace', path)
                while 1:
                    line = f.readline()
                    if not line:
                        syntax_error('unexpected end of file', path)
                    line = line.strip()
                    if line == '}':
                        break
                    if not line or line.startswith('//'):
                        continue
                    filelist.append(get_normalized_path(line))
            elif line == 'debug,0':
                debug = 0
            elif line == 'debug,1':
                debug = 1
            elif line == 'error,0':
                error = 0
            elif line == 'error,1':
                error = 1
            elif line == 'propertyhandler,0':
                pass
            elif line == 'propertyhandler,1':
                pass
            else:
                logging.debug("misaka.py: unknown directive '{0}'".format(line))
    return filelist, debug, error

###   LEXER   ###

TOKEN_NEWLINE       = 1
TOKEN_WHITESPACE    = 2
TOKEN_OPEN_BRACE    = 3
TOKEN_CLOSE_BRACE   = 4
TOKEN_OPEN_PAREN    = 5
TOKEN_CLOSE_PAREN   = 6
TOKEN_OPEN_BRACKET  = 7
TOKEN_CLOSE_BRACKET = 8
TOKEN_DOLLAR        = 9
TOKEN_COMMA         = 11
TOKEN_SEMICOLON     = 12
TOKEN_OPERATOR      = 13
TOKEN_DIRECTIVE     = 14
TOKEN_TEXT          = 15


class Lexer:

    re_comment = re.compile(r'[ \t]*//[^\r\n]*')
    re_newline = re.compile(r'\r\n|\r|\n')
    patterns = [
        (TOKEN_NEWLINE,       re_newline),
        (TOKEN_WHITESPACE,    re.compile(r'[ \t]+')),
        (TOKEN_OPEN_BRACE,    re.compile(r'{')),
        (TOKEN_CLOSE_BRACE,   re.compile(r'}')),
        (TOKEN_OPEN_PAREN,    re.compile(r'\(')),
        (TOKEN_CLOSE_PAREN,   re.compile(r'\)')),
        (TOKEN_OPEN_BRACKET,  re.compile(r'\[')),
        (TOKEN_CLOSE_BRACKET, re.compile(r'\]')),
        (TOKEN_DOLLAR,        re.compile(r'\$')),
        (TOKEN_COMMA,         re.compile(r',')),
        (TOKEN_SEMICOLON,     re.compile(r';')),
        (TOKEN_OPERATOR,      re.compile(r'[!=]=|[<>]=?|=[<>]|&&|\|\||\+\+|--|[+\-*/]?=|[+\-*/%\^]')),
        (TOKEN_DIRECTIVE,     re.compile(r'#[_A-Za-z][_A-Za-z0-9]*')),
        #(TOKEN_TEXT,          re.compile(r"[!&|]|(\\[,\"]|[\x01\x02#'.0-9:?@A-Z\\_`a-z~]|[\x80-\xff].)+")),
        (TOKEN_TEXT,          re.compile(r"(\"[^\"]*\")|[!&|]|(\\[,\"]|[\x01\x02#'.0-9:?@A-Z\\_`a-z~\"]|[" + "\u0080" + "-" + "\uffff" + r"]+)+")),
        ]
    token_names = {
        TOKEN_WHITESPACE:    'whitespace',
        TOKEN_NEWLINE:       'a newline',
        TOKEN_OPEN_BRACE:    'an open brace',
        TOKEN_CLOSE_BRACE:   'a close brace',
        TOKEN_OPEN_PAREN:    'an open paren',
        TOKEN_CLOSE_PAREN:   'a close paren',
        TOKEN_OPEN_BRACKET:  'an open bracket',
        TOKEN_CLOSE_BRACKET: 'a close bracket',
        TOKEN_DOLLAR:        'a dollar',
        TOKEN_COMMA:         'a comma',
        TOKEN_SEMICOLON:     'a semicolon',
        TOKEN_OPERATOR:      'an operator',
        TOKEN_DIRECTIVE:     'an directive',
        TOKEN_TEXT:          'text',
        }

    def __init__(self, charset='CP932'):
        self.buffer = []
        self.charset = charset

    def _match(self, data, line, column, path):
        pos = 0
        end = len(data)
        while pos < end:
            if column == 0:
                match = self.re_comment.match(data, pos)
                if match:
                    column = column + len(match.group())
                    pos = match.end()
                    continue
            for token, pattern in self.patterns:
                match = pattern.match(data, pos)
                if match:
                    lexeme = match.group()
                    if token == TOKEN_TEXT and \
                       lexeme.startswith('"') and lexeme.endswith('"'):
                        if '$' not in lexeme:
                            self.buffer.append((token, lexeme[1:-1], (line, column)))
                        else: # XXX
                            line, column = self._match(lexeme[1:-1], line, column + 1, path)
                            column += 1
                    else:
                        self.buffer.append((token, lexeme, (line, column)))
                    pos = match.end()
                    break
            else:
                ###print(data[pos:pos + 100])
                lexical_error(path, (line, column))
            if token == TOKEN_NEWLINE:
                line = line + 1
                column = 0
            else:
                column = column + len(lexeme)
        return line, column

    def read(self, f, path=None):
        line = 1
        column = 0
        data = f.read()
        line, column = self._match(data, line, column, path)
        self.buffer.append((TOKEN_NEWLINE, '', (line, column)))
        self.path = path

    def get_position(self):
        return self.position

    def pop(self):
        try:
            token, lexeme, self.position = self.buffer.pop(0)
        except IndexError:
            syntax_error('unexpected end of file', self.path)
        ###print(token, repr(lexeme))
        return token, lexeme

    def pop_check(self, expected):
        token, lexeme = self.pop()
        if token != expected:
            syntax_error(''.join(('exptected ', self.token_names[expected], ', but returns ', self.token_names[token])),
                         self.path, self.get_position())
        return lexeme

    def look_ahead(self, index=0):
        try:
            token, lexeme, position = self.buffer[index]
        except IndexError:
            syntax_error('unexpected end of file', self.path)
        return token, lexeme

    def skip_space(self, accept_eof=0):
        while self.buffer:
            token, lexeme = self.look_ahead()
            if token not in [TOKEN_NEWLINE, TOKEN_WHITESPACE]:
                return 0
            self.pop()
        if not accept_eof:
            syntax_error('unexpected end of file', self.path)
        return 1

    def skip_line(self):
        while self.buffer:
            token, lexeme = self.pop()
            if token == TOKEN_NEWLINE:
                break


###   PARSER   ###

NODE_TEXT        = 1
NODE_STRING      = 2
NODE_VARIABLE    = 3
NODE_INDEXED     = 4
NODE_ASSIGNMENT  = 5
NODE_FUNCTION    = 6
NODE_IF          = 7
NODE_CALC        = 8
NODE_AND_EXPR    = 9
NODE_OR_EXPR     = 10
NODE_COMP_EXPR   = 11
NODE_ADD_EXPR    = 12
NODE_MUL_EXPR    = 13
NODE_POW_EXPR    = 14


class Parser:

    def __init__(self, charset):
        self.lexer = Lexer(charset=charset)
        self.charset = charset

    def read(self, f, path=None):
        self.lexer.read(f, path)
        self.path = path

    def get_dict(self):
        common = None
        groups = []
        # skip junk tokens at the beginning of the file
        while 1:
            if self.lexer.skip_space(1):
                return common, groups
            token, lexeme = self.lexer.look_ahead()
            if token == TOKEN_DIRECTIVE and lexeme == '#_Common' or \
               token == TOKEN_DOLLAR:
                break
            self.lexer.skip_line()
        token, lexeme = self.lexer.look_ahead()
        if token == TOKEN_DIRECTIVE and lexeme == '#_Common':
            common = self.get_common()
            if self.lexer.skip_space(1):
                return common, groups
        while 1:
            groups.append(self.get_group())
            if self.lexer.skip_space(1):
                return common, groups
        return None, []

    def get_common(self):
        self.lexer.pop()
        token, lexeme = self.lexer.look_ahead()
        if token == TOKEN_WHITESPACE:
            self.lexer.pop()
        self.lexer.pop_check(TOKEN_NEWLINE)
        condition = self.get_brace_expr()
        self.lexer.pop_check(TOKEN_NEWLINE)
        return condition

    def get_group(self):
        # get group name
        buf = []
        self.lexer.pop_check(TOKEN_DOLLAR)
        while 1:
            token, lexeme = self.lexer.look_ahead()
            ##if token == TOKEN_TEXT or \
            ##   token == TOKEN_OPERATOR and lexeme in ['+', '-', '*', '/', '%']:
            ## XXX
            if token not in [TOKEN_NEWLINE, TOKEN_COMMA, TOKEN_SEMICOLON]:
                token, lexeme = self.lexer.pop()
                buf.append(self.unescape(lexeme))
            else:
                break
        if not buf:
            syntax_error('null identifier',
                         self.path, self.lexer.get_position())
        name = ''.join(('$', ''.join(buf)))
        # get group parameters
        parameters = []
        while 1:
            token, lexeme = self.lexer.look_ahead()
            if token == TOKEN_WHITESPACE:
                self.lexer.pop()
            token, lexeme = self.lexer.look_ahead()
            if token == TOKEN_NEWLINE:
                break
            elif token in [TOKEN_COMMA, TOKEN_SEMICOLON]:
                self.lexer.pop()
            else:
                syntax_error('expected a delimiter',
                             self.path, self.lexer.get_position())
            token, lexeme = self.lexer.look_ahead()
            if token == TOKEN_WHITESPACE:
                self.lexer.pop()
            token, lexeme = self.lexer.look_ahead()
            if token == TOKEN_NEWLINE:
                break
            token, lexeme = self.lexer.look_ahead()
            if token == TOKEN_OPEN_BRACE:
                parameters.append(self.get_brace_expr())
            elif token == TOKEN_TEXT:
                token, lexeme = self.lexer.pop()
                parameters.append([NODE_TEXT, self.unescape(lexeme)])
            else:
                syntax_error('expected a parameter or brace expression',
                             self.path, self.lexer.get_position())
        # get sentences
        self.lexer.pop_check(TOKEN_NEWLINE)
        sentences = []
        while 1:
            if self.lexer.skip_space(1):
                break
            token, lexeme = self.lexer.look_ahead()            
            if token == TOKEN_DOLLAR:
                break
            sentence = self.get_sentence()
            if not sentence:
                continue
            sentences.append(sentence)
        return (name, parameters, sentences)

    def get_sentence(self):
        token, lexeme = self.lexer.look_ahead()
        if token == TOKEN_OPEN_BRACE:
            token, lexeme = self.lexer.look_ahead(1)
            if token == TOKEN_NEWLINE:
                self.lexer.pop_check(TOKEN_OPEN_BRACE)
                token, lexeme = self.lexer.look_ahead()
                if token == TOKEN_WHITESPACE:
                    self.lexer.pop()
                self.lexer.pop_check(TOKEN_NEWLINE)
                line = []
                while 1:
                    self.lexer.skip_space()
                    token, lexeme = self.lexer.look_ahead()
                    if token == TOKEN_CLOSE_BRACE:
                        break
                    for node in self.get_line():
                        line.append(node)
                    self.lexer.pop_check(TOKEN_NEWLINE)
                self.lexer.pop_check(TOKEN_CLOSE_BRACE)
                token, lexeme = self.lexer.look_ahead()
                if token == TOKEN_WHITESPACE:
                    self.lexer.pop()
            else:
                try:
                    line = self.get_line() # beginning with brace expression
                except MisakaError as error:
                    logging.debug(error)
                    self.lexer.skip_line()
                    return None
        else:
            try:
                line = self.get_line()
            except MisakaError as error:
                logging.debug(error)
                self.lexer.skip_line()
                return None
        self.lexer.pop_check(TOKEN_NEWLINE)
        return line

    def is_whitespace(self, node):
        return node[0] == NODE_TEXT and not node[1].strip()

    def unescape(self, text):
        text = text.replace(r'\,', ',')
        text = text.replace(r'\"', '"')
        return text

    def get_word(self):
        buf = []
        while 1:
            token, lexeme = self.lexer.look_ahead()
            if token == TOKEN_TEXT:
                token, lexeme = self.lexer.pop()
                if self.unescape(lexeme).startswith('"') and \
                   self.unescape(lexeme).endswith('"'):
                    buf.append([NODE_STRING, self.unescape(lexeme)[1:-1]])
                else:
                    buf.append([NODE_TEXT, self.unescape(lexeme)])
            elif token in [TOKEN_WHITESPACE, TOKEN_DOLLAR]:
                token, lexeme = self.lexer.pop()
                buf.append([NODE_TEXT, lexeme])
            elif token == TOKEN_OPEN_BRACE:
                buf.append(self.get_brace_expr())
            else:
                break
        # strip whitespace at the beginning and/or end of line
        if buf and self.is_whitespace(buf[0]):
            del buf[0]
        if buf and self.is_whitespace(buf[-1]):
            del buf[-1]
        return buf

    def get_line(self):
        buf = []
        while 1:
            token, lexeme = self.lexer.look_ahead()
            if token in [TOKEN_NEWLINE, TOKEN_CLOSE_BRACE]:
                break
            elif token == TOKEN_TEXT:
                token, lexeme = self.lexer.pop()
                buf.append([NODE_TEXT, self.unescape(lexeme)])
            elif token in [TOKEN_WHITESPACE, TOKEN_DOLLAR,
                           TOKEN_OPEN_PAREN, TOKEN_CLOSE_PAREN,
                           TOKEN_OPEN_BRACKET, TOKEN_CLOSE_BRACKET,
                           TOKEN_OPERATOR, TOKEN_COMMA, TOKEN_SEMICOLON,
                           TOKEN_DIRECTIVE]:
                token, lexeme = self.lexer.pop()
                buf.append([NODE_TEXT, lexeme])
            elif token == TOKEN_OPEN_BRACE:
                buf.append(self.get_brace_expr())
            else:
                raise RuntimeError('should not reach here')
        # strip whitespace at the beginning and/or end of line
        if buf and self.is_whitespace(buf[0]):
            del buf[0]
        if buf and self.is_whitespace(buf[-1]):
            del buf[-1]
        return buf

    def get_brace_expr(self):
        self.lexer.pop_check(TOKEN_OPEN_BRACE)
        self.lexer.skip_space()
        self.lexer.pop_check(TOKEN_DOLLAR)
        self.lexer.skip_space()
        # get identifier (function or variable)
        nodelist = [[NODE_TEXT, '$']]
        while 1:
            token, lexeme = self.lexer.look_ahead()
            if token == TOKEN_TEXT or \
               token == TOKEN_OPERATOR and lexeme in ['+', '-', '*', '/', '%']:
                token, lexeme = self.lexer.pop()
                nodelist.append([NODE_TEXT, self.unescape(lexeme)])
            elif token == TOKEN_OPEN_BRACE:
                nodelist.append(self.get_brace_expr())
            else:
                break
        if len(nodelist) == 1:
            syntax_error('null identifier',
                         self.path, self.lexer.get_position())
        self.lexer.skip_space()
        token, lexeme = self.lexer.look_ahead()
        if token == TOKEN_OPEN_PAREN:
            # function
            name = ''.join([node[1] for node in nodelist])
            if name == '$if':
                cond_expr = self.get_cond_expr()
                then_clause = [[NODE_TEXT, 'true']]
                else_clause = [[NODE_TEXT, 'false']]
                self.lexer.skip_space()
                token, lexeme = self.lexer.look_ahead()
                if token == TOKEN_OPEN_BRACE:
                    self.lexer.pop_check(TOKEN_OPEN_BRACE)
                    self.lexer.skip_space()
                    then_clause = []
                    while 1:
                        line = self.get_line()
                        for node in line:
                            then_clause.append(node)
                        token, lexem = self.lexer.pop()
                        if token == TOKEN_CLOSE_BRACE:
                            break
                        assert (token == TOKEN_NEWLINE)
                    self.lexer.skip_space()
                    token, lexeme = self.lexer.look_ahead()
                    if token == TOKEN_TEXT and lexeme == 'else':
                        self.lexer.pop()
                        self.lexer.skip_space()
                        self.lexer.pop_check(TOKEN_OPEN_BRACE)
                        self.lexer.skip_space()
                        else_clause = []
                        while 1:
                            line = self.get_line()
                            for node in line:
                                else_clause.append(node)
                            token, lexem = self.lexer.pop()
                            if token == TOKEN_CLOSE_BRACE:
                                break
                            assert (token == TOKEN_NEWLINE)
                    elif token == TOKEN_OPEN_BRACE: ## XXX
                        self.lexer.pop_check(TOKEN_OPEN_BRACE)
                        self.lexer.skip_space()
                        else_clause = []
                        while 1:
                            line = self.get_line()
                            for node in line:
                                else_clause.append(node)
                            token, lexem = self.lexer.pop()
                            if token == TOKEN_CLOSE_BRACE:
                                break
                            assert (token == TOKEN_NEWLINE)
                    else:
                        else_clause = [[NODE_TEXT, '']]
                node = [NODE_IF, cond_expr, then_clause, else_clause]
            elif name == '$calc':
                node = [NODE_CALC, self.get_add_expr()]
            else:
                self.lexer.pop_check(TOKEN_OPEN_PAREN)
                self.lexer.skip_space()
                args = []
                token, lexeme = self.lexer.look_ahead()
                if token != TOKEN_CLOSE_PAREN:
                    while 1:
                        args.append(self.get_argument())
                        self.lexer.skip_space()
                        token, lexeme = self.lexer.look_ahead()
                        if token != TOKEN_COMMA:
                            break
                        self.lexer.pop()
                        self.lexer.skip_space()
                self.lexer.pop_check(TOKEN_CLOSE_PAREN)
                node = [NODE_FUNCTION, name, args]
        else:
            # variable
            token, lexeme = self.lexer.look_ahead()
            if token == TOKEN_OPERATOR:
                operator = self.lexer.pop_check(TOKEN_OPERATOR)
                if operator in ['=', '+=', '-=', '*=', '/=']:
                    self.lexer.skip_space()
                    value = self.get_add_expr()
                elif operator == '++':
                    operator = '+='
                    value = [[NODE_TEXT, '1']]
                elif operator == '--':
                    operator = '-='
                    value = [[NODE_TEXT, '1']]
                else:
                    syntax_error(''.join(('bad operator ', operator)),
                                 self.path, self.lexer.get_position())
                node = [NODE_ASSIGNMENT, nodelist, operator, value]
            elif token == TOKEN_OPEN_BRACKET:
                self.lexer.pop_check(TOKEN_OPEN_BRACKET)
                self.lexer.skip_space()
                index = self.get_word()
                self.lexer.skip_space()
                self.lexer.pop_check(TOKEN_CLOSE_BRACKET)
                node = [NODE_INDEXED, nodelist, index]
            else:
                node = [NODE_VARIABLE, nodelist]
        self.lexer.skip_space()
        self.lexer.pop_check(TOKEN_CLOSE_BRACE)
        return node

    def get_argument(self):
        buf = []
        while 1:
            token, lexeme = self.lexer.look_ahead()
            if token == TOKEN_TEXT:
                token, lexeme = self.lexer.pop()
                if self.unescape(lexeme).startswith('"') and \
                   self.unescape(lexeme).endswith('"'):
                    buf.append([NODE_STRING, self.unescape(lexeme)[1:-1]])
                else:
                    buf.append([NODE_TEXT, self.unescape(lexeme)])
            elif token in [TOKEN_WHITESPACE, TOKEN_DOLLAR,
                           TOKEN_OPEN_BRACKET, TOKEN_CLOSE_BRACKET,
                           TOKEN_OPERATOR, TOKEN_SEMICOLON]:
                token, lexeme = self.lexer.pop()
                buf.append([NODE_TEXT, lexeme])
            elif token == TOKEN_OPEN_BRACE:
                buf.append(self.get_brace_expr())
            elif token == TOKEN_NEWLINE:
                self.lexer.skip_space()
            else:
                break
        # strip whitespace at the beginning and/or end of line
        if buf and self.is_whitespace(buf[0]):
            del buf[0]
        if buf and self.is_whitespace(buf[-1]):
            del buf[-1]
        return buf

    def get_cond_expr(self):
        self.lexer.pop_check(TOKEN_OPEN_PAREN)
        self.lexer.skip_space()
        or_expr = self.get_or_expr()
        self.lexer.skip_space()
        self.lexer.pop_check(TOKEN_CLOSE_PAREN)
        return or_expr

    def get_or_expr(self):
        buf = [NODE_OR_EXPR]
        buf.append(self.get_and_expr())
        while 1:
            self.lexer.skip_space()
            token, lexeme = self.lexer.look_ahead()
            if not (token == TOKEN_OPERATOR and lexeme == '||'):
                break
            self.lexer.pop()
            self.lexer.skip_space()
            buf.append(self.get_and_expr())
        return buf if len(buf) > 2 else buf[1]

    def get_and_expr(self):
        buf = [NODE_AND_EXPR]
        buf.append(self.get_sub_expr())
        while 1:
            self.lexer.skip_space()
            token, lexeme = self.lexer.look_ahead()
            if not (token == TOKEN_OPERATOR and lexeme == '&&'):
                break
            self.lexer.pop()
            self.lexer.skip_space()
            buf.append(self.get_sub_expr())
        return buf if len(buf) > 2 else buf[1]

    def get_sub_expr(self):
        token, lexeme = self.lexer.look_ahead()
        if token == TOKEN_OPEN_PAREN:
            return self.get_cond_expr()
        else:
            return self.get_comp_expr()

    def get_comp_expr(self):
        buf = [NODE_COMP_EXPR]
        buf.append(self.get_add_expr())
        self.lexer.skip_space()
        token, lexeme = self.lexer.look_ahead()
        if token == TOKEN_OPERATOR and \
           lexeme in ['==', '!=', '<', '<=', '=<', '>', '>=', '=>']:
            if lexeme == '=<':
                lexeme = '<='
            elif lexeme == '=>':
                lexeme = '>='
            buf.append(lexeme)
            self.lexer.pop()
            self.lexer.skip_space()
            buf.append(self.get_add_expr())
        if len(buf) == 2:
            buf.append('==')
            buf.append([[NODE_TEXT, 'true']])
        return buf

    def get_add_expr(self):
        buf = [NODE_ADD_EXPR]
        buf.append(self.get_mul_expr())
        while 1:
            self.lexer.skip_space()
            token, lexeme = self.lexer.look_ahead()
            if not (token == TOKEN_OPERATOR and lexeme in ['+', '-']):
                break
            buf.append(lexeme)
            self.lexer.pop()
            self.lexer.skip_space()
            buf.append(self.get_mul_expr())
        return [buf] if len(buf) > 2 else buf[1]

    def get_mul_expr(self):
        buf = [NODE_MUL_EXPR]
        buf.append(self.get_pow_expr())
        while 1:
            self.lexer.skip_space()
            token, lexeme = self.lexer.look_ahead()
            if not (token == TOKEN_OPERATOR and lexeme in ['*', '/', '%']):
                break
            buf.append(lexeme)
            self.lexer.pop()
            self.lexer.skip_space()
            buf.append(self.get_pow_expr())
        return [buf] if len(buf) > 2 else buf[1]

    def get_pow_expr(self):
        buf = [NODE_POW_EXPR]
        buf.append(self.get_unary_expr())
        while 1:
            self.lexer.skip_space()
            token, lexeme = self.lexer.look_ahead()
            if not (token == TOKEN_OPERATOR and lexeme == '^'):
                break
            self.lexer.pop()
            self.lexer.skip_space()
            buf.append(self.get_unary_expr())
        return [buf] if len(buf) > 2 else buf[1]

    def get_unary_expr(self):
        token, lexeme = self.lexer.look_ahead()
        if token == TOKEN_OPERATOR and lexeme in ['+', '-']:
            buf = [NODE_ADD_EXPR, [[NODE_TEXT, '0']], lexeme]
            self.lexer.pop()
            self.lexer.skip_space()
            buf.append(self.get_unary_expr())
            return [buf]
        else:
            return self.get_factor()

    def get_factor(self):
        token, lexeme = self.lexer.look_ahead()
        if token == TOKEN_OPEN_PAREN:
            self.lexer.pop_check(TOKEN_OPEN_PAREN)
            self.lexer.skip_space()
            add_expr = self.get_add_expr()
            self.lexer.skip_space()
            self.lexer.pop_check(TOKEN_CLOSE_PAREN)
            return add_expr
        else:
            return self.get_word()

    # for debug
    def dump_list(self, nodelist, depth=0):
        for node in nodelist:
            self.dump_node(node, depth)

    def dump_node(self, node, depth):
        indent = '  ' * depth
        if node[0] == NODE_TEXT:
            print(''.join((indent, 'TEXT')), \
                  ''.join(('"', node[1], '"')))
        elif node[0] == NODE_STRING:
            print(''.join((indent, 'STRING')), \
                  ''.join(('"', node[1], '"')))
        elif node[0] == NODE_VARIABLE:
            print(''.join((indent, 'VARIABLE')))
            self.dump_list(node[1], depth + 1)
        elif node[0] == NODE_INDEXED:
            print(''.join((indent, 'INDEXED')))
            self.dump_list(node[1], depth + 1)
            print(''.join((indent, 'index')))
            self.dump_list(node[2], depth + 1)
        elif node[0] == NODE_ASSIGNMENT:
            print(''.join((indent, 'ASSIGNMENT')))
            self.dump_list(node[1], depth + 1)
            print(''.join((indent, 'op')), node[2])
            self.dump_list(node[3], depth + 1)
        elif node[0] == NODE_FUNCTION:
            print(''.join((indent, 'FUNCTION')), node[1])
            for i in range(len(node[2])):
                print(''.join((indent, 'args[{0:d}]'.format(i))))
                self.dump_list(node[2][i], depth + 1)
        elif node[0] == NODE_IF:
            print(''.join((indent, 'IF')))
            self.dump_node(node[1], depth + 1)
            print(''.join((indent, 'then')))
            self.dump_list(node[2], depth + 1)
            print(''.join((indent, 'else')))
            self.dump_list(node[3], depth + 1)
        elif node[0] == NODE_CALC:
            print(''.join((indent, 'CALC')))
            self.dump_list(node[1], depth + 1)
        elif node[0] == NODE_AND_EXPR:
            print(''.join((indent, 'AND_EXPR')))
            for i in range(1, len(node)):
                self.dump_node(node[i], depth + 1)
        elif node[0] == NODE_OR_EXPR:
            print(''.join((indent, 'OR_EXPR')))
            for i in range(1, len(node)):
                self.dump_node(node[i], depth + 1)
        elif node[0] == NODE_COMP_EXPR:
            print(''.join((indent, 'COMP_EXPR')))
            self.dump_list(node[1], depth + 1)
            print(''.join((indent, 'op')), node[2])
            self.dump_list(node[3], depth + 1)
        elif node[0] == NODE_ADD_EXPR:
            print(''.join((indent, 'ADD_EXPR')))
            self.dump_list(node[1], depth + 1)
            for i in range(2, len(node), 2):
                print(''.join((indent, 'op')), node[i])
                self.dump_list(node[i + 1], depth + 1)
        elif node[0] == NODE_MUL_EXPR:
            print(''.join((indent, 'MUL_EXPR')))
            self.dump_list(node[1], depth + 1)
            for i in range(2, len(node), 2):
                print(''.join((indent, 'op')), node[i])
                self.dump_list(node[i + 1], depth + 1)
        elif node[0] == NODE_POW_EXPR:
            print(''.join((indent, 'POW_EXPR')))
            self.dump_list(node[1], depth + 1)
            for i in range(2, len(node)):
                print(''.join((indent, 'op ^')))
                self.dump_list(node[i], depth + 1)
        else:
            print(node)
            raise RuntimeError('should not reach here')

# <<< syntax >>>
# dict       := sp ( common sp )? ( group sp )*
# sp         := ( NEWLINE | WHITESPACE )*
# common     := "#_Common" NEWLINE brace_expr NEWLINE
# group      := group_name ( delimiter ( STRING | brace_expr ) )*
#               ( delimiter )? NEWLINE ( sentence )*
# group_name := DOLLAR ( TEXT )+
# delimiter  := ( WHITESPACE )? ( COMMA | SEMICOLON ) ( WHITESPACE )?
# sentence   := line NEWLINE |
#               OPEN_BRACE NEWLINE ( line NEWLINE )* CLOSE_BRACE NEWLINE
# word       := ( TEXT | WHITESPACE | DOLLAR | brace_expr | string )*
# line       := ( TEXT | WHITESPACE | DOLLAR | brace_expr | QUOTE |
#                 OPEN_PAREN | CLOSE_PAREN | OPEN_BRACKET | CLOSE_BRACKET |
#                 OPERATOR | COMMA | SEMICOLON | DIRECTIVE )*
# string     := QUOTE (
#                 TEXT | WHITESPACE | DOLLAR | OPEN_BRACE | CLOSE_BRACE |
#                 OPEN_PAREN | CLOSE_PAREN | OPEN_BRACKET | CLOSE_BRACKET |
#                 OPERATOR | COMMA | SEMICOLON | DIRECTIVE )*
#               QUOTE
# brace_expr := variable | indexed | assignment | increment |
#               function | if | calc
# variable   := OPEN_BRACE sp var_name sp CLOSE_BRACE
# indexed    := OPEN_BRACE sp var_name sp
#               OPEN_BRACKET sp word sp CLOSE_BRACKET sp CLOSE_BRACE
# var_name   := DOLLAR ( TEXT | brace_expr )+
# assignment := OPEN_BRACE sp var_name sp assign_ops sp add_expr sp CLOSE_BRACE
# assign_ops := "=" | "+=" | "-=" | "*=" | "/="
# increment  := OPEN_BRACE sp var_name sp inc_ops sp CLOSE_BRACE
# inc_ops    := "+" "+" | "-" "-"
# function   := OPEN_BRACE sp DOLLAR sp ( TEXT )+ sp OPEN_PAREN
#               ( sp argument ( sp COMMA sp argument )* sp )?
#               CLOSE_PAREN sp CLOSE_BRACE
# argument   := ( TEXT | WHITESPACE | DOLLAR | brace_expr | string |
#                 OPEN_BRACKET | CLOSE_BRACKET | OPERATOR | SEMICOLON )*
# if         := OPEN_BRACE sp DOLLAR sp "if" sp cond_expr
#               ( sp OPEN_BRACE ( sp line )* sp CLOSE_BRACE ( sp "else"
#                 sp OPEN_BRACE ( sp line )* sp CLOSE_BRACE )? sp )?
#               sp CLOSE_BRACE
# calc       := OPEN_BRACE sp DOLLAR sp "calc" sp add_expr sp CLOSE_BRACE
# cond_expr  := OPEN_PAREN sp or_expr sp CLOSE_PAREN
# or_expr    := and_expr ( sp "||" sp and_expr )*
# and_expr   := sub_expr ( sp "&&" sp sub_expr )*
# sub_expr   := comp_expr | cond_expr
# comp_expr  := add_expr ( sp comp_op sp add_expr )?
# comp_op    := "==" | "!=" | "<" | "<=" | ">" | ">="
# add_expr   := mul_expr (sp add_op sp mul_expr)*
# add_op     := "+" | "-"
# mul_expr   := pow_expr (sp mul_op sp pow_expr)*
# mul_op     := "*" | "/" | "%"
# pow_expr   := unary_expr (sp pow_op sp unary_expr)*
# pow_op     := "^"
# unary_expr := unary_op sp unary_expr | factor
# unary_op   := "+" | "-"
# factor     := OPEN_PAREN sp add_expr sp CLOSE_PAREN | word


###   INTERPRETER   ###

class Group:

    def __init__(self, misaka, name, item_list=None):
        self.misaka = misaka
        self.name = name
        self.list = item_list or []

    def __len__(self):
        return len(self.list)

    def __getitem__(self, index):
        return self.list[index]

    def get(self):
        if not self.list:
            return None
        return random.choice(self.list)

    def copy(self, name):
        return Array(self.misaka, name, self.list[:])

class NonOverlapGroup(Group):

    def __init__(self, misaka, name, item_list=None):
        Group.__init__(self, misaka, name, item_list)
        self.indexes = []

    def get(self):
        if not self.list:
            return None
        if not self.indexes:
            self.indexes = list(range(len(self.list)))
        index = self.indexes.pop(random.choice(range(len(self.indexes))))
        return self.list[index]

class SequentialGroup(Group):

    def __init__(self, misaka, name, item_list=None):
        Group.__init__(self, misaka, name, item_list)
        self.indexes = []

    def get(self):
        if not self.list:
            return None
        if not self.indexes:
            self.indexes = list(range(len(self.list)))
        index = self.indexes.pop(0)
        return self.list[index]

class Array(Group):

    def append(self, s):
        self.list.append([[NODE_TEXT, str(s)]])

    def index(self, s):
        for i in range(len(self.list)):
            if s == self.misaka.expand(self.list[i]):
                return i
        return -1

    def pop(self):
        if not self.list:
            return None
        return self.list.pop(random.choice(range(len(self.list))))

    def popmatchl(self, s):
        buf = []
        for i in range(len(self.list)):
            t = self.misaka.expand(self.list[i])
            if t.startswith(s):
                buf.append(i)
        if not buf:
            return None
        return self.list.pop(random.choice(buf))


TYPE_SCHOLAR = 1
TYPE_ARRAY   = 2


class Shiori:

    DBNAME = b'misaka.db'

    def __init__(self, dll_name):
        self.dll_name = dll_name
        self.charset = 'CP932'

    def use_saori(self, saori):
        self.saori = saori

    def find(self, top_dir, dll_name):
        result = 0
        if list_dict(top_dir):
            result = 210
        elif os.path.isfile(os.path.join(top_dir, b'misaka.ini')):
            result = 200
        return result

    def show_description(self):
        logging.info(
            'Shiori: MISAKA compatible module for ninix\n'
            '        Copyright (C) 2002 by Tamito KAJIYAMA\n'
            '        Copyright (C) 2002, 2003 by MATSUMURA Namihiko\n'
            '        Copyright (C) 2002-2013 by Shyouzou Sugitani')

    def reset(self):
        self.dict = {}
        self.variable = {}
        self.constant = {}
        self.misaka_debug = 0
        self.misaka_error = 0
        self.reference = (None, None, None, None, None, None, None, None)
        self.mouse_move_count = {}
        self.random_talk = -1
        self.otherghost = []
        self.communicate = ['', '']
        self.reset_request()

    def load(self, misaka_dir=os.curdir):
        self.misaka_dir = misaka_dir
        self.dbpath = os.path.join(self.misaka_dir, self.DBNAME)
        self.saori_library = SaoriLibrary(self.saori, self.misaka_dir)
        self.reset()
        try:
            filelist, debug, error = read_misaka_ini(self.misaka_dir)
        except IOError:
            logging.debug('cannot read misaka.ini')
            return 0
        except MisakaError as error:
            logging.debug(error)
            return 0
        self.misaka_debug = debug
        self.misaka_error = error
        global_variables = []
        global_constants = []
        # charset auto-detection
        if chardet is not None:
            detector = chardet.universaldetector.UniversalDetector()
            for filename in filelist:
                path = os.path.join(self.misaka_dir, filename)
                try:
                    f = open(path, 'rb')
                except IOError:
                    logging.debug('cannot read {0}'.format(filename))
                    continue
                basename, ext = os.path.splitext(filename)
                if ext == '.__1':
                    f = io.StringIO(self.crypt(f.read()))
                detector.reset()
                for line in f:
                    detector.feed(line)
                    if detector.done:
                        break
                detector.close()
                f.close()
                if detector.result['confidence'] > 0.98: # XXX
                    self.charset = detector.result['encoding']
                    if self.charset == 'SHIFT_JIS':
                        self.charset = 'CP932' # XXX
                    break
        for filename in filelist:
            path = os.path.join(self.misaka_dir, filename)
            basename, ext = os.path.splitext(filename)
            try:
                if ext == b'.__1': # should read lines as bytes
                    f = open(path, 'rb')
                else:
                    f = open(path, encoding=self.charset)
            except IOError:
                logging.debug('cannot read {0}'.format(filename))
                continue
            if ext == b'.__1':
                f = io.StringIO(self.crypt(f.read()))
            try:
                variables, constants = self.read(f, path)
            except MisakaError as error:
                logging.debug(error)
                continue
            global_variables.extend(variables)
            global_constants.extend(constants)
        self.eval_globals(global_variables, 0)
        self.load_database()
        self.eval_globals(global_constants, 1)
        if '$_OnRandomTalk' in self.dict:
            self.random_talk = 0
        return 1

    def crypt(self, data):
        return str(b''.join([int.to_bytes(c ^ 0xff, 1, 'little') for c in data]), self.charset)

    def eval_globals(self, sentences, constant):
        for sentence in sentences:
            for node in sentence:
                if node[0] == NODE_ASSIGNMENT:
                    self.eval_assignment(node[1], node[2], node[3], constant)
                elif node[0] == NODE_FUNCTION:
                    self.eval_function(node[1], node[2])

    def read(self, f, path=None):
        parser = Parser(self.charset)
        parser.read(f, path)
        common, dic = parser.get_dict()
        variables = []
        constants = []
        for name, parameters, sentences in dic:
            if not sentences:
                continue
            elif name == '$_Variable':
                variables.extend(sentences)
                continue
            elif name == '$_Constant':
                constants.extend(sentences)
                continue
            GroupClass = Group
            conditions = []
            if common is not None:
                conditions.append(common)
            for node in parameters:
                if node[0] == NODE_IF:
                    conditions.append(node)
                elif node[0] == NODE_TEXT and node[1] == 'nonoverlap':
                    GroupClass = NonOverlapGroup
                elif node[0] == NODE_TEXT and node[1] == 'sequential':
                    GroupClass = SequentialGroup
                else:
                    pass # ignore unknown parameters
            group = GroupClass(self, name, sentences)
            try:
                grouplist = self.dict[name]
            except KeyError:
                grouplist = self.dict[name] = []
            grouplist.append((group, conditions))
        return variables, constants

    def strip_newline(self, line):
        if line.endswith('\r\n'):
            return line[:-2]
        elif line.endswith('\r') or line.endswith('\n'):
            return line[:-1]
        return line

    def load_database(self):
        try:
            with open(self.dbpath) as f:
                while 1:
                    try:
                        line = f.readline()
                    except UnicodeError:
                        logging.debug(
                            'misaka.py: malformed database (ignored)')
                        break
                    if not line:
                        break
                    header = self.strip_newline(line).split()
                    if header[0] == 'SCHOLAR':
                        name = header[1]
                        value = self.strip_newline(f.readline())
                        self.variable[name] = (TYPE_SCHOLAR, value)
                    elif header[0] == 'ARRAY':
                        try:
                            size = int(header[1])
                        except ValueError:
                            logging.debug(
                                'misaka.py: malformed database (ignored)')
                            break
                        name = header[2]
                        array = Array(self, name)
                        for _ in range(size):
                            value = self.strip_newline(f.readline())
                            array.append(value)
                        self.variable[name] = (TYPE_ARRAY, array)
                    else:
                        raise RuntimeError('should not reach here')
        except IOError:
            return

    def save_database(self):
        try:
            with open(self.dbpath, 'w') as f:
                for name in self.variable:
                    value_type, value = self.variable[name]
                    if value_type == TYPE_SCHOLAR:
                        if value == '':
                            continue
                        f.write('SCHOLAR {0}\n{1}\n'.format(name, value))
                    elif value_type == TYPE_ARRAY:
                        f.write('ARRAY {0:d} {1}\n'.format(len(value), name))
                        for item in value:
                            f.write('{0}\n'.format(self.expand(item)))
                    else:
                        raise RuntimeError('should not reach here')
        except IOError:
            logging.debug('misaka.py: cannot write database (ignored)')
            return

    def unload(self):
        self.save_database()
        self.saori_library.unload()
        return 1

    def reset_request(self):
        self.req_command = ''
        self.req_protocol = ''
        self.req_key = []
        self.req_header = {}

    def request(self, req_string):
        header = req_string.splitlines()
        line = header.pop(0)
        if line:
            line = str(line, self.charset).strip()
            req_list = line.split()
            if len(req_list) >= 2:
                self.req_command = req_list[0].strip()
                self.req_protocol = req_list[1].strip()
            for line in header:
                line = str(line, self.charset).strip()
                if not line:
                    continue
                if ':' not in line:
                    continue
                key, value = line.split(':', 1)
                key = key.strip()
                try:
                    value = int(value.strip())
                except:
                    value = value.strip()
                self.req_key.append(key)
                self.req_header[key] = value
        # FIXME
        ref0 = self.get_ref(0)
        ref1 = self.get_ref(1)
        ref2 = self.get_ref(2)
        ref3 = self.get_ref(3)
        ref4 = self.get_ref(4)
        ref5 = self.get_ref(5)
        ref6 = self.get_ref(6)
        ref7 = self.get_ref(7)
        event = self.req_header['ID']
        script = None
        if event == 'otherghostname':
            n = 0
            refs = []
            while 1:
                ref = self.get_ref(n)
                if ref:
                    refs.append(ref)
                else:
                    break
                n += 1
            self.otherghost = []
            for ref in refs:
                name, s0, s1 = ref.split(chr(1))
                self.otherghost.append([name, s0, s1])
        if event == 'OnMouseMove':
            key = (ref3, ref4) # side, part
            count = self.mouse_move_count.get(key, 0)
            self.mouse_move_count[key] = count + 5
        if event == 'OnCommunicate':
            self.communicate = [ref0, ref1]
            script = self.get('$_OnGhostCommunicateReceive', default=None)
        elif event == 'charset':
            script = self.charset
        else:
            self.reference = (ref0, ref1, ref2, ref3, ref4, ref5, ref6, ref7)
            script = self.get(''.join(('$', event)), default=None)
        if event == 'OnSecondChange':
            if self.random_talk == 0:
                self.reset_random_talk_interval()
            elif self.random_talk > 0:
                self.random_talk -= 1
                if self.random_talk == 0:
                    script = self.get('$_OnRandomTalk', default=None)
        if script is not None:
            script = script.strip()
        else:
            script = ''
        self.reset_request()
        response = b''.join((b'SHIORI/3.0 200 OK\r\n',
                             b'Sender: Misaka\r\n',
                             b'Charset: ',
                             self.charset.encode(self.charset, 'ignore'),
                             b'\r\n',
                             b'Value: ',
                             script.encode(self.charset, 'ignore'),
                             b'\r\n'))
        to = self.eval_variable([[NODE_TEXT, '$to']])
        if to:
            del self.variable['$to']
            response = b''.join((response,
                                 b'Reference0: ',
                                 to.encode(self.charset, 'ignore'),
                                 b'\r\n'))
        response = b''.join((response, b'\r\n'))
        return response

    def get_ref(self, num):
        key = ''.join(('Reference', str(num)))
        return self.req_header.get(key)

    # internal
    def reset_random_talk_interval(self):
        interval = 0
        value_type, value = self.variable.get('$_talkinterval', (TYPE_SCHOLAR, ''))
        if value_type == TYPE_SCHOLAR:
            try:
                interval = max(int(value), 0)
            except ValueError:
                pass
        self.random_talk = int(interval * random.randint(5, 15) / 10.0)

    def get(self, name, default=''):
        grouplist = self.dict.get(name)
        if grouplist is None:
            return default
        for group, conditions in grouplist:
            if self.eval_and_expr(conditions) == 'true':
                return self.expand(group.get())
        return default

    def expand(self, nodelist):
        if nodelist is None:
            return ''
        buf = []
        for node in nodelist:
            buf.append(self.eval(node))
        return ''.join(buf)

    def expand_args(self, nodelist):
        tmp = self.expand(nodelist)
        return self.eval_variable(nodelist) if tmp.startswith('$') else tmp

    def eval(self, node):
        if node[0] == NODE_TEXT:
            result = node[1]
        elif node[0] == NODE_STRING:
            result = node[1]
        elif node[0] == NODE_VARIABLE:
            result = self.eval_variable(node[1])
        elif node[0] == NODE_INDEXED:
            result = self.eval_indexed(node[1], node[2])
        elif node[0] == NODE_ASSIGNMENT:
            result = self.eval_assignment(node[1], node[2], node[3])
        elif node[0] == NODE_FUNCTION:
            result = self.eval_function(node[1], node[2])
        elif node[0] == NODE_IF:
            result = self.eval_if(node[1], node[2], node[3])
        elif node[0] == NODE_CALC:
            result = self.eval_calc(node[1])
        elif node[0] == NODE_COMP_EXPR:
            result = self.eval_comp_expr(node[1], node[2], node[3])
        elif node[0] == NODE_AND_EXPR:
            result = self.eval_and_expr(node[1:])
        elif node[0] == NODE_OR_EXPR:
            result = self.eval_or_expr(node[1:])
        elif node[0] == NODE_ADD_EXPR:
            result = self.eval_add_expr(node[1:])
        elif node[0] == NODE_MUL_EXPR:
            result = self.eval_mul_expr(node[1:])
        elif node[0] == NODE_POW_EXPR:
            result = self.eval_pow_expr(node[1:])
        else:
            ##print(node)
            raise RuntimeError('should not reach here')
        return result

    SYSTEM_VARIABLES = [
        # current date and time
        '$year', '$month', '$day', '$hour', '$minute', '$second',
        # day of week (0 = Sunday)
        '$dayofweek',
        # ghost uptime
        '$elapsedhour', '$elapsedminute', '$elapsedsecond',
        # OS uptime
        '$elapsedhouros', '$elapsedminuteos', '$elapsedsecondos',
        # OS accumlative uptime
        '$elapsedhourtotal', '$elapsedminutetotal', '$elapsedsecondtotal',
        # system information
        '$os.version',
        '$os.name',
        '$os.phisicalmemorysize',
        '$os.freememorysize',
        '$os.totalmemorysize',
        '$cpu.vendorname',
        '$cpu.name',
        '$cpu.clockcycle',
        # number of days since the last network update
        '$daysfromlastupdate',
        # number of days since the first boot
        '$daysfromfirstboot',
        # name of the ghost with which it has communicated
        ##'$to',
        '$sender',
        '$lastsentence',
        '$otherghostlist',
        ]

    def eval_variable(self, name):
        now = time.localtime(time.time())
        name = self.expand(name)
        if name == '$year':
            result = str(now[0])
        elif name == '$month':
            result = str(now[1])
        elif name == '$day':
            result = str(now[2])
        elif name == '$hour':
            result = str(now[3])
        elif name == '$minute':
            result = str(now[4])
        elif name == '$second':
            result = str(now[5])
        elif name == '$dayofweek':
            result = str([1, 2, 3, 4, 5, 6, 0][now[6]])
        elif name == '$lastsentence':
            result = self.communicate[1]
        elif name == '$otherghostlist':
            if self.otherghost:
                ghost = random.choice(self.otherghost)
                result = ghost[0]
            else:
                result = ''
        elif name == '$sender':
            result = self.communicate[0]
        elif name in self.SYSTEM_VARIABLES:
            result = ''
        elif name in self.dict:
            result = self.get(name)
        elif name in self.constant:
            value_type, value = self.constant[name]
            if value_type == TYPE_ARRAY:
                result = self.expand(value.get())
            else:
                result = str(value)
        elif name in self.variable:
            value_type, value = self.variable[name]
            if value_type == TYPE_ARRAY:
                result = self.expand(value.get())
            else:
                result = str(value)
        else:
            result = ''
        return result

    def eval_indexed(self, name, index):
        name = self.expand(name)
        index = self.expand(index)
        try:
            index = int(index)
        except ValueError:
            index = 0
        if name == '$otherghostlist':
            group = []
            for ghost in self.otherghost:
                group.append(ghost[0])
        elif name in self.SYSTEM_VARIABLES:
            return ''
        elif name in self.dict:
            grouplist = self.dict[name]
            group, constraints = grouplist[0]
        elif name in self.constant:
            return ''
        elif name in self.variable:
            value_type, group = self.variable[name]
            if value_type != TYPE_ARRAY:
                return ''
        else:
            return ''
        if index < 0 or index >= len(group):
            return ''
        return self.expand(group[index])

    def eval_assignment(self, name, operator, value, constant=0):
        name = self.expand(name)
        value = self.expand(value)
        if operator in ['+=', '-=', '*=', '/=']:
            try:
                operand = int(value)
            except ValueError:
                operand = 0
            value_type, value = self.constant.get(name, (None, ''))
            if value_type is None:
                value_type, value = self.variable.get(name, (TYPE_SCHOLAR, ''))
            if value_type == TYPE_ARRAY:
                return ''
            try:
                value = int(value)
            except ValueError:
                value = 0
            if operator == '+=':
                value += operand
            elif operator == '-=':
                value -= operand
            elif operator == '*=':
                value *= operand
            elif operator == '/=' and operand != 0:
                value //= operand
        if constant or name in self.constant:
            self.constant[name] = (TYPE_SCHOLAR, value)
        else:
            self.variable[name] = (TYPE_SCHOLAR, value)
        if name == '$_talkinterval':
            self.reset_random_talk_interval()
        return ''

    def eval_comp_expr(self, operand1, operator, operand2):
        value1 = self.expand(operand1)
        value2 = self.expand(operand2)
        try:
            operand1 = int(value1)
            operand2 = int(value2)
        except ValueError:
            operand1 = value1
            operand2 = value2
        if operator == '==' and operand1 == operand2 or \
           operator == '!=' and operand1 != operand2 or \
           operator == '<'  and operand1 <  operand2 or \
           operator == '<=' and operand1 <= operand2 or \
           operator == '>'  and operand1 >  operand2 or \
           operator == '>=' and operand1 >= operand2:
            return 'true'
        return 'false'

    def eval_and_expr(self, conditions):
        for condition in conditions:
            boolean = self.eval(condition)
            if boolean.strip() != 'true':
                return 'false'
        return 'true'

    def eval_or_expr(self, conditions):
        for condition in conditions:
            boolean = self.eval(condition)
            if boolean.strip() == 'true':
                return 'true'
        return 'false'

    def eval_add_expr(self, expression):
        value = self.expand(expression[0])
        try:
            value = int(value)
        except ValueError:
            value = 0
        for i in range(1, len(expression), 2):
            operand = self.expand(expression[i + 1])
            try:
                operand = int(operand)
            except ValueError:
                operand = 0
            if expression[i] == '+':
                value += operand
            elif expression[i] == '-':
                value -= operand
        return str(value)

    def eval_mul_expr(self, expression):
        value = self.expand(expression[0])
        try:
            value = int(value)
        except ValueError:
            value = 0
        for i in range(1, len(expression), 2):
            operand = self.expand(expression[i + 1])
            try:
                operand = int(operand)
            except ValueError:
                operand = 0
            if expression[i] == '*':
                value *= operand
            elif expression[i] == '/' and operand != 0:
                value //= operand
            elif expression[i] == '%' and operand != 0:
                value %= operand
        return str(value)

    def eval_pow_expr(self, expression):
        value = self.expand(expression[-1])
        try:
            value = int(value)
        except ValueError:
            value = 0
        for i in range(1, len(expression)):
            operand = self.expand(expression[-i - 1])
            try:
                operand = int(operand)
            except ValueError:
                operand = 0
            value = operand**value
        return str(value)

    def eval_if(self, condition, then_clause, else_clause):
        boolean = self.eval(condition)
        if boolean.strip() == 'true':
            return self.expand(then_clause)
        else:
            return self.expand(else_clause)

    def eval_calc(self, expression):
        return self.expand(expression)

    def eval_function(self, name, args):
        function = self.SYSTEM_FUNCTIONS.get(name)
        if function is None:
            return ''
        return function(self, args)

    def is_number(self, s):
        return s and all([bool(c in '0123456789') for c in s])

    def split(self, s):
        buf = []
        i, j = 0, len(s)
        while i < j:
            if ord(s[i]) < 0x80:
                buf.append(s[i])
                i += 1
            else:
                buf.append(s[i:i + 2])
                i += 2
        return buf

    def exec_reference(self, args):
        if len(args) != 1:
            return ''
        n = self.expand_args(args[0])
        if n in '01234567':
            value = self.reference[int(n)]
            if value is not None:
                return str(value)
        return ''

    def exec_random(self, args):
        if len(args) != 1:
            return ''
        n = self.expand_args(args[0])
        try:
            return str(random.randrange(int(n)))
        except ValueError:
            return '' # n < 1 or not a number

    def exec_choice(self, args):
        if not args:
            return ''
        return self.expand_args(random.choice(args))

    def exec_getvalue(self, args):
        if len(args) != 2:
            return ''
        try:
            n = int(self.expand_args(args[1]))
        except ValueError:
            return ''
        if n < 0:
            return ''
        value_list = self.expand_args(args[0]).split(',')
        try:
            return value_list[n]
        except IndexError:
            return ''

    def exec_search(self, args):
        namelist = []
        for arg in args:
            name = self.expand_args(arg)
            if name.strip():
                namelist.append(name)
        if not namelist:
            return ''
        keylist = []
        for key in self.keys(): # dict, variable, constant
            for name in namelist:
                if name not in key:
                    break
            else:
                keylist.append(key)
        if not keylist:
            return ''
        return self.eval_variable([[NODE_TEXT, random.choice(keylist)]])

    def keys(self):
        buf = list(self.dict.keys())
        buf.extend(self.variable.keys())
        buf.extend(self.constant.keys())
        return buf

    def exec_backup(self, args):
        if not args:
            self.save_database()
        return ''

    def exec_getmousemovecount(self, args):
        if len(args) == 2:
            side = self.expand_args(args[0])
            part = self.expand_args(args[1])
            try:
                key = (int(side), part)
            except ValueError:
                pass
            else:
                return str(self.mouse_move_count.get(key, 0))
        return ''

    def exec_resetmousemovecount(self, args):
        if len(args) == 2:
            side = self.expand_args(args[0])
            part = self.expand_args(args[1])
            try:
                key = (int(side), part)
            except ValueError:
                pass
            else:
                self.mouse_move_count[key] = 0
        return ''

    def exec_substring(self, args):
        if len(args) != 3:
            return ''
        value = self.expand_args(args[2])
        try:
            count = int(value)
        except ValueError:
            return ''
        if count < 0:
            return ''
        value = self.expand_args(args[1])
        try:
            offset = int(value)
        except ValueError:
            return ''
        if offset < 0:
            return ''
        s = self.expand_args(args[0]).encode(self.charset, 'replace')
        return str(s[offset:offset + count], self.charset, 'replace')

    def exec_substringl(self, args):
        if len(args) != 2:
            return ''
        value = self.expand_args(args[1])
        try:
            count = int(value)
        except ValueError:
            return ''
        if count < 0:
            return ''
        s = self.expand_args(args[0]).encode(self.charset, 'replace')
        return str(s[:count], self.charset, 'replace')

    def exec_substringr(self, args):
        if len(args) != 2:
            return ''
        value = self.expand_args(args[1])
        try:
            count = int(value)
        except ValueError:
            return ''
        if count < 0:
            return ''
        s = self.expand_args(args[0]).encode(self.charset, 'replace')
        return str(s[len(s) - count:], self.charset, 'replace')

    def exec_substringfirst(self, args):
        if len(args) != 1:
            return ''
        buf = self.expand_args(args[0])
        return '' if not buf else buf[0]

    def exec_substringlast(self, args):
        if len(args) != 1:
            return ''
        buf = self.expand_args(args[0])
        return '' if not buf else buf[-1]

    def exec_length(self, args):
        if len(args) != 1:
            return ''
        return str(
            len(self.expand_args(args[0]).encode(self.charset, 'replace')))

    katakana2hiragana = {
        'ァ': 'ぁ', 'ア': 'あ', 'ィ': 'ぃ', 'イ': 'い', 'ゥ': 'ぅ',
        'ウ': 'う', 'ェ': 'ぇ', 'エ': 'え', 'ォ': 'ぉ', 'オ': 'お',
        'カ': 'か', 'ガ': 'が', 'キ': 'き', 'ギ': 'ぎ', 'ク': 'く',
        'グ': 'ぐ', 'ケ': 'け', 'ゲ': 'げ', 'コ': 'こ', 'ゴ': 'ご',
        'サ': 'さ', 'ザ': 'ざ', 'シ': 'し', 'ジ': 'じ', 'ス': 'す',
        'ズ': 'ず', 'セ': 'せ', 'ゼ': 'ぜ', 'ソ': 'そ', 'ゾ': 'ぞ',
        'タ': 'た', 'ダ': 'だ', 'チ': 'ち', 'ヂ': 'ぢ', 'ッ': 'っ',
        'ツ': 'つ', 'ヅ': 'づ', 'テ': 'て', 'デ': 'で', 'ト': 'と',
        'ド': 'ど', 'ナ': 'な', 'ニ': 'に', 'ヌ': 'ぬ', 'ネ': 'ね',
        'ノ': 'の', 'ハ': 'は', 'バ': 'ば', 'パ': 'ぱ', 'ヒ': 'ひ',
        'ビ': 'び', 'ピ': 'ぴ', 'フ': 'ふ', 'ブ': 'ぶ', 'プ': 'ぷ',
        'ヘ': 'へ', 'ベ': 'べ', 'ペ': 'ぺ', 'ホ': 'ほ', 'ボ': 'ぼ',
        'ポ': 'ぽ', 'マ': 'ま', 'ミ': 'み', 'ム': 'む', 'メ': 'め',
        'モ': 'も', 'ャ': 'ゃ', 'ヤ': 'や', 'ュ': 'ゅ', 'ユ': 'ゆ',
        'ョ': 'ょ', 'ヨ': 'よ', 'ラ': 'ら', 'リ': 'り', 'ル': 'る',
        'レ': 'れ', 'ロ': 'ろ', 'ヮ': 'ゎ', 'ワ': 'わ', 'ヰ': 'ゐ',
        'ヱ': 'ゑ', 'ヲ': 'を', 'ン': 'ん', 'ヴ': 'う゛',
        }

    def exec_hiraganacase(self, args):
        if len(args) != 1:
            return ''
        buf = []
        string = str(self.expand_args(args[0]), self.charset, 'replace')
        #string = str(args[0], self.charset, 'replace')
        for c in string:
            c = self.katakana2hiragana.get(c, c)
            buf.append(c)
        return ''.join(buf)

    def exec_isequallastandfirst(self, args):
        if len(args) != 2:
            return 'false'
        buf0 = self.expand_args(args[0])
        buf1 = self.expand_args(args[1])
        return 'true' if buf0 and buf1 and buf0[-1] == buf1[0] else 'false'

    def exec_append(self, args):
        if len(args) == 2:
            name = self.expand_args(args[0])
            if name and name.startswith('$'):
                value_type, value = self.variable.get(name, (TYPE_SCHOLAR, ''))
                if value_type == TYPE_SCHOLAR:
                    value_type = TYPE_ARRAY
                    array = Array(self, name)
                    if value != '':
                        array.append(value)
                    value = array
                member_list = self.expand_args(args[1]).split(chr(1))
                for member in member_list:
                    value.append(member)
                self.variable[name] = (value_type, value)
        return ''

    def exec_stringexists(self, args):
        if len(args) != 2:
            return ''
        name = self.expand_args(args[0])
        value_type, value = self.variable.get(name, (TYPE_SCHOLAR, ''))
        if value_type == TYPE_SCHOLAR:
            return ''
        if value.index(self.expand_args(args[1])) < 0:
            return 'false'
        else:
            return 'true'

    def exec_copy(self, args):
        if len(args) != 2:
            return ''
        src_name = self.expand_args(args[0])
        if not (src_name and src_name.startswith('$')):
            return ''
        new_name = self.expand_args(args[1])
        if not (new_name and new_name.startswith('$')):
            return ''
        source = None
        if src_name in self.dict:
            grouplist = self.dict[src_name]
            source, conditions = grouplist[0]
        elif src_name in self.variable:
            value_type, value = self.variable.get(src_name)
            if value_type == TYPE_ARRAY:
                source = value
        if source is None:
            value = Array(self, new_name)
        else:
            value = source.copy(new_name)
        self.variable[new_name] = (TYPE_ARRAY, value)
        return ''

    def exec_pop(self, args):
        if len(args) == 1:
            name = self.expand_args(args[0])
            value_type, value = self.variable.get(name, (TYPE_SCHOLAR, ''))
            if value_type == TYPE_ARRAY:
                return self.expand(value.pop())
        return ''

    def exec_popmatchl(self, args):
        if len(args) == 2:
            name = self.expand_args(args[0])
            value_type, value = self.variable.get(name, (TYPE_SCHOLAR, ''))
            if value_type == TYPE_ARRAY:
                return self.expand(value.popmatchl(self.expand_args(args[1])))
        return ''

    def exec_index(self, args):
        if len(args) == 2:
            pos = self.expand_args(args[1]).find(self.expand_args(args[0]))
            if pos > 0:
                s = self.expand_args(args[0])[pos]
                pos = len(s.encode(self.charset, 'replace'))
            return str(pos)
        return ''

    def exec_insentence(self, args):
        if not args:
            return ''
        s = self.expand_args(args[0])
        for i in range(1, len(args)):
            if self.expand_args(args[i]) not in s:
                return 'false'
        return 'true'

    def exec_substringw(self, args):
        if len(args) != 3:
            return ''
        value = self.expand_args(args[2])
        try:
            count = int(value)
        except ValueError:
            return ''
        if count < 0:
            return ''
        value = self.expand_args(args[1])
        try:
            offset = int(value)
        except ValueError:
            return ''
        if offset < 0:
            return ''
        buf = self.expand_args(args[0])
        return buf[offset:offset + count]

    def exec_substringwl(self, args):
        if len(args) != 2:
            return ''
        value = self.expand_args(args[1])
        try:
            count = int(value)
        except ValueError:
            return ''
        if count < 0:
            return ''
        buf = self.expand_args(args[0])
        return buf[:count]

    def exec_substringwr(self, args):
        if len(args) != 2:
            return ''
        value = self.expand_args(args[1])
        try:
            count = int(value)
        except ValueError:
            return ''
        if count < 0:
            return ''
        buf = self.expand_args(args[0])
        return buf[len(buf) - count:]

    prefixes = ['さん', 'ちゃん', '君', 'くん', '様', 'さま', '氏', '先生']

    def exec_adjustprefix(self, args):
        if len(args) != 2:
            return ''
        s = self.expand_args(args[0])
        for prefix in self.prefixes:
            if s.endswith(prefix):
                s = ''.join((s[:-len(prefix)], self.expand_args(args[1])))
                break
        return s

    def exec_count(self, args):
        if len(args) != 1:
            return ''
        name = self.expand_args(args[0])
        try:
            value_type, value = self.variable[name]
            if value_type == TYPE_SCHOLAR:
                return '1'
            return str(len(value))
        except KeyError:
            return '-1'

    def exec_inlastsentence(self, args):
        if not args:
            return ''
        s = self.communicate[1]
        for i in range(1, len(args)):
            if self.expand_args(args[i]) not in s:
                return 'false'
        return 'true'

    def saori(self, args):
        saori_statuscode = ''
        saori_header = []
        saori_value = {}
        saori_protocol = ''
        req = b''.join((b'EXECUTE SAORI/1.0\r\n',
                        b'Sender: MISAKA\r\n',
                        b'SecurityLevel: local\r\n',
                        b'Charset: ',
                        self.charset.encode(self.charset),
                        b'\r\n'))
        for i in range(1, len(args)):
            req = b''.join(
                (req,
                 b'Argument', str(i).encode(self.charset), b': ', 
                 self.expand_args(args[i]).encode(self.charset, 'ignore'),
                 b'\r\n'))
        req = b''.join((req, b'\r\n'))
        response = self.saori_library.request(self.expand_args(args[0]), req)
        header = response.splitlines()
        if header:
            line = header.pop(0)
            line = str(line, self.charset).strip()
            if ' ' in line:
                saori_protocol, saori_statuscode = [x.strip for x in line.split(' ', 1)]
            for line in header:
                line = str(line, self.charset).strip()
                if not line:
                    continue
                if ':' not in line:
                    continue
                key, value = [x.strip() for x in line.split(':', 1)]
                if key:
                    saori_header.append(key)
                    saori_value[key] = value
        return saori_value.get('Result', '')

    def load_saori(self, args):
        if not args:
            return ''
        result = self.saori_library.load(self.expand_args(args[0]),
                                         self.misaka_dir)
        return '' if not result else str(result)

    def unload_saori(self, args):
        if not args:
            return ''
        result = self.saori_library.unload(self.expand_args(args[0]))
        return '' if not result else str(result)

    def exec_isghostexists(self, args):
        result = 'false'
        if len(args) != 1:
            if len(self.otherghost):
                result = 'true'
                self.variable['$to'] = (TYPE_SCHOLAR,
                                        random.choice(self.otherghost)[0])
        else:
            for ghost in self.otherghost:
                if ghost[0] == self.expand_args(args[0]):
                    result = 'true'
                    break
        return result

    SYSTEM_FUNCTIONS = {
        '$reference':           exec_reference,
        '$random':              exec_random,
        '$choice':              exec_choice,
        '$getvalue':            exec_getvalue,
        '$inlastsentence':      exec_inlastsentence,
        '$isghostexists':       exec_isghostexists,
        '$search':              exec_search,
        '$backup':              exec_backup,
        '$getmousemovecount':   exec_getmousemovecount,
        '$resetmousemovecount': exec_resetmousemovecount,
        '$substring':           exec_substring,
        '$substringl':          exec_substringl,
        '$substringr':          exec_substringr,
        '$substringfirst':      exec_substringfirst,
        '$substringlast':       exec_substringlast,
        '$length':              exec_length,
        '$hiraganacase':        exec_hiraganacase,
        '$isequallastandfirst': exec_isequallastandfirst,
        '$append':              exec_append,
        '$stringexists':        exec_stringexists,
        '$copy':                exec_copy,
        '$pop':                 exec_pop,
        '$popmatchl':           exec_popmatchl,
        '$index':               exec_index,
        '$insentence':          exec_insentence,
        '$substringw':          exec_substringw,
        '$substringwl':         exec_substringwl,
        '$substringwr':         exec_substringwr,
        '$adjustprefix':        exec_adjustprefix,
        '$count':               exec_count,
        '$saori':               saori,
        '$loadsaori':           load_saori,
        '$unloadsaori':         unload_saori,
        }


class SaoriLibrary:

    def __init__(self, saori, top_dir):
        self.saori_list = {}
        self.saori = saori

    def load(self, name, top_dir):
        result = 0
        head, name = os.path.split(name.replace('\\', '/')) # XXX: don't use os.fsencode() here
        top_dir = os.path.join(top_dir, os.fsencode(head))
        if name not in self.saori_list:
            module = self.saori.request(name)
            if module:
                self.saori_list[name] = module
        if name in self.saori_list:
            result = self.saori_list[name].load(top_dir)
        return result

    def unload(self, name=None):
        if name:
            name = os.path.split(name.replace('\\', '/'))[-1] # XXX: don't use os.fsencode() here
            if name in self.saori_list:
                self.saori_list[name].unload()
                del self.saori_list[name]
        else:
            for saori in self.saori_list.values():
                saori.unload()
        return None

    def request(self, name, req):
        result = '' # FIXME
        name = os.path.split(name.replace('\\', '/'))[-1] # XXX: don't use os.fsencode() here
        if name and name in self.saori_list:
            result = self.saori_list[name].request(req)
        return result


def misaka_open(top_dir):
    misaka = Shiori(top_dir)
    misaka.load()
    return misaka

###   TEST   ###

def test_ini(dirlist):
    for top_dir in dirlist:
        top_dir = os.fsencode(top_dir)
        print('Reading', os.path.join(top_dir, b'misaka.ini'), '...')
        filelist, debug, error = read_misaka_ini(top_dir)
        print('number of dictionaries =', len(filelist))
        for filename in filelist:
            print(filename)
        print('debug =', debug)
        print('error =', error)

def test_lexer(filelist, charset='CP932'):
    lexer = Lexer(charset=charset)
    for filename in filelist:
        filename = os.fsencode(filename)
        print('Reading', filename, '...')
        lexer.read(open(filename, encoding=charset))
    token_names = {
        TOKEN_WHITESPACE:    'TOKEN_WHITESPACE',
        TOKEN_NEWLINE:       'TOKEN_NEWLINE',
        TOKEN_OPEN_BRACE:    'TOKEN_OPEN_BRACE',
        TOKEN_CLOSE_BRACE:   'TOKEN_CLOSE_BRACE',
        TOKEN_OPEN_PAREN:    'TOKEN_OPEN_PAREN',
        TOKEN_CLOSE_PAREN:   'TOKEN_CLOSE_PAREN',
        TOKEN_OPEN_BRACKET:  'TOKEN_OPEN_BRACKET',
        TOKEN_CLOSE_BRACKET: 'TOKEN_CLOSE_BRACKET',
        TOKEN_DOLLAR:        'TOKEN_DOLLAR',
        TOKEN_COMMA:         'TOKEN_COMMA',
        TOKEN_SEMICOLON:     'TOKEN_SEMICOLON',
        TOKEN_OPERATOR:      'TOKEN_OPERATOR',
        TOKEN_DIRECTIVE:     'TOKEN_DIRECTIVE',
        TOKEN_TEXT:          'TOKEN_TEXT',
        }
    for token, lexeme, position in lexer.buffer:
        print('L{0:d} : C{1:-3d} :'.format(*position), token_names[token], ':',)
        if token in [TOKEN_WHITESPACE, TOKEN_NEWLINE,
                     TOKEN_OPEN_BRACE, TOKEN_CLOSE_BRACE,
                     TOKEN_OPEN_PAREN, TOKEN_CLOSE_PAREN,
                     TOKEN_OPEN_BRACKET, TOKEN_CLOSE_BRACKET,
                     TOKEN_DOLLAR,
                     TOKEN_COMMA, TOKEN_SEMICOLON]:
            print(repr(lexeme))
        else:
            print(lexeme)

def test_parser(filelist, charset='CP932'):
    for filename in filelist:
        filename = os.fsencode(filename)
        print('Reading', filename, '...')
        parser = Parser(charset=charset)
        parser.read(open(filename, encoding=charset))
        common, dic = parser.get_dict()
        if common is not None:
            print('Common')
            parser.dump_node(common, depth=2)
        for name, parameters, sentences in dic:
            print('Group', name)
            if parameters:
                print('Parameter:')
                for parameter in parameters:
                    print('>>>', parameter)
                    parser.dump_node(parameter, depth=2)
            if sentences:
                print('Sentence:')
                for sentence in sentences:
                    print('>>>', sentence)
                    parser.dump_list(sentence, depth=2)

def test_interpreter(top_dir):
    misaka = Shiori('misaka.dll') # XXX
    misaka.load(os.fsencode(top_dir))
    while 1:
        try:
            line = input('>>> ')
        except KeyboardInterrupt:
            print('Break')
            continue
        except EOFError:
            print()
            break
        command = line.split()
        if not command:
            continue
        elif len(command) == 1 and command[0].startswith('$'):
            print(misaka.eval_variable([[NODE_TEXT, command[0]]]))
        elif len(command) == 1 and command[0] == 'reload':
            misaka.load()
        else:
            print('list of commands:')
            print('  $id     get variable value')
            print('  reload  reload dictionaries')

if __name__ == '__main__':
    logger = logging.getLogger()
    logger.setLevel(logging.DEBUG) # XXX
    USAGE= 'Usage: misaka.py [ini|lexer|parser|interp] ...'
    charset = 'CP932' # XXX
    if len(sys.argv) == 1:
        print(USAGE)
    elif sys.argv[1] == 'ini':
        test_ini(sys.argv[2:])
    elif sys.argv[1] == 'lexer':
        test_lexer(sys.argv[2:], charset)
    elif sys.argv[1] == 'parser':
        test_parser(sys.argv[2:], charset)
    elif sys.argv[1] == 'interp':
        test_interpreter(sys.argv[2])
    else:
        print(USAGE)
