#!/usr/bin/env ruby
# This software is a part of NOODLYBOX.
# This software is distributed under the terms of the new BSD License.
# Copyright (c) 2009, molelord
# All rights reserved.

if (ARGV.length < 2)
    print <<EOF
Usage : dpidivide.rb -dll FOOBAR_dpi.h
        (for generate an implementation for dll side)
                 or
        dpidivide.rb -exe FOOBAR_dpi.h
        (for generate an implementation for exe side)
EOF
    exit 1
end

class Arg
    def initialize(type, name)
        @type          = type
        @name          = name
        @changedByFunc = false
        @fullCopy      = false
        @stringCopy    = false

        if (type =~ /\*$/) # pointer?
            if (type !~ /^const/ && type !~ /^void\*/)
                @changedByFunc = true
            end
            if (type =~ /int/ || type =~ /svLogicVecVal/)
                @fullCopy = true
            end
            if (type =~ /const char\*/)
                @stringCopy = true
            end
        end
    end
    attr_accessor :type
    attr_accessor :name
    attr_accessor :changedByFunc
    attr_accessor :fullCopy
    attr_accessor :stringCopy
end

class Function
    def initialize(type, name, dllside)
        @type             = type
        @name             = name
        @dllside          = dllside
        @numOfArgs        = type.length - 1

        i = 1
        max = 0
        while (i <= @numOfArgs)
            max = (max > type[i].length) ? max : type[i].length
            i += 1
        end
        @maxlenOfArgTypes = max
    end

    public
    def type
        return @type[0]
    end
    def name
        return @name[0]
    end
    def args
        tmp = []
        i = 1
        while (i <= @numOfArgs)
            tmp.push(Arg.new(@type[i], @name[i]))
            i += 1
        end
        return tmp
    end
    attr_accessor :dllside
    attr_accessor :numOfArgs
    attr_accessor :maxlenOfArgTypes
    def flipDllside
        @dllside = !@dllside
    end
    def dump
        printf "Function  type : %s\n", type()
        printf "          name : %s\n", name()
        printf "      dll side : %s\n", @dllside ? "true" : "false"
        printf "number of args : %d\n", @numOfArgs
        arg = args()
        arg.each do |a|
            printf "  argument type: %s\n", a.type
            printf "  argument name: %s\n", a.name
            printf "changed by func: %s\n", a.changedByFunc ? "true" : "false"
            printf "      full copy: %s\n", a.fullCopy ? "true" : "false"
            printf "    string copy: %s\n", a.stringCopy ? "true" : "false"
        end
    end
end

# ---- make foobar.cpp ------------------------------------------------
def putImplementation(dllImpl, commandline, dpiheader, func)
exeImpl = !dllImpl

printf "// %s\n", commandline
printf "#include <cstdlib> // size_t\n"
printf "#include \"svdpi.h\"\n"
printf "#include \"%s\"\n", dpiheader

if (dllImpl)
    printf "#include \"cClientSidePipe.h\" // ClientSidePipe\n"
else
    printf "#include \"cServerSidePipe.h\" // ServerSidePipe\n"
end

print <<EOF
#define PIPE_NAME "\\\\\\\\.\\\\pipe\\\\hello_pipe3"
#define RECEIVEBUF_SIZE 256
#define CHARS_AT_STRUCT 64

static nbox::Channel *getChannelInstance()
{
EOF

printf "    static nbox::%s instance(PIPE_NAME, 1024);\n",
    dllImpl ? "ClientSidePipe" : "ServerSidePipe"

print <<EOF
    return &instance;
}
static void writeToChannel(const void *buf, size_t size)
{
    getChannelInstance()->write(buf, size);
}
static void readFromChannel(void *buf, size_t size)
{
    getChannelInstance()->read(buf, size);
}
static void waitforReturn(void *rbuf);

enum FuncId {
    ID_RETURN,
EOF

func.each do |f|
    printf "    ID_%s,\n", f.name.upcase
end
printf "};\n"
printf "\n"

func.each do |f|
    arg = f.args
    maxlen = f.maxlenOfArgTypes

    if (f.dllside)
        # stab
        if (f.numOfArgs >= 1)
            printf "%s %s(\n", f.type, f.name
            rest = f.numOfArgs
            arg.each do |a|
                rest -= 1
                printf "    %-*s %s%s\n", maxlen, a.type, a.name,
                    rest != 0 ? "," : ")"
            end
        else
            printf "%s %s()\n", f.type, f.name
        end
        printf "{\n"

        # calling -----------------------------------
        printf "    struct W%s {\n", f.name
        printf "        %-*s len_;\n", maxlen, "size_t"
        printf "        %-*s id_;\n", maxlen, "FuncId"
        arg.each do |a|
            if (a.stringCopy)
                printf "        %-*s %s[CHARS_AT_STRUCT];\n", maxlen,
                    "char", a.name
            elsif (a.fullCopy)
                printf "        %-*s %s;\n", maxlen,
                    a.type.sub(/\*/, ""), a.name
            else
                printf "        %-*s %s;\n", maxlen, a.type, a.name
            end
        end
        printf "    } wbuf = {\n"
        printf "        sizeof(W%s),\n", f.name
        printf "        ID_%s,\n", f.name.upcase
        arg.each do |a|
            if (a.stringCopy)
                printf "        %s,\n", "0"
            elsif (a.fullCopy)
                printf "        *%s,\n", a.name
            else
                printf "        %s,\n", a.name
            end
        end
        printf "    };\n"
        arg.each do |a|
            if (a.stringCopy)
                printf "    strncpy(wbuf.%s, %s, CHARS_AT_STRUCT);\n",
                    a.name, a.name
                printf "    wbuf.%s[CHARS_AT_STRUCT-1] = '\\0';\n",
                    a.name
            end
        end
        printf "    writeToChannel(&wbuf, sizeof(wbuf));\n"
        printf "\n"

        # examining -----------------------------------
        # アライメントの問題があるので、charではなくlong longで確保する
        printf "    union R%s {\n", f.name
        printf "        long long x[RECEIVEBUF_SIZE/sizeof(long long)];\n"
        printf "        struct {\n", f.name
        printf "            %-*s len_;\n", maxlen, "size_t"
        printf "            %-*s id_;\n", maxlen, "FuncId"
        printf "            %-*s result;\n", maxlen, f.type
        arg.each do |a|
            if (a.changedByFunc)
                if (a.fullCopy)
                    printf "            %-*s %s;\n", maxlen,
                        a.type.sub(/\*/, ""), a.name
                else
                    printf "            %-*s %s;\n", maxlen, a.type, a.name
                end
            end
        end
        printf "        } o;\n"
        printf "    } rbuf;\n"
        printf "    waitforReturn(&rbuf);\n"
        arg.each do |a|
            if (a.changedByFunc)
                if (a.fullCopy)
                    printf "    %-*s = rbuf.o.%s;\n", maxlen,
                        "*" + a.name, a.name
                else
                    printf "    %-*s = rbuf.o.%s;\n", maxlen, a.name, a.name
                end
            end
        end
        printf "\n"

        printf "    return rbuf.o.result;\n"

    else # !dllside
        # skelton
        printf "static void skelton_%s(void *rbuf)\n", f.name
        printf "{\n"

        # examining -----------------------------------
        if (f.numOfArgs >= 1)
            printf "    struct R%s {\n", f.name
            printf "        %-*s len_;\n", maxlen, "size_t"
            printf "        %-*s id_;\n", maxlen, "FuncId"
            arg.each do |a|
                if (a.stringCopy)
                    printf "        %-*s %s[CHARS_AT_STRUCT];\n",
                        maxlen, "char", a.name
                else
                    printf "        %-*s %s;\n", maxlen,
                        a.type.sub(/\*/, a.fullCopy ? "" : "*"), a.name
                end
            end
            printf "    };\n"
            printf "    R%s *p = static_cast<R%s*>(rbuf);\n",
                f.name, f.name

            printf "    %s result = %s(\n", f.type, f.name
            rest = f.numOfArgs
            arg.each do |a|
                rest -= 1
                printf "        %sp->%s%s\n",
                    a.fullCopy ? "&" : "", a.name, rest != 0 ? "," : ");"
            end
        else
            printf "    %s result = %s();\n", f.type, f.name
        end
        printf "\n"

        # returning -----------------------------------
        printf "    struct W%s {\n", f.name
        printf "        %-*s len_;\n", maxlen, "size_t"
        printf "        %-*s id_;\n", maxlen, "FuncId"
        printf "        %-*s result;\n", maxlen, f.type
        arg.each do |a|
            if (a.changedByFunc)
                printf "        %-*s %s;\n", maxlen,
                    a.type.sub(/\*/, a.fullCopy ? "" : "*"), a.name
            end
        end
        printf "    } wbuf = {\n"
        printf "        sizeof(W%s),\n", f.name
        printf "        ID_%s,\n", "RETURN"
        printf "        result,\n"
        arg.each do |a|
            if (a.changedByFunc)
                printf "        p->%s,\n", a.name
            end
        end
        printf "    };\n"
        printf "    writeToChannel(&wbuf, sizeof(wbuf));\n"
    end

    printf "}\n"
    printf "\n"
end

print <<EOF
static void waitforReturn(void *rbuf)
{
    struct Head {
        size_t len;
        FuncId id;
    };
    Head *h = static_cast<Head *>(rbuf);
    char *p = static_cast<char *>(rbuf);

    for (;;) {
        readFromChannel(h, sizeof(Head));
        readFromChannel(p + sizeof(Head), h->len-sizeof(Head));
        switch (h->id) {
            case ID_RETURN: return;
EOF

func.each do |f|
    if (f.dllside)
        # stab
    else
        # skelton
        printf "            case ID_%s: skelton_%s(rbuf); break;\n",
            f.name.upcase, f.name
    end
end

print <<EOF
            default: break;
        }
    }
}

EOF

end

# ---- make foobar.cpp ------------------------------------------------
def putMain()
print <<EOF


int main(int argc, const char *argv[])
{
    char rbuf[RECEIVEBUF_SIZE];

    // wait for calling
    waitforReturn(rbuf);

    return 0;
}
EOF
end

# ---- main ---------------------------------------------------------------

dpiheader = ARGV[1];

# Foobar_dpi.h parser
file  = open(dpiheader, "r")
func  = []
token = []
while line = file.gets()
    if (line =~ /(^DPI_LINK_DECL\s*)/)
        line.sub!($1, "").chop!

        # 関数の終わりまでをひとつの文字列にする
        while nextline = file.gets()
            line += " " + nextline.sub(/^\s*/, "").chop
            if (nextline =~ /\);/)
                break
            end
        end

        # シミュレータ側から呼び出される関数には、DPI_DLLESPECが付いている
        dllside = false
        if (line =~ /(DPI_DLLESPEC\s*)/)
            line.sub!($1, "")
            dllside = true
        end

        #print "line : ", line
        while (line =~ /([A-Za-z0-9_\*]+)/)
            token.push($1)
            #print "token : ", $1, "\n"
            line.sub!($1, "")
        end

        prefix = ""
        type   = []
        name   = []
        count  = 0
        token.each do |t|
            if (t == "const" || t == "unsigned")
                prefix += t + " "
            else
                t = prefix + t
                prefix = ""
                count += 1
                if ((count & 1) == 1)
                    type.push(t)
                else
                    name.push(t)
                end
            end
        end
        #print "len ", type.length, " ", name.length, "\n"
        func.push(Function.new(type, name, dllside))
        token = []
    end
end
file.close

#func.each do |f|
#    f.dump
#    print "\n"
#end

commandline = sprintf "dpidivide.rb %s %s", ARGV[0], ARGV[1]

if (ARGV[0] == "-dll") then
    putImplementation(true, commandline, dpiheader, func)
elsif (ARGV[0] == "-exe") then
    func.each do |f|
        f.flipDllside
    end
    putImplementation(false, commandline, dpiheader, func)
    putMain()
end
