// This software is a part of NOODLYBOX.
// This software is distributed under the terms of the new BSD License.
// Copyright (c) 2009, molelord
// All rights reserved.

// '_X' means 'Negative'.
// 'i_' means 'internal'.

`timescale 1 ns / 1 ps
`default_nettype none

module SH7751 # (
    parameter MPU_ID = 0,
    parameter WIDTH  = 32,
    parameter TPD    = 6,   // TransParent Delay
    parameter TPDMIN = 1.5) // TransParent Delay(min)
(
    output wire             CS_X,
    output wire             OE_X,
    output wire             WE_X,
    output wire             BS_X,
    output wire             RDWR,
    output wire [31:0]      A,
    inout  wire [WIDTH-1:0] D,
    input  wire             CLK,
    input  wire             RESET_X);

    wire [31:0] i_A;
    wire [31:0] i_DIN;
    wire [31:0] i_DOUT;
    wire [31:0] i_CTRLI;
    wire [31:0] i_CTRLO;
    wire        i_HIZ;

    reg  [31:0] i_DOUT_DLY;

    MPU # (
        .MPU_ID(MPU_ID),
        .MPU_NAME("SH7751"))
    uMPU (
        .A     (i_A),
        .DIN   (i_DIN),
        .DOUT  (i_DOUT),
        .CTRLI (i_CTRLI),
        .CTRLO (i_CTRLO),
        .CLK   (CLK));

    // CTRLI mapping
    assign i_CTRLI[0] = RESET_X;

    assign #TPD A = i_A;

    // CTRLO mapping
    assign #TPD CS_X  = i_CTRLO[0];
    assign #TPD OE_X  = i_CTRLO[1];
    assign #TPD WE_X  = i_CTRLO[2];
    assign      i_HIZ = i_CTRLO[3];

    localparam UNDEF_DRIVE = 'hdeadbeef;

    // refer rjj09b0283_7751hm.pdf figure 23.13
    always @(i_HIZ) begin
        if (i_HIZ == 1'b0) begin
                            i_DOUT_DLY <= UNDEF_DRIVE;
            #(TPD)          i_DOUT_DLY <= i_DOUT;
        end
        else if (i_HIZ == 1'b1 && RESET_X == 1'b1) begin
            #(TPDMIN)       i_DOUT_DLY <= UNDEF_DRIVE;
            #(TPD - TPDMIN) i_DOUT_DLY <= {WIDTH{1'bz}};
        end
        else if (i_HIZ == 1'b1 && RESET_X == 1'b0) begin
            #(TPD)          i_DOUT_DLY <= {WIDTH{1'bz}};
        end
    end
    assign D = i_DOUT_DLY[WIDTH-1:0];

    assign #TPD BS_X  = i_CTRLO[4];
    assign #TPD RDWR  = i_CTRLO[5];

    generate if (WIDTH < 32) begin
        assign i_DIN = {{32-WIDTH{1'b0}}, D};
    end
    else begin
        assign i_DIN = D;
    end endgenerate

endmodule
