#ifndef CABSORBER_H_
#define CABSORBER_H_
// This software is a part of NOODLYBOX.
// This software is distributed under the terms of the new BSD License.
// Copyright (c) 2009, molelord
// All rights reserved.

#if defined(REALWORLD)
    typedef unsigned int   u32_c;
    typedef unsigned short u16_c;
    typedef unsigned char   u8_c;
    typedef signed   int   s32_c;
    typedef signed   short s16_c;
    typedef signed   char   s8_c;
#else
#   include <stdint.h> // uint??_t, int??_t
#   include "cFpga.h"  // Fpga
#   include "cMpuIf.h" // MpuIf

namespace nbox {

    // シミュレータと実機の違いを吸収するクラス
    // Fpgaクラスのインスタンスへのポインタを
    // ・保持している状態では
    //   操作される(メソッドが呼び出される)と、シミュレータ上のデバイスに対して
    //   アクセスを行う。
    // ・保持していない状態では
    //   操作される(メソッドが呼び出される)と、組み込み型のラッパーとして
    //   振舞う(デバイスアクセスしない)
    //   これは、operator+等で作る、計算式の途中経過のオブジェクトのための
    //   動作である。
    template <class T> class Absorber { 
    private:
        uint32_t addr; // constを付けるとFPGAレジスタを配列にできない
        T        value;
        Fpga     *fpga;
    public:
        Absorber() : addr(0), value(0), fpga(0)
        {}
        Absorber(T value) : addr(0), value(value), fpga(0)
        {}
        Absorber(uint32_t addr, Fpga *fpga) : addr(addr), value(0), fpga(fpga)
        {}
        ~Absorber()
        {}
        void set(uint32_t addr, Fpga *fpga)
        {
            this->addr = addr;
            this->fpga = fpga;
        }
        // read()とwrite(T value)は、Fpgaレジスタの幅 > Fpgaのバス幅 のときに、
        // 複数回のアクセスに直す役目を持つ(一般的なメモリコントローラと同様
        // の動作を行っているつもり)。
        // たとえば、バス幅が8bitのFpgaが持つ32bitのレジスタを読もうとすると、
        // 8bitのリードアクセスが4回発生する。
        T read() const
        {
            T result = this->value;

            if (fpga && fpga->mpu) {
                MpuIf *mpu = fpga->mpu;
                T     tmp;
                switch (sizeof(T) * 8) {
                    case 32 : switch (fpga->buswidth) {
                        case 32 : tmp  = mpu->readW(addr);
                                  break;
                        case 16 : tmp  = mpu->readH(addr);
                                  tmp |= mpu->readH(addr+2)<<16;
                                  break;
                        default : tmp  = mpu->readB(addr);
                                  tmp |= mpu->readB(addr+1)<<8;
                                  tmp |= mpu->readB(addr+2)<<16;
                                  tmp |= mpu->readB(addr+3)<<24;
                                  break;
                    }
                    break;
                    case 16 : switch (fpga->buswidth) {
                        case 32 :
                        case 16 : tmp  = mpu->readH(addr);
                                  break;
                        default : tmp  = mpu->readB(addr);
                                  tmp |= mpu->readB(addr+1)<<8;
                                  break;
                    }
                    break;
                    default : {
                        tmp = mpu->readB(addr);
                    }
                    break;
                }
                result = tmp;
            }
            return result;
        }
        void write(T value)
        {
            this->value = value;

            if (fpga && fpga->mpu) {
                MpuIf *mpu = fpga->mpu;
                switch (sizeof(T) * 8) {
                    case 32 : switch (fpga->buswidth) {
                        case 32 : mpu->writeW(addr, value);
                                  break;
                        case 16 : mpu->writeH(addr,   value);
                                  mpu->writeH(addr+2, value>>16);
                                  break;
                        default : mpu->writeB(addr,   value);
                                  mpu->writeB(addr+1, value>>8);
                                  mpu->writeB(addr+2, value>>16);
                                  mpu->writeB(addr+3, value>>24);
                                  break;
                    }
                    break;
                    case 16 : switch (fpga->buswidth) {
                        case 32 :
                        case 16 : mpu->writeH(addr, value);
                                  break;
                        default : mpu->writeB(addr, value);
                                  break;
                    }
                    break;
                    default : {
                        mpu->writeB(addr, value);
                    }
                    break;
                }
            }
        }
        Absorber operator+(const Absorber &rhs) const
        {
            // tmpはアドレスおよびFpgaへのポインタを与えずに作るので
            // tmpを操作しても、デバイスアクセスは発生しない
            Absorber tmp;
            tmp.value = this->read() + rhs.read();
            return tmp;
        }
        Absorber operator-(const Absorber &rhs) const
        {
            // tmpはアドレスおよびFpgaへのポインタを与えずに作るので
            // tmpを操作しても、デバイスアクセスは発生しない
            Absorber tmp;
            tmp.value = this->read() - rhs.read();
            return tmp;
        }
        Absorber &operator=(const Absorber &rhs)
        {
            this->write(rhs.read());
            return *this;
        }
        Absorber &operator=(T value)
        {
            this->write(value);
            return *this;
        }
        // 型変換キャスト
        // uint32_t等の型が必要なときに、暗黙的に呼ばれる
        operator T() {
            return this->read();
        }
    };

}

typedef nbox::Absorber<uint32_t> u32_c;
typedef nbox::Absorber<uint16_t> u16_c;
typedef nbox::Absorber<uint8_t>   u8_c;
typedef nbox::Absorber<int32_t>  s32_c;
typedef nbox::Absorber<int16_t>  s16_c;
typedef nbox::Absorber<int8_t>    s8_c;

#endif

#endif
