// This software is a part of NOODLYBOX.
// This software is distributed under the terms of the new BSD License.
// Copyright (c) 2009, molelord
// All rights reserved.

#include <iostream>     // cout
#include "cNamedPipeIo.h" // NamedPipeIo

namespace nbox {

    static bool debug = false;

    // コンストラクタ。パイプを作り、クライアントの接続を待つ。
    NamedPipeIo::NamedPipeIo(
        const char *readpipe_name, const char *writepipe_name) :
        Channel(), hReadPipe(0), hWritePipe(0)
    {
        static const DWORD bufsize = 1024;
        hReadPipe = CreateNamedPipe(
            readpipe_name,               // パイプ名
            PIPE_ACCESS_INBOUND,         // 読み出し方向
            PIPE_WAIT                    // ブロッキング・モード
            | PIPE_READMODE_BYTE         // バイト・モード
            | PIPE_TYPE_BYTE,
            PIPE_UNLIMITED_INSTANCES,    // インスタンス数の制限なし
            bufsize,                     // 出力バッファ・サイズ
            bufsize,                     // 入力バッファ・サイズ
            0,                           // WaitNamedPipeした場合のタイムアウト
            NULL);                       // セキュリティ属性なし
        if (hReadPipe == INVALID_HANDLE_VALUE) {
            int number = GetLastError();
            if (number == 123) {
                std::cerr << __FILE__ "(" << __LINE__ << ") : "
                    << "Pipename format is wrong" << std::endl;
                exit(number);
            } else {
                err_exit("CreateNamedPipe");
            }
        }
        if (debug) {
            std::cerr << "% Pipe has successfully created" << std::endl;
        }

        hWritePipe = CreateNamedPipe(
            writepipe_name,              // パイプ名
            PIPE_ACCESS_OUTBOUND,        // 書き込み方向
            PIPE_WAIT                    // ブロッキング・モード
            | PIPE_READMODE_BYTE         // バイト・モード
            | PIPE_TYPE_BYTE,
            PIPE_UNLIMITED_INSTANCES,    // インスタンス数の制限なし
            bufsize,                     // 出力バッファ・サイズ
            bufsize,                     // 入力バッファ・サイズ
            0,                           // WaitNamedPipeした場合のタイムアウト
            NULL);                       // セキュリティ属性なし
        if (hWritePipe == INVALID_HANDLE_VALUE) {
            int number = GetLastError();
            if (number == 123) {
                std::cerr << __FILE__ "(" << __LINE__ << ") : "
                    << "Pipename format is wrong" << std::endl;
                exit(number);
            } else {
                err_exit("CreateNamedPipe");
            }
        }
        if (debug) {
            std::cerr << "% Pipe has successfully created" << std::endl;
        }

        int rc;
        // クライアントの接続待ち
        rc = ConnectNamedPipe(hReadPipe, NULL);
        if (!rc) {
            err_close_exit("ConnectNamedPipe");
        }
        if (debug) {
            std::cerr << "% Client has connected" << std::endl;
        }
         // クライアントの接続待ち
        rc = ConnectNamedPipe(hWritePipe, NULL);
        if (!rc) {
            // 接続待ちをし始める前にクライアント側から接続済みになる
            // 可能性があるので、その場合はエラー扱いしない
            if (GetLastError() != ERROR_PIPE_CONNECTED) {
                err_close_exit("ConnectNamedPipe");
            }
        }
        if (debug) {
            std::cerr << "% Client has connected" << std::endl;
        }
    }

    NamedPipeIo::~NamedPipeIo()
    {
        CloseHandle(hReadPipe);
        CloseHandle(hWritePipe);
    }

    void NamedPipeIo::read(void *ptr, size_t len)
    {
        // バッファbuf(長さbuflen)に、lenで指定されたバイト数読み込む。
        static const size_t buflen = 256;

        // アドレス計算のために、char*にキャストする
        char *buf = static_cast<char *>(ptr);

        size_t already_read = 0;
        if (len >= buflen) {
            err_close_exit("Too small buffer");
        }

        while (already_read < len) {
            DWORD actual_len;
            int ret = ReadFile(hReadPipe, buf + already_read,
                len - already_read, &actual_len, NULL);
            if (debug) {
                std::cout << "Read "<< actual_len << " bytes" << std::endl;
            }
            already_read += actual_len;
            if (!ret) {
                err_close_exit("ReadFile");
            }
        }
        if (debug) {
            std::cout << "Server read: " << buf << std::endl;
        }
    }

    void NamedPipeIo::write(const void *ptr, size_t len)
    {
        // アドレス計算のために、char*にキャストする
        const char *buf = static_cast<const char *>(ptr);
        if (debug) {
            std::cerr << "% Writing to HANDLE " << hWritePipe << std::endl;
        }
        // バッファbufから、lenで指定されたバイト数書き込む。
        size_t already_written = 0;
        while (already_written < len) {
            DWORD actual_len;
            int ret = WriteFile(hWritePipe, buf + already_written,
                len - already_written, &actual_len, NULL);
            if (debug) {
                std::cout << "% Wrote "<< actual_len << " bytes"
                    << " (tried to write " << len << ")" << std::endl;
            }
            already_written += actual_len;
            if (!ret) {
                err_close_exit("WriteFile");
            }
        }
    }

    void NamedPipeIo::flush()
    {
        // リード側のパイプはflushする意味が無いので、ライト側だけ
        FlushFileBuffers(hWritePipe);
    }

    // エラーを表示して終了する
    void NamedPipeIo::err_exit(const char *str)
    {
        int number = GetLastError();
        std::cerr << "Error: " << str << " errcode=" << number << std::endl;
        exit(number);    
    }

    // エラーを表示して、パイプを閉じて終了する
    void NamedPipeIo::err_close_exit(const char *str)
    {
        int number = GetLastError();
        std::cerr << "Error: " << str << " errcode=" << number << std::endl;
        CloseHandle(hReadPipe);    
        CloseHandle(hWritePipe);    
        exit(number);    
    }

} // namespace nbox
