// This software is a part of NOODLYBOX.
// This software is distributed under the terms of the new BSD License.
// Copyright (c) 2009, molelord
// All rights reserved.

#include <cassert>       // assert()
#include <cstring>       // strcmp()

#include "cSh7751.h" // Sh7751
#include "cMpuFactoryEnlarger.h" // MpuFactoryEnlarger
#if defined(MPU_PIPE)
#   include "MPU_pipe.h" // v_waitClock()
#else
#   include "MPU_dpi.h"  // v_waitClock()
#endif

// fsm.tclおよびsh4config.tclでやっていたことのほとんどが、このクラスで
// 実装されている

namespace nbox {
    const uint32_t Sh7751::CS_X = 1<<0;
    const uint32_t Sh7751::OE_X = 1<<1;
    const uint32_t Sh7751::WE_X = 1<<2;
    const uint32_t Sh7751::HIZ  = 1<<3;
    const uint32_t Sh7751::BS_X = 1<<4;
    const uint32_t Sh7751::RDWR = 1<<5;

    static const uint32_t ReadBodyWidth  = 0;
    static const uint32_t WriteBodyWidth = 0;

    Sh7751::Sh7751(int mpu_id) : Mpu(mpu_id),
        state(&Sh7751::stFetch), subState(FIRST),
        execAtFall(&Sh7751::exEmpty), execAtRise(&Sh7751::exEmpty)
    {
    }

    Sh7751::~Sh7751()
    {
    }

    Mpu *Sh7751::create(int mpu_id, const char *mpu_name)
    {
        assert(mpu_name != 0);
        Mpu *p = 0;
        if (std::strcmp("SH7751", mpu_name) == 0) {
            p = new Sh7751(mpu_id);
        }
        return p;
    }

    bool Sh7751::reflect(
        svLogicVecVal       *a,
        svLogicVecVal       *dout,
        const svLogicVecVal *ctrli,
        svLogicVecVal       *ctrlo,
        int                  clk)
    {
        // 入力信号をクラスのメンバ変数に取り込む
        this->ctrli = *ctrli;

        if (!reset()) {
            if (clk) {
                need_next_rising = false;
                (this->*execAtRise)();
                this->execAtRise = &Sh7751::exEmpty;
                (this->*state)();
            }
            else {
                (this->*execAtFall)();
                this->execAtFall = &Sh7751::exEmpty;
            }
        }

        // 出力信号に対して、クラスの内部状態を反映させる
        *a     = this->a;
        *dout  = this->dout;
        *ctrlo = this->ctrlo;

        return need_next_rising;
    }

    void Sh7751::reflect_din(
        const svLogicVecVal *din)
    {
        // 入力信号をクラスのメンバ変数に取り込む
        this->din = *din;
    }

    void Sh7751::nop(uint32_t clocks)
    {
        pack.count     = clocks;
        this->state    = &Sh7751::stNop;
        this->subState = FIRST;

        // 終わるまでループ
        do {
            v_waitClock();
        } while (this->state != &Sh7751::stFetch);
    }

    uint32_t Sh7751::readGeneric(uint32_t addr, int bits)
    {
        assert(bits == 32 | bits == 16 | bits == 8);

        uint32_t result = 0xdeadbeefU;
        uint32_t mask   = 0xffffffffU;
        if (bits == 16) {
            result = 0xdead;
            mask   = 0xffff;
        }
        else if (bits == 8) {
            result = 0xde;
            mask   = 0xff;
        }

        // これからリードを始めるので、まずは失敗ということにしておく
        this->read_success = false;

        pack.addr      = addr;
        this->state    = &Sh7751::stRead;
        this->subState = FIRST;

        // 読み出しが終わるまでループ
        do {
            v_waitClock();
        } while (this->state != &Sh7751::stFetch);

        // 'x'または'z'ならば、bvalは0でない値になっているので、もし
        // 0であれば正常に読み出せたことになる
        uint32_t bval = this->din.bval & mask;
        if (bval == 0) {
            this->read_success = true;
            result = din.aval & mask;
        }

        return result;
    }

    uint8_t  Sh7751::readB(uint32_t addr)
    {
        return readGeneric(addr, 8);
    }
    uint16_t Sh7751::readH(uint32_t addr)
    {
        return readGeneric(addr, 16);
    }
    uint32_t Sh7751::readW(uint32_t addr)
    {
        return readGeneric(addr, 32);
    }

    void Sh7751::writeB(uint32_t addr, uint8_t  data)
    {
        writeW(addr, data);
    }
    void Sh7751::writeH(uint32_t addr, uint16_t data)
    {
        writeW(addr, data);
    }
    void Sh7751::writeW(uint32_t addr, uint32_t data)
    {
        pack.addr      = addr;
        pack.data      = data;
        this->state    = &Sh7751::stWrite;
        this->subState = FIRST;

        // 書き込み動作が終わるまでループ
        do {
            v_waitClock();
        } while (this->state != &Sh7751::stFetch);
    }

    void Sh7751::activate(uint32_t x, bool act)
    {
        uint32_t set = 0;
        uint32_t clr = 0;

        if (x & CS_X) { clr |= CS_X; }
        if (x & OE_X) { clr |= OE_X; }
        if (x & WE_X) { clr |= WE_X; }
        if (x & HIZ)  { set |= HIZ; }
        if (x & BS_X) { clr |= BS_X; }
        if (x & RDWR) { set |= RDWR; }

        // 逆に使えばdeactivateになる
        if (act)
            this->ctrlo.aval = (this->ctrlo.aval | set) & ~clr;
        else
            this->ctrlo.aval = (this->ctrlo.aval | clr) & ~set;
    }

    void Sh7751::deactivate(uint32_t x)
    {
        activate(x, false);
    }

    void Sh7751::exEmpty() {}

    // Tcl実装と違って、C++実装では特に何もする必要が無い
    void Sh7751::stFetch() {}

    void Sh7751::stNop()
    {
        switch (this->subState) {
            case FIRST :
                deactivate(CS_X);
                this->subState = NOPTAIL;
                break;
            case NOPTAIL :
            default      :
                break;
        }

        pack.count--;
        if (pack.count == 0) {
            this->state    = &Sh7751::stFetch;
        }
    }

    void Sh7751::exEndOfReadT1()
    {
        activate(OE_X);
        deactivate(BS_X);
    }

    void Sh7751::exEndOfReadT2()
    {
        deactivate(OE_X);
    }

    void Sh7751::stRead()
    {
        switch (this->subState) {
            case FIRST :
                // Read T1
                activate(CS_X | BS_X | RDWR);
                this->a.aval     = pack.addr;
                this->execAtRise = &Sh7751::exEndOfReadT1;
                pack.count       = ReadBodyWidth;

                this->subState = READBODY;
                if (pack.count == 0) {
                    this->subState = READTAIL;
                }
                break;
            case READBODY :
                // Read Tw
                pack.count--;
                if (pack.count == 0) {
                    this->subState = READTAIL;
                }
                break;
            case READTAIL :
            default       :
                // Read T2
                this->execAtRise = &Sh7751::exEndOfReadT2;
                this->state      = &Sh7751::stFetch;
                need_next_rising = true;
                break;
        }
    }

    void Sh7751::exActivateWE()
    {
        activate(WE_X);
    }
    void Sh7751::exEndOfWriteT1()
    {
        this->dout.aval = pack.data;
        deactivate(BS_X | HIZ);
    }

    void Sh7751::exDeactivateWE()
    {
        deactivate(WE_X);
    }
    void Sh7751::exEndOfWriteT2()
    {
        activate(HIZ);
    }

    void Sh7751::stWrite()
    {
        switch (this->subState) {
            case FIRST :
                // Write T1
                activate(CS_X | BS_X);
                deactivate(RDWR);
                this->a.aval     = pack.addr;

                this->execAtFall = &Sh7751::exActivateWE;
                this->execAtRise = &Sh7751::exEndOfWriteT1;
                pack.count     = WriteBodyWidth;
                this->subState = WRITEBODY;
                if (pack.count == 0) {
                    this->subState = WRITETAIL;
                }
                break;
            case WRITEBODY :
                // Write Tw
                pack.count--;
                if (pack.count == 0) {
                    this->subState = WRITETAIL;
                }
                break;
            case WRITETAIL :
            default        :
                // Write T2
                this->execAtFall = &Sh7751::exDeactivateWE;
                this->execAtRise = &Sh7751::exEndOfWriteT2;
                this->state = &Sh7751::stFetch;
                break;
        }
    }
 
    // この記述によって、ライブラリ(DLL)のロード時点でcreate()が
    // MpuFactoryに対して登録される
    static MpuFactoryEnlarger x(Sh7751::create);

} // namespace nbox
