// This software is a part of NOODLYBOX.
// This software is distributed under the terms of the new BSD License.
// Copyright (c) 2009, molelord
// All rights reserved.

#include <cstdlib>  // abort(), exit(), strtol()
#include <cstring>  // strlen()
#include <iostream> // cerr

#include "MPU_pipe.h"

#include "cMpu.h"         // Mpu
#include "cMpuFactory.h"  // MpuFactory
#include "inlineutil.h"   // nbox::hton()
#include "cNamedPipeIo.h" // NamedPipeIo

enum {EXTC2V, A, DOUT, CTRLO};
static const unsigned int REQ_NORMAL = 0x00000000U;
static const unsigned int ACK_NORMAL = 0x00000001U;
static const unsigned int REQ_STOP   = 0x00000002U;
static const unsigned int ACK_STOP   = 0x00000003U;

static void usage(void)
{
    std::cerr << "Usage : c_side_pipe mpu_id mpu_name" << std::endl;
    std::cerr << "   ex : c_side_pipe 0 SH7751" << std::endl;
}

static svLogicVecVal *bufTosvLogicVecVal(
    const unsigned char *buf, int width, svLogicVecVal *obj)
{
    assert(buf != 0);
    assert(0 < width && width <= 32);
    assert(obj != 0);

    unsigned a = 0;
    unsigned b = 0;

    for (int i = 0; i < width; i++) {
        switch (buf[i]) {
            // Verilog
            case '0' : a = (a<<1) | 0; b = (b<<1) | 0; break;
            case '1' : a = (a<<1) | 1; b = (b<<1) | 0; break;
            case 'z' : a = (a<<1) | 0; b = (b<<1) | 1; break;
            case 'x' : a = (a<<1) | 1; b = (b<<1) | 1; break;

            // VHDL std_logic_vectorをwriteすると0～7が出てくる
            // svLogicVecValは4値なので、U,W,L,HはX,X,0,1に丸めている
            case 0   : a = (a<<1) | 1; b = (b<<1) | 1; break; // U→X
            case 1   : a = (a<<1) | 1; b = (b<<1) | 1; break; // X
            case 2   : a = (a<<1) | 0; b = (b<<1) | 0; break; // 0
            case 3   : a = (a<<1) | 1; b = (b<<1) | 0; break; // 1
            case 4   : a = (a<<1) | 0; b = (b<<1) | 1; break; // Z
            case 5   : a = (a<<1) | 1; b = (b<<1) | 1; break; // W→X
            case 6   : a = (a<<1) | 0; b = (b<<1) | 0; break; // L→0
            case 7   : a = (a<<1) | 1; b = (b<<1) | 0; break; // H→1

            default  : abort(); break;
        }
    }
    obj->aval = a;
    obj->bval = b;
    return obj;
}

// 美しくないが、暫定的にグローバル変数で管理する
static nbox::Channel *gCh;
static nbox::Mpu     *gMpu;

int main(int argc, const char *argv[])
{
    // 引数が規定の数に満たない場合
    if (argc < 3) {
        usage();
        exit(EXIT_FAILURE);
    }
    // 数値に変換不可能な文字列がidとして指定されていた場合
    char *endptr;
    int mpu_id_i = strtol(argv[1], &endptr, 10);
    if (argv[1] != '\0' && *endptr != '\0') {
        usage();
        exit(EXIT_FAILURE);
    }

    // 引数を元にパイプのファイル名を作って、NamedPipeIoコンストラクタに渡す
    static const char pipe_name_v2c[] = "\\\\.\\pipe\\v2c";
    static const char pipe_name_c2v[] = "\\\\.\\pipe\\c2v";
    const char *mpu_id   = argv[1];
    const char *mpu_name = argv[2];
    size_t len = std::strlen(pipe_name_v2c)
        + std::strlen(mpu_id) + std::strlen(mpu_name);

    char *v2c = new char[len + 1];
    char *c2v = new char[len + 1];
    std::sprintf(v2c, "%s%s%s", pipe_name_v2c, mpu_id, mpu_name);
    std::sprintf(c2v, "%s%s%s", pipe_name_c2v, mpu_id, mpu_name);
    nbox::Channel *ch = new nbox::NamedPipeIo(v2c, c2v);
    gCh = ch;
    delete [] c2v;
    delete [] v2c;

    // c_startup1st()の代わりの処理
    nbox::MpuFactory *factory = nbox::MpuFactory::getInstance();
    nbox::Mpu *mpu = factory->create(mpu_id_i, mpu_name);
    gMpu = mpu;

    // c_startup2nd()の代わりの処理
    mpu->startup();

    // 後始末
    delete mpu; gMpu = 0;

    // REQ_STOPを送って、ACK_STOPが返るまで待つ
    {
        unsigned int wbuf[4];
        wbuf[EXTC2V] = nbox::hton(REQ_STOP);
        wbuf[A]      = nbox::hton(0U);
        wbuf[DOUT]   = nbox::hton(0U);
        wbuf[CTRLO]  = nbox::hton(0U);
        ch->write(wbuf, sizeof(wbuf));
        ch->flush();

        unsigned char rbuf[32 * 3];
        unsigned char *EXTV2C = &rbuf[32 * 0];
        svLogicVecVal extv2c;
        do {
            ch->read(rbuf, sizeof(rbuf));
            bufTosvLogicVecVal(EXTV2C, 32, &extv2c);
        } while (extv2c.aval != ACK_STOP);
    }

    delete ch; gCh = 0;

	return 0;
}

// DPI-C向けに作ったソースからの移植なので、
// 'v_'で始まる関数 は Verilog-HDL による実装だったもの
// int v_stop(void) 未実装
// int v_write(int x) 未実装

// Verilog HDLの$fread()はビッグエンディアンを期待しているようなので、
// 送る前にhton()で変換を行っている
int v_waitClock(void)
{
    nbox::Mpu     *mpu = gMpu;
    nbox::Channel *ch  = gCh;
    svLogicVecVal a;
    svLogicVecVal dout;
    svLogicVecVal ctrlo;

    // v_waitClock()が呼ばれたときに、前回呼ばれたときの状態を
    // 知る必要があるため、static変数にしてある。
    static bool need_next_rising = false;
    static svLogicVecVal ctrli;

    // MPU.vの出力信号はバイナリ EXTC2V(拡張)とAとDOUTとCTRLOで4つ
    unsigned int wbuf[4];

    // MPU.vの入力信号はテキスト(2進数32桁) EXTV2C(拡張)とDINとCTRLIの3つ
    unsigned char rbuf[32 * 3];
    //unsigned char *EXTV2C = &rbuf[32 * 0];
    unsigned char *DIN    = &rbuf[32 * 1];
    unsigned char *CTRLI  = &rbuf[32 * 2];

    // クロックの立ち上がりでの処理 ---------------------------------
    if (!need_next_rising) {
        ch->read(rbuf, sizeof(rbuf));
        bufTosvLogicVecVal(CTRLI, 32, &ctrli);
    }
    need_next_rising = mpu->reflect(&a, &dout, &ctrli, &ctrlo, 1);
    wbuf[EXTC2V] = nbox::hton(REQ_NORMAL);
    wbuf[A]      = nbox::hton(a.aval);
    wbuf[DOUT]   = nbox::hton(dout.aval);
    wbuf[CTRLO]  = nbox::hton(ctrlo.aval);
    ch->write(wbuf, sizeof(wbuf));
    // --------------------------------- クロックの立ち上がりでの処理

    // クロックの立ち下がりでの処理 ---------------------------------
    // 高速化のため、立ち下がりでのDIN, CTRLIは貰わないことにした
    //ch->read(rbuf, sizeof(rbuf));
    //bufTosvLogicVecVal(CTRLI, 32, &ctrli);
    need_next_rising = mpu->reflect(&a, &dout, &ctrli, &ctrlo, 0);
    wbuf[EXTC2V] = nbox::hton(REQ_NORMAL);
    wbuf[A]      = nbox::hton(a.aval);
    wbuf[DOUT]   = nbox::hton(dout.aval);
    wbuf[CTRLO]  = nbox::hton(ctrlo.aval);
    ch->write(wbuf, sizeof(wbuf));
    ch->flush();
    // --------------------------------- クロックの立ち下がりでの処理

    // データバスの値を取り込む必要がある場合は、v_waitClock()
    // を抜ける前にreadだけ行う
    if (need_next_rising) {
        svLogicVecVal din;
        ch->read(rbuf, sizeof(rbuf));
        bufTosvLogicVecVal(DIN,   32, &din);
        bufTosvLogicVecVal(CTRLI, 32, &ctrli);
        mpu->reflect_din(&din);
    }

	return 0;
}
