// This software is a part of NOODLYBOX.
// This software is distributed under the terms of the new BSD License.
// Copyright (c) 2008-2009, molelord
// All rights reserved.

#include <stdio.h>  // fopen(), printf(), ...
#include <stdlib.h> // EXIT_FAILURE
#include <string.h> // strcmp()
#include <assert.h> // assert()
#include <stdarg.h> // va_list, va_start(), ...
#include "noodlybox.h"

// stdoutに書き込むたびにfflush()をしている理由
// すぐに伝えてやらないと、それを待っているシミュレータが止まってしまうから。

static FILE *logfp;
static const char * const logfilename = "readfrompipe.log";
static char buf[256];
static int read_success;

// NOODLYBOXのデータバス幅は16進数で何文字か
static const size_t BUSWIDTH_HEX = 32/4;

// CRまたはLFが見つかったら'\0'にする。
// 注意：破壊的な動作をする(引数に渡されたバッファの中を書き換える)。
static char *chomp(char *str)
{
    char *p;
    for (p = str; *p != '\0';p++) {
        if (*p == '\r' || *p == '\n') {
            *p = '\0';
            break;
        }
    }
    return str;
}

// エラー出力
static void errprintf(const char *format, ...)
{
    va_list args;
    assert(format != NULL);

    va_start(args, format);
    vfprintf(stderr, format, args);
    if (logfp) {
        vfprintf(logfp, format, args);
    }
    va_end(args);
}

static void nop(u32 clocks)
{
    printf("opNop %u\n", clocks);
    fflush(stdout);
}

static void sleep(u32 time, const char *unit)
{
    assert(strcmp(unit,"ms")==0 || strcmp(unit,"us")==0 || strcmp(unit,"ns")==0);

    printf("opSleep %u %s\n", time, unit);
    fflush(stdout);
}

static int readSuccess(void)
{
    return read_success;
}

static u32 readGeneric(u32 addr, int bits)
{
    u32 result;
    read_success = 0;

    assert(bits == 32 || bits == 16 || bits == 8);

    result = 0xdeadbeef;
    if (bits == 16) {
        result = 0xdead;
    }
    else if (bits == 8) {
        result = 0xde;
    }

    printf("opRead 16#%08x\n", addr);
    puts(  "putReadResult");
    fflush(stdout);

    if (fgets(buf, sizeof(buf), stdin) == NULL) {
        // 読み出しができなかったとき
        errprintf("Error: read%c(%08x) failed in fgets().\n",
            (bits == 32) ? 'W' : (bits == 16) ? 'H' : 'B', addr);
        exit(EXIT_FAILURE);
    }

    fprintf(logfp, "# %s", buf);
    if (strlen(chomp(buf)) == BUSWIDTH_HEX) {
        unsigned i;
        u32 tmp;
        char *endptr;

        // 最左端を文字位置0として、文字位置valid_posからは有効桁
        unsigned valid_pos = (32 - bits) / 4;

        for (i = 0; i < BUSWIDTH_HEX; i++) {
            if (buf[i] == 'X' || (i >= valid_pos && buf[i] == 'Z')) {
                // 不定またはHi-Zの信号が混ざっている
                return result;
            }
        }

        // 不定またはHi-Zがなければここへ到達する
        tmp = strtoul(buf + valid_pos, &endptr, 16);
        if (*endptr == '\0') {
            // 変換成功
            result = tmp;
            read_success = 1;
        }
    }
    return result;
}

static u8 readB(u32 addr)
{
    return (u8)readGeneric(addr, 8);
}

static u16 readH(u32 addr)
{
    return (u16)readGeneric(addr, 16);
}

static u32 readW(u32 addr)
{
    return readGeneric(addr, 32);
}

static void writeB(u32 addr,  u8 data)
{
    printf("opWrite 16#%08x 16#%08x\n", addr, data);
    fflush(stdout);
}

static void writeH(u32 addr, u16 data)
{
    printf("opWrite 16#%08x 16#%08x\n", addr, data);
    fflush(stdout);
}

static void writeW(u32 addr, u32 data)
{
    printf("opWrite 16#%08x 16#%08x\n", addr, data);
    fflush(stdout);
}

// 注：返却されたポインタのfreeやdeleteは「してはならない」。
Processor *Processor_instance(void)
{
    static Processor funcinst;
    static int       initialized;
    if (!initialized) {
        initialized = 1;
        funcinst.nop         = nop;
        funcinst.sleep       = sleep;
        funcinst.readB       = readB;
        funcinst.readH       = readH;
        funcinst.readW       = readW;
        funcinst.readSuccess = readSuccess;
        funcinst.writeB      = writeB;
        funcinst.writeH      = writeH;
        funcinst.writeW      = writeW;
    }
    return &funcinst;
}

static void force(const char *target, const char *value, u32 delayed)
{
    assert(target != NULL);
    assert(value  != NULL);

    printf("simForce %s %s \"%u ns\"\n", target, value, delayed);
    fflush(stdout);
}

static char *examine(char *resultbuf, const char *target)
{
    assert(resultbuf != NULL);
    assert(target    != NULL);

    printf("simExamine %s\n", target);
    fflush(stdout);

    if (fgets(buf, sizeof(buf), stdin)) {
        fprintf(logfp, "# %s", buf);
        strcpy(resultbuf, chomp(buf));
    }
    else {
        // 読み出しができなかったとき
        errprintf("Error: examine() failed in fgets().\n");
        exit(EXIT_FAILURE);
    }
    return resultbuf;
}

static void config(const char *filename)
{
    assert(filename != NULL);
    printf("source %s\n", filename);
}

static void endOfConfig(void)
{
    puts("endOfConfig");
    fflush(stdout);
}


static void rawoutput(const char *str)
{
    assert(str != NULL);

    puts(str);
    fflush(stdout);
}

static char *rawinput(char *resultbuf)
{
    assert(resultbuf != NULL);

    if (fgets(buf, sizeof(buf), stdin)) {
        fprintf(logfp, "# %s", buf);
        strcpy(resultbuf, chomp(buf));
    }
    else {
        // 読み出しができなかったとき
        errprintf("Error: rawinput() failed in fgets().\n");
        exit(EXIT_FAILURE);
    }
    return resultbuf;
}

static void endOfSimulation(void)
{
    //puts("endOfSimulation");
    //fflush(stdout);
}

// 後始末
static void terminator(void)
{
    fclose(logfp);
}

// 注：返却されたポインタのfreeやdeleteは「してはならない」。
Simulator *Simulator_instance(void)
{
    static Simulator funcinst;
    static int       initialized;
    if (!initialized) {
        initialized = 1;

        logfp = fopen(logfilename, "w");
        if (logfp == NULL) {
            errprintf("Error: %s could not be created.\n", logfilename);
            exit(EXIT_FAILURE);
        }

        funcinst.config          = config;
        funcinst.endOfConfig     = endOfConfig;
        funcinst.force           = force;
        funcinst.examine         = examine;
        funcinst.rawoutput       = rawoutput;
        funcinst.rawinput        = rawinput;
        funcinst.endOfSimulation = endOfSimulation;

        atexit(terminator);
    }
    return &funcinst;
}
