// This software is a part of NOODLYBOX.
// This software is distributed under the terms of the new BSD License.
// Copyright (c) 2009, molelord
// All rights reserved.

`timescale 1 ns / 1 ns

module MPU # (
    parameter MPU_ID   = 0,
    parameter MPU_NAME = "")
(
    output wire [31:0] A,
    input  wire [31:0] DIN,
    output wire [31:0] DOUT,
    input  wire [31:0] CTRLI,
    output wire [31:0] CTRLO,
    input  wire        CLK);

    // C側で名前付きパイプを双方向にしてあっても、それに$fwrite()しようと
    // すると失敗した。そこで、C側で単方向の名前付きパイプを2本作成し、
    // WriteとReadは別々のパイプを経由するようにした。
    integer    fdw; // file descriptor for writing
    integer    fdr; // file descriptor for reading

    integer    rc; // return code

    localparam RBUF_WORDS = 4;
    reg [RBUF_WORDS*32-1:0] i_RBUF;

    reg  [7:0] i_MPU_ID = MPU_ID + 'h30; // integer to char

    localparam PAD        = 32'h00000000;
    localparam REQ_NORMAL = 32'h00000000;
    localparam ACK_NORMAL = 32'h80000001;
    localparam REQ_STOP   = 32'h00000002;
    localparam ACK_STOP   = 32'h80000003;

    wire [31:0] i_EXTV2C  = ACK_NORMAL;

    // 拡張情報を扱う
    task handle_extension; begin
        if (i_RBUF[RBUF_WORDS*32-1:RBUF_WORDS*32-32] == REQ_STOP) begin
            $fwriteb(fdw, ACK_STOP[31:0], PAD, PAD);
            $fflush(fdw);
            $fclose(fdw);
            $fclose(fdr);
            $stop;
        end
    end endtask

    initial begin
        fdw = $fopen({"\\\\.\\pipe\\v2c", i_MPU_ID, MPU_NAME}, "wb");
        fdr = $fopen({"\\\\.\\pipe\\c2v", i_MPU_ID, MPU_NAME}, "rb");

        forever begin
            @(posedge CLK);
            $fwriteb(fdw, i_EXTV2C, DIN, CTRLI);
            // flushしないと、バッファに溜まるだけでC側に伝わらない
            $fflush(fdw);
            rc = $fread(i_RBUF, fdr);
            handle_extension();

            @(negedge CLK);
            // 高速化のため、立ち下がりでのDIN, CTRLIは伝えないことにした
            //$fwriteb(fdw, i_EXTV2C, DIN, CTRLI);
            // flushしないと、バッファに溜まるだけでC側に伝わらない
            //$fflush(fdw);
            rc = $fread(i_RBUF, fdr);
            handle_extension();
        end
    end
    assign {A, DOUT, CTRLO} = i_RBUF[(RBUF_WORDS-1)*32-1:0];
        
endmodule
