-- This software is a part of NOODLYBOX.
-- This software is distributed under the terms of the new BSD License.
-- Copyright (c) 2009, molelord
-- All rights reserved.

library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all; -- conv_std_logic_vector

entity MPU is
    generic (
        MPU_ID   : integer := 0;
        MPU_NAME : string  := "");
    port (
        A     : out std_logic_vector(31 downto 0);
        DIN   : in  std_logic_vector(31 downto 0);
        DOUT  : out std_logic_vector(31 downto 0);
        CTRLI : in  std_logic_vector(31 downto 0);
        CTRLO : out std_logic_vector(31 downto 0);
        CLK   : in  std_logic);
end MPU;

architecture SIM of MPU is
    -- C側で名前付きパイプを双方向にしてあっても、それに$fwrite()しようと
    -- すると失敗した。そこで、C側で単方向の名前付きパイプを2本作成し、
    -- WriteとReadは別々のパイプを経由するようにした。

    subtype  BLOCK_FOR_WRITE is std_logic_vector(32*3-1 downto 0);
    type     BLOCK_FOR_READ  is array (0 to 3) of integer;
    type     F_OF_BW         is file of BLOCK_FOR_WRITE;
    type     F_OF_BR         is file of BLOCK_FOR_READ;
    constant i_MPU_ID : character := character'val(MPU_ID + 16#30#); -- +'0'

    file fdw : F_OF_BW open WRITE_MODE is "\\.\pipe\v2c" & i_MPU_ID & MPU_NAME;
    file fdr : F_OF_BR open READ_MODE  is "\\.\pipe\c2v" & i_MPU_ID & MPU_NAME;

    subtype  slv32 is std_logic_vector(31 downto 0);
    constant REQ_NORMAL : slv32 := CONV_STD_LOGIC_VECTOR(16#00000000#, 32);
    constant ACK_NORMAL : slv32 := CONV_STD_LOGIC_VECTOR(16#00000001#, 32);
    constant REQ_STOP   : slv32 := CONV_STD_LOGIC_VECTOR(16#00000002#, 32);
    constant ACK_STOP   : slv32 := CONV_STD_LOGIC_VECTOR(16#00000003#, 32);

    -- integerをエンディアン変換したものを、std_logic_vector形式で返す
    function bswap(X:integer) return std_logic_vector is
        variable Y : std_logic_vector(31 downto 0);
    begin
        Y := conv_std_logic_vector(X, 32);
        return Y(7 downto 0) & Y(15 downto 8)
            & Y(23 downto 16) & Y(31 downto 24);
    end;

    signal ENDOFSIM : std_logic := '0';

    -- 拡張情報を扱う
    procedure handle_extension(
        BUF : in BLOCK_FOR_READ; signal ENDOFSIM : out std_logic) is
        constant PAD : slv32 := (others=>'0');
        variable EXTC2V : slv32;
    begin
        EXTC2V := bswap(BUF(0));
        if (EXTC2V = REQ_STOP) then
            write(fdw, ACK_STOP & PAD & PAD);
            file_close(fdw);
            file_close(fdr);
            ENDOFSIM <= '1';
            wait;
        end if;
    end;

    -- flushがサポートされたのはVHDL2008からで、一方ModelSimはVHDL2002までしか
    -- サポートしていない。そこで、ModelSimにおいては、プロジェクトファイルを
    -- create_project.tclで修正することでバッファリングを禁止した。

begin

    process
        --variable WBUF : BLOCK_FOR_WRITE;
        variable RBUF   : BLOCK_FOR_READ;
        constant EXTV2C : slv32 := ACK_NORMAL;
    begin
        wait until CLK'event and CLK = '1';
        write(fdw, EXTV2C & DIN & CTRLI);
        -- flushしないと、バッファに溜まるだけでC側に伝わらない
        --flush(fdw);
        read(fdr, RBUF);
        -- c_side_pipe.cppにおいてもエンディアン変換しているので無駄ではあるが
        -- C側の実装をVerilog HDL/VHDL共通にしたいので、よしとした。
        handle_extension(RBUF, ENDOFSIM);
        A     <= bswap(RBUF(1));
        DOUT  <= bswap(RBUF(2));
        CTRLO <= bswap(RBUF(3));

        wait until CLK'event and CLK = '0';
        -- 高速化のため、立ち下がりでのDIN, CTRLIは伝えないことにした
        --write(fdw, EXTV2C & DIN & CTRLI);
        -- flushしないと、バッファに溜まるだけでC側に伝わらない
        --flush(fdw);
        read(fdr, RBUF);
        handle_extension(RBUF, ENDOFSIM);
        A     <= bswap(RBUF(1));
        DOUT  <= bswap(RBUF(2));
        CTRLO <= bswap(RBUF(3));
    end process;

end;
