-- This software is a part of NOODLYBOX.
-- This software is distributed under the terms of the new BSD License.
-- Copyright (c) 2009, molelord
-- All rights reserved.

-- '_X' means 'Negative'.
-- 'i_' means 'internal'.

library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all; -- conv_std_logic_vector()
use work.MPU_PKG.all;

entity SH7751 is
    generic (
        MPU_ID : integer := 0;
        WIDTH  : integer := 32;
        TPD    : time    := 6 ns;    -- TransParent Delay
        TPDMIN : time    := 1.5 ns); -- TransParent Delay(min)
    port (
        CS_X    : out   std_logic;
        OE_X    : out   std_logic;
        WE_X    : out   std_logic;
        BS_X    : out   std_logic;
        RDWR    : out   std_logic;
        A       : out   std_logic_vector(31 downto 0);
        D       : inout std_logic_vector(WIDTH-1 downto 0);
        CLK     : in    std_logic;
        RESET_X : in    std_logic);
end;

architecture SIM of SH7751 is
    signal i_A        : std_logic_vector(31 downto 0);
    signal i_DIN      : std_logic_vector(31 downto 0);
    signal i_DOUT     : std_logic_vector(31 downto 0);
    signal i_CTRLI    : std_logic_vector(31 downto 0);
    signal i_CTRLO    : std_logic_vector(31 downto 0);
    signal i_HIZ      : std_logic;

    signal i_DOUT_DLY : std_logic_vector(31 downto 0);

    constant UNDEF_DRIVE : std_logic_vector(31 downto 0) :=
        conv_std_logic_vector(16#deadbeef#, 32);

begin
    uMPU : MPU generic map (
        MPU_ID   => MPU_ID,
        MPU_NAME => "SH7751")
    port map (
        A     => i_A,
        DIN   => i_DIN,
        DOUT  => i_DOUT,
        CTRLI => i_CTRLI,
        CTRLO => i_CTRLO,
        CLK   => CLK);

    -- CTRLI mapping
    i_CTRLI(0) <= RESET_X;

    A <= transport i_A after TPD;

    -- CTRLO mapping
    CS_X  <= transport i_CTRLO(0) after TPD;
    OE_X  <= transport i_CTRLO(1) after TPD;
    WE_X  <= transport i_CTRLO(2) after TPD;
    i_HIZ <= i_CTRLO(3);

    -- refer rjj09b0283_7751hm.pdf figure 23.13
    process (i_HIZ) begin
        if (i_HIZ = '0') then
            i_DOUT_DLY <= UNDEF_DRIVE;
            i_DOUT_DLY <= transport I_DOUT after TPD;
        elsif (i_HIZ = '1' and RESET_X = '1') then
            i_DOUT_DLY <= transport UNDEF_DRIVE   after TPDMIN;
            i_DOUT_DLY <= transport (others=>'Z') after (TPD - TPDMIN);
        elsif (i_HIZ = '1' and RESET_X = '0') then
            i_DOUT_DLY <= transport (others=>'Z') after TPD;
        end if;
    end process;
    D <= i_DOUT_DLY(WIDTH-1 downto 0);

    BS_X <= transport i_CTRLO(4) after TPD;
    RDWR <= transport i_CTRLO(5) after TPD;

    i_DIN(D'range) <= D;
    din_others : if (WIDTH < 32) generate
        i_DIN(i_DIN'left downto WIDTH) <= (others=>'0');
    end generate;

end;
