# This software is a part of NOODLYBOX.
# This software is distributed under the terms of the new BSD License.
# Copyright (c) 2008-2009, molelord
# All rights reserved.

source ../../tcl/noodlybox_util.tcl

silent [package require cmdline]
silent [package require stooop]

# Flying Spaghetti Machine
stooop::class Fsm {
    set (meatball)       2
    set (channelMode)    ""
    set (channelTarget)  ""
    set (waveTclScript)  ""
    set (topDesignUnit)  tNOODLYBOX
    set (noodlyboxPath)  /$(topDesignUnit)/uBOX
    set (readBodyWidth)  1
    set (writeBodyWidth) 1

    set (clkPeriod)      -1
    set (clkFuture)      0
    set (syncRequired)   0

    set (nextPhase)      "reset"
    set (execAtNextRise) ""
    set (execAtNextFall) ""
    set (nopCount)       0
    set (readBodyCount)  0
    set (writeBodyCount) 0
    set (result)         0
    set (iChannel)       0
    set (oChannel)       0
    set (clkTick)        0
    set (dbg)            0
    set (hdl)            ""

    set (msim)           0
    set (isesim)         0
    set (isesim11)       0

    set (starttime)      0

    # for ISE Simulator
    variable forceArray
    variable forceCountArray
    variable forceLastArray
    variable forced
    set (zero)           0
    set (one)            1

    proc Fsm {this} {
    }
    proc ~Fsm {this} {
    }

    proc checkEnvironment {} {
        set name [info nameofexecutable]
        # Remove the path and the extension.
        regsub "^.*/" $name "" name
        regsub "\.exe$" $name "" name

        putdbg $name

        if {$name eq "vish"} {
            set (msim)   1
        } else {
            set (isesim) 1
            if {[info command ::examine] eq ""} {
                # for ISE11
                set (isesim11) 1
            }
        }

        if [info exists ::hdl] {
            set (hdl) $::hdl
        }
    }

    # Interpret all specified options.
    proc interpretOptions {} {
        set specified ""
        if {$(msim)} {
            global 1
            while {$::argc > 0} {
                append specified " " $1
                shift
            }
            alias fsm "do fsm.tcl $specified"
        } else {
            set specified $::env(FSMARGS)
        }
        putdbg "-- $specified --"

        while {[::cmdline::getopt specified {f.arg c.arg w.arg l.arg q t} optvar valvar]} {
            if {$optvar eq "f"} {
                set (channelMode)   "file"
                set (channelTarget) $valvar
            } elseif {$optvar eq "c"} {
                set (channelMode)   "command"
                set (channelTarget) $valvar
            } elseif {$optvar eq "w"} {
                set (waveTclScript) $valvar
            } elseif {$optvar eq "l"} {
                set (hdl) $valvar
            } elseif {$optvar eq "q"} {
                if $(msim) {
                    onbreak { quit -force }
                }
            } elseif {$optvar eq "t"} {
                if $(msim) {
                    set (starttime) [clock clicks -milliseconds]
                    onbreak { puts stdout \
                        "time : [expr [clock clicks -milliseconds] - $Fsm::(starttime)] ms"}
                }
            }
        }
        if {$(channelMode) eq ""} {
            puts stderr "fsm.tcl needs to specify -f file or -c command." 
            abort
        }
        if {$(isesim) && $(hdl) eq "vhdl"} {
            set (zero) "'0'"
            set (one)  "'1'"
        }
    }

    # Put the debug information.
    proc putdbg {str} {
        if {$(dbg)} { puts stderr $str }
    }

    # The following descriptions absorb the difference
    # of the simulators.
    proc simForce {target value {time "0 ns"}} {
        regsub " ns" $time "" time
        set time [expr $time + $(clkPeriod) * $(clkFuture)]
        append time " ns"

        if {$(msim)} {
            putdbg "force -deposit $target $value $time"
            force -deposit $target $value $time
        } elseif {$(isesim)} {
            variable forceArray
            variable forceCountArray
            variable forceLastArray

            regsub " ns" $time "" time
            if {$(isesim11)} {
                # Convert from ns to ps
                set time [expr $time * 1000]
            } else {
                # Convert from ns to femto
                set time [expr $time * 1000 * 1000]
            }

            if {[array get forceArray $target] eq ""} {
                set forceArray($target) "isim force add $target"
                set forceCountArray($target) 0
                if {[array get forceLastArray $target] ne ""} {
                    append forceArray($target) " $forceLastArray($target) -value"
                    incr forceCountArray($target)
                }
            } else {
                append forceArray($target) " -value"
            }
            append forceArray($target) " $value -time $time"
            incr forceCountArray($target)

            # When specified options are too long,
            # ISE Simulator fails in the interpretation.
            if {$forceCountArray($target) == 12} {
                set (syncRequired) 1
            }

            # The last value is continued.
            set forceLastArray($target) $value
        }
    }
    proc simForceNbox {target value {time "0 ns"}} {
        simForce $(noodlyboxPath)/$target $value $time
    }

    proc simExamine args {
        if {$(clkFuture) != 0} {
            puts stderr "FSM internal error : clkFuture is $(clkFuture)." 
            set (clkTick) 1
            ignoreClkRise
            stop
        }
        if {$(isesim11)} {
            # -dec => -radix dec
            foreach word {default dec bin oct hex unsigned ascii} {
                set i [lsearch $args "-$word"]
                if {$i != -1} {
                    set args [lreplace $args $i $i]
                    lappend args "-radix" $word
                    puts stdout $args
                }
            }
            return [eval show value $args]
        } else {
            return [eval examine $args]
        }
    }

    proc hookRstNegate {} {
        if {$(msim)} {
            eval "when -label atRstNegate { $(noodlyboxPath)/RESET_X == 1 } { Fsm::rstNegate }"
        } elseif {$(isesim)} {
            # Workaround of "incorrect bit"
            variable nboxPath $(noodlyboxPath)
            eval "isim condition add { \$\{Fsm::nboxPath\}/RESET_X == $(one) } {
                Fsm::rstNegate
            } -label atRstNegate"
        }
    }
    proc ignoreRstNegate {} {
        if {$(msim)} {
            nowhen atRstNegate
        } elseif {$(isesim)} {
            isim condition remove -label atRstNegate
        }
    }

    proc hookClkRise {} {
        if {$(msim)} {
            eval "when -label atClkRise { $(noodlyboxPath)/CLK == 1 } { Fsm::clkRise }"
        } elseif {$(isesim)} {
            # Workaround of "incorrect bit"
            variable nboxPath $(noodlyboxPath)
            eval "isim condition add { \$\{Fsm::nboxPath\}/CLK == $(one) } {
                Fsm::clkRise
            } -label atClkRise"
            putdbg "trap clkRise"
        }
    }
    proc ignoreClkRise {} {
        if {$(msim)} {
            nowhen atClkRise
        } elseif {$(isesim)} {
            isim condition remove -label atClkRise
        }
    }

    proc hookWakeUp {time unit} {
        if {$(msim)} {
            eval "when -label atWakeUp { \$now == $time $unit } { Fsm::wakeup }"
        } elseif {$(isesim)} {
            putdbg "trap wakeup begin"
            isim force add $(noodlyboxPath)/i_WAKEUP $(zero) -time "$time $unit"

            if {$(isesim11)} {
                scope $(noodlyboxPath)
                isim condition add { i_WAKEUP } {
                    Fsm::wakeup
                } -label atWakeUp
                scope /
            } else {
                eval "isim condition add $(noodlyboxPath)/i_WAKEUP { Fsm::wakeup } -label atWakeUp"
            }
        }
    }
    proc ignoreWakeUp {} {
        if {$(msim)} {
            nowhen atWakeUp
        } elseif {$(isesim)} {
            isim force remove $(noodlyboxPath)/i_WAKEUP
            isim condition remove -label atWakeUp
        }
    }
    proc str2hex {str} {
        set value "16#"
        append value [string map {\
           { } 20 ! 21 \" 22 # 23 \$ 24 % 25 & 26 ' 27 \( 28 \) 29 * 2A + 2B , 2C - 2D . 2E / 2F \
            0 30 1 31 2 32 3 33 4 34 5 35 6 36 7 37 8 38 9 39 : 3A ; 3B < 3C = 3D > 3E ? 3F \
            @ 40 A 41 B 42 C 43 D 44 E 45 F 46 G 47 H 48 I 49 J 4A K 4B L 4C M 4D N 4E O 4F \
            P 50 Q 51 R 52 S 53 T 54 U 55 V 56 W 57 X 58 Y 59 Z 5A \[ 5B \\ 5C \] 5D ^ 5E _ 5F \
            ` 60 a 61 b 62 c 63 d 64 e 65 f 66 g 67 h 68 i 69 j 6A k 6B l 6C m 6D n 6E o 6F \
            p 70 q 71 r 72 s 73 t 74 u 75 v 76 w 77 x 78 y 79 z 7A} $str]
        return $value
   }

    # Put current phase to i_PHASE signal.
    proc putPhase {str} {
        set phaseStringLen 32
        
        append str [string repeat " " $phaseStringLen]

        set str [string range $str 0 [expr $phaseStringLen - 1]]

        if {$(hdl) eq "verilog"} {
            simForceNbox i_PHASE [str2hex $str]
        } elseif {$(msim)} {
            simForceNbox i_PHASE $str
        }
    }

    proc config {} {
        # Configurable variables.
        #  topDesignUnit
        #  noodlyboxPath
        #  readBodyWidth
        #  writeBodyWidth
        #  dbg

        if {$(channelMode) eq "file"} {
            set (iChannel) [open $(channelTarget) r]
            set (oChannel) stdout
        } elseif {$(channelMode) eq "command"} {
            set (iChannel) [open "|$(channelTarget)" r+]
            set (oChannel) $(iChannel)
            fconfigure $(oChannel) -buffering line
        }
        #set server localhost
        #set sockchannel [socket $server 9900]

        set longline ""
        while {1} {
            # Get next line.
            set rc [gets $(iChannel) line]
            if {$rc == -1} {
                puts stderr "FSMError:endOfConfig is not found."
                abort
            }

            regsub "^ +" $line "" line
            putdbg "config: $line"
            if {$line eq "endOfConfig"} {
                return
            }

            # Ignore any comments
            if [regexp "^\s*#" $line] {
                continue
            }
            append longline $line

            # Replace "\".
            set match ""

            if {[regexp "\[\{\}\] *(.)$" $line dummy match]} {
                set to " "
            } elseif {[regexp "(.)$" $line dummy match]} {
                set to ";"
            }

            if {$match eq "\\"} {
                regsub ".$" $longline $to longline
            } else {
                putdbg $longline
                eval $longline
                set longline ""
            }
        }
    }

    proc loadDesign {designUnit} {
        if {$(msim)} {
            if {$(hdl) eq "vhdl"} {
                vsim -t ns $designUnit
            } else {
                vsim -t ns -L unisims_ver $designUnit
            }
        }
    }
    
    proc examineClkPeriod {} {
        if {$(isesim)} {
            if {$(hdl) eq "vhdl"} {
                set time [simExamine $(noodlyboxPath)/clock_period]
            } else {
                set time [simExamine -dec $(noodlyboxPath)/CLOCK_PERIOD]
            }
        } else {
            set time [simExamine $(noodlyboxPath)/CLOCK_PERIOD]
        }
        regsub "\{" $time "" time
        regsub "\}" $time "" time

        if       {[regsub " ns$" $time "" time]} {
            ;
        } elseif {[regsub " ps$" $time "" time]} {
            set time [expr $time / 1000]
        } elseif {[regsub " fs$" $time "" time]} {
            set time [expr $time / 1000 / 1000]
        }

        putdbg $time
        set (clkPeriod) $time
    }

    # Called when -w foo.do is not specified.
    proc defaultWave {designUnit} {
        if {$(msim)} {
            set full $(clkPeriod)
            set half [expr $full / 2]
            append full "ns"
            append half "ns"
            configure wave -gridoffset $half -gridperiod $full -timeline 1
            add wave -hex -r /*
        } elseif {$(isesim11)} {
            isim set radix hex
            wave add -wcfg fsm.wcfg -r /
        } elseif {$(isesim)} {
            ntrace select -m /$designUnit -l all
            ntrace start
        }
    }

    proc dataLatch {} {
        set (result) [simExamine -hex $(noodlyboxPath)/D]
        putdbg "latch: $(result)"
    }

    proc reset {} {
        putPhase "reset"
        hookRstNegate
    }

    proc rstNegate {} {
        putdbg "rstNegate"
        ignoreRstNegate
        set (nextPhase) "fetch"
        hookClkRise
    }

    # Sleep
    proc opSleep {time unit} {
        set (nextPhase) "fetch"
        set (clkTick)   1

        changeAtOpNop
        putPhase "opSleep $time $unit"

        if {$unit eq "ms"} {
            set time [expr $time * 1000000]
        } elseif {$unit eq "us"} {
            set time [expr $time * 1000]
        }
        set clocks [expr $time / $(clkPeriod) - 1]

        incr (clkFuture) $clocks
        set (syncRequired) 1
    }
    proc wakeup {} {
        ignoreWakeUp
        hookClkRise
    }

    proc changeAtOpNop {} {
        simForceNbox i_HIZ  1
        simForceNbox i_CS_X 1
        simForceNbox i_OE_X 1
        simForceNbox i_WE_X 1
    }

    # No operation
    proc opNop {count} {
        set (nextPhase) "opNop_tail"
        set (nopCount)  $count
        set (clkTick)   1

        # Update the output directional signals.
        changeAtOpNop

        incr (nopCount) -1
        if {$(nopCount) == 0} { set (nextPhase) "fetch" }
    }
    proc opNop_tail {} {
        set (clkTick) 1

        incr (nopCount) -1
        if {$(nopCount) == 0} { set (nextPhase) "fetch" }
    }

    proc changeAtOpRead {addr} {
        simForceNbox i_A    $addr
        simForceNbox i_HIZ  1
        simForceNbox i_CS_X 0
        simForceNbox i_OE_X 1
        simForceNbox i_WE_X 1
    }
    proc changeAtOpRead_body {} {
        simForceNbox i_OE_X 0
        putdbg "changeAtOpRead_body"
        set (syncRequired) 1
    }
    proc changeAtOpRead_tail {} {
        simForceNbox i_OE_X 1
        dataLatch
    }

    # Read operation
    proc opRead {addr} {
        set (nextPhase)     "opRead_body"
        set (clkTick)       1
        set (readBodyCount) $(readBodyWidth)

        # Update the output directional signals.
        changeAtOpRead $addr

        if {$(readBodyCount) == 0} {
            set (nextPhase) "opRead_tail"
        }
    }
    proc opRead_body {} {
        set (clkTick) 1

        changeAtOpRead_body

        incr (readBodyCount) -1
        if {$(readBodyCount) == 0} {
            set (nextPhase) "opRead_tail"
        }
    }
    proc opRead_tail {} {
        set (nextPhase) "fetch"
        set (clkTick) 1

        # Update the output directional signals.
        changeAtOpRead_tail
    }

    proc putReadResult {} {
        puts $(oChannel) $(result)
    }

    proc changeAtOpWrite {addr data} {
        simForceNbox i_A    $addr
        simForceNbox i_D    $data
        simForceNbox i_HIZ  0
        simForceNbox i_CS_X 0
        simForceNbox i_OE_X 1
        simForceNbox i_WE_X 1
    }
    proc changeAtOpWrite_body {} {
        simForceNbox i_WE_X 0
    }
    proc changeAtOpWrite_tail {} {
        simForceNbox i_WE_X 1
    }

    # Write operation
    proc opWrite {addr data} {
        set (nextPhase)      "opWrite_body"
        set (clkTick)        1
        set (writeBodyCount) $(writeBodyWidth)

        # Update the output directional signals.
        changeAtOpWrite $addr $data

        if {$(writeBodyCount) == 0} { set (nextPhase) "opWrite_tail" }
    }
    proc opWrite_body {} {
        set (clkTick) 1

        # If it is 1st, ...
        if {$(writeBodyCount) == $(writeBodyWidth)} {
            # Update the output directional signals.
            changeAtOpWrite_body
        }

        incr (writeBodyCount) -1
        if {$(writeBodyCount) == 0} { set (nextPhase) "opWrite_tail" }

    }
    proc opWrite_tail {} {
        set (nextPhase) "fetch"
        set (clkTick) 1

        # Update the output directional signals.
        changeAtOpWrite_tail
    }

    # End of simulation.
    proc endOfSimulation {} {
        set (clkTick) 1
        set (syncRequired) 1

        putdbg "endOfSimulation"
        #close $iChannel
        if {$(isesim11)} {
            quit -f
        } else {
            stop
        }
    }

    # Fetch new line from the channel, and execute it.
    proc fetch {} {
        set rc [gets $(iChannel) line]
        if {$rc == -1} {
            # End of file is end of sim.
            endOfSimulation
            putPhase "endOfSimulation"
        } else {
            putdbg "fetch: $line"

            # Ignore any comments
            if [regexp "^\s*#" $line] {
                return
            }

            eval $line
            putPhase $line
        }
    }

    # This function is called at least rising edge of the clock.
    proc clkRise {} {
        ignoreClkRise

        set (clkFuture) 0
        while {1} {

            if {$(nextPhase) ne "fetch"} {
                putdbg $(nextPhase)
                putPhase $(nextPhase)
            }

            eval $(execAtNextRise)
            set (execAtNextRise) ""

            # $(syncRequired) == 0 means that examining is not required.
            set (syncRequired) 0

            # $clkTick == 0 means that time does not pass.
            set (clkTick) 0
            while {$(clkTick) == 0} {
                # Call procedure that was named $nextPhase.
                eval $(nextPhase)
            }

            if {$(execAtNextFall) ne ""} {
                set tmp $(clkFuture)
                set (clkFuture) [expr $(clkFuture) + 0.5]

                eval $(execAtNextFall)
                set (execAtNextFall) ""

                set (clkFuture) $tmp
            }

            incr (clkFuture)

            if {$(syncRequired) == 1} {
                if {$(msim)} {
                    putdbg "syncRequired clkFuture:$(clkFuture) @ $::now"
                } elseif {$(isesim)} {
                    putdbg "syncRequired clkFuture:$(clkFuture)"

                    variable forceArray
                    variable forced
                    set id [array startsearch forceArray]

                    while {1} {
                        set target [array nextelement forceArray $id]
                        if {$target eq ""} {
                            break
                        }
                        putdbg $target
                        putdbg $forceArray($target)

                        if {[array get forced $target] eq ""} {
                            set forced($target) 1
                        } else {
                            isim force remove $target
                        }

                        if {[regsub -all "16#" $forceArray($target) "" forceArray($target)] != 0} {
                            isim set radix hex
                            eval "$forceArray($target)"
                            isim set radix default
                        } else {
                            eval "$forceArray($target)"
                        }
                    }
                    unset forceArray
                }
                if {$(clkFuture) == 1} {
                    hookClkRise
                } else {
                    hookWakeUp [expr $(clkPeriod) * $(clkFuture) - 1] "ns"
                }
                break
            }
        }
    }
}

# main ----
if [catch {
    Fsm::checkEnvironment

    Fsm::interpretOptions

    set compileSuccess 1

    if {$Fsm::(msim)} {
        quit -sim
        set compileSuccess [checkResult "project compileoutofdate"]
    }

    if {$compileSuccess} {

        Fsm::config
        Fsm::loadDesign $Fsm::(topDesignUnit)
        Fsm::examineClkPeriod

        if {$Fsm::(waveTclScript) ne ""} {
            source $Fsm::(waveTclScript)
        } else {
            Fsm::defaultWave $Fsm::(topDesignUnit)
        }

        Fsm::reset

        set StdArithNoWarnings 1
        run 0 ns
        set StdArithNoWarnings 0
        if {$Fsm::(isesim11)} {
            run all
        } else {
            run -all
        }
    }

    if {$Fsm::(isesim) && !$Fsm::(isesim11)} {
        quit
    }
} errorKind] {
    # Exception trap
    puts stderr $errorInfo
    error $errorKind
}
