#!/bin/sh
# the next line restarts using wish \
exec wish "$0" "$@"

# This software is a part of NOODLYBOX.
# This software is distributed under the terms of the new BSD License.
# Copyright (c) 2011 molelord
# All rights reserved.

# Timing Chart Viewer

# Usage : tcv.tcl file

proc posy {level} {
    set y 0
    switch $level {
        "0" { set y 20 }
        "1" { set y  0 }
        "z" { set y 10 }
        "x" { set y  0 }
    }
    return $y
}

proc putclock {origin_x origin_y} {
    set x(0) $origin_x
    set x(1) $x(0)
    set x(2) [expr $x(0) + 25]
    set x(3) $x(2)
    set x(4) [expr $x(0) + 50]

    set y(0) [expr $origin_y + [posy "0"]]
    set y(1) [expr $origin_y + [posy "1"] + 3]
    set y(2) $y(1)
    set y(3) $y(0)
    set y(4) $y(0)

    for {set i 0} {$i < 4} {incr i} {
        .c create line $x($i) $y($i) $x([expr $i + 1]) $y([expr $i + 1]) \
            -fill "black" -width 2
    }
}

# putwaveの補助関数
proc putwave_sub {level x0 y0 x1} {
    set add 0
    set color "black"
    set y0 [expr $y0 + [posy $level]]
    set y1 $y0

    if {$level == "z"} {
        set color "brown"
    }

    if {$level == "x"} {
        .c create rectangle $x0 $y0 $x1 [expr $y1 + 20] \
            -fill "gray" -width 0
        .c create line $x0 [expr $y0 + 20] $x1 [expr $y1 + 20] \
            -fill $color -width 2
    } \
    elseif {[regexp {[A-Z]} $level]} {
        .c create line $x0 [expr $y0 + 20] $x1 [expr $y1 + 20] \
            -fill $color -width 2
    }
    .c create line $x0 $y0 $x1 $y1 \
        -fill $color -width 2
}


# prev : 直前の信号 "/" "0" "1" "z" "x"
# tie  : 中間       "|" " " "v" "^"
# now  : 今の信号   "/" "0" "1" "z" "x"
proc putwave {prev tie now logical_x logical_y} {
    set phy_x  [expr $logical_x * 50]
    set phy_y  [expr $logical_y * 30]

    if {$prev == "/"} {
        putclock $phy_x $phy_y
        return
    }

    switch $prev {
        "0" -
        "1" -
        "z" -
        "x" {}
        default {
            set msg "Invalid charactor : $prev"
            error $msg
        }
    }
    switch $now {
        "0" -
        "1" -
        "z" -
        "x" {}
        default {
            set msg "Invalid charactor : $now"
            error $msg
        }
    }
    switch $tie {
        "|" -
        " " -
        "v" -
        "^" {}
        default {
            set msg "Invalid charactor : $tie"
            error $msg
        }
    }

    set x(0) [expr $phy_x + 0]
    set x(1) [expr $x(0) + 10]
    set x(2) [expr $x(0) + 16]
    set x(3) [expr $x(0) + 50]

    # 頭
    putwave_sub $prev $x(0) $phy_y $x(1)

    # 尾っぽ
    putwave_sub $now  $x(2) $phy_y $x(3)

    # 中間
    if {$tie == "v" || $tie == "^"} {
        set reverse "1"
        if {$prev == "1"} {
            set reverse "0"
        }
        .c create polygon $x(1)            [expr $phy_y + [posy $prev]] \
                          [expr $x(1) + 3] [expr $phy_y + [posy $reverse]] \
                          $x(2)            [expr $phy_y + [posy $now]] \
            -fill "gray" -width 2 -outline "black"
    } \
    elseif {$prev == $now} {
        # 同一の場合
        putwave_sub $prev $x(1) $phy_y $x(2)
    } \
    elseif {($prev == "0" && $now == "1") || ($prev == "1" && $now == "0")} {
        # 両方とも0または1で、かつ、異なる場合
        .c create polygon $x(1)            [expr $phy_y + [posy $prev]] \
                          $x(2)            [expr $phy_y + [posy $prev]] \
                          [expr $x(2) + 6] [expr $phy_y + [posy $now]] \
                          $x(2)            [expr $phy_y + [posy $now]] \
            -fill "gray" -width 2 -outline "black"
        if {$prev == "1" && $now == "0"} {
            # 1→0の変化のときはpolygonだけでは角が欠けるので、lineで補う
            .c create line $x(1)            [expr $phy_y + [posy $prev]] \
                           [expr $x(1) + 1] [expr $phy_y + [posy $prev]] \
                -fill "black" -width 2
        }
    } \
    elseif {($prev == "z" || $now == "z") && $prev != "x" && $now != "x"} {
        # はじめか終わりのどちらかがzで、かつ、xを含まない場合
        .c create line $x(1) [expr $phy_y + [posy $prev]] \
                       $x(2) [expr $phy_y + [posy $now]] \
            -fill "brown" -width 2
    } \
    elseif {$prev == "x"} {
        # xで始まり、0か1かzで終わる場合
        .c create polygon $x(1) [expr $phy_y +  0] \
                          $x(1) [expr $phy_y + 20] \
                          $x(2) [expr $phy_y + [posy $now]] \
            -fill "gray"
        .c create line $x(1) [expr $phy_y +  0] \
                       $x(2) [expr $phy_y + [posy $now]] \
            -fill "black" -width 2
        .c create line $x(1) [expr $phy_y + 20] \
                       $x(2) [expr $phy_y + [posy $now]] \
            -fill "black" -width 2
    } \
    elseif {$now == "x"} {
        # 0か1かzで始まり、xで終わる場合
        .c create polygon $x(1) [expr $phy_y + [posy $prev]] \
                          $x(2) [expr $phy_y + 0] \
                          $x(2) [expr $phy_y + 20] \
            -fill "gray"
        .c create line $x(1) [expr $phy_y + [posy $prev]] \
                       $x(2) [expr $phy_y + 0] \
            -fill "black" -width 2
        .c create line $x(1) [expr $phy_y + [posy $prev]] \
                       $x(2) [expr $phy_y + 20] \
            -fill "black" -width 2
    }
}

# prev : 直前の信号 [A-Z] "x"
# tie  : 中間       "|" " "
# now  : 今の信号   [A-Z] "x"
proc putbus {prev tie now logical_x logical_y comment} {
    set phy_x  [expr $logical_x * 50]
    set phy_y  [expr $logical_y * 30]

    switch -glob $prev {
        [A-Z] -
        "x" {}
        default {
            set msg "Invalid charactor : $prev"
            error $msg
        }
    }
    switch -glob $now {
        [A-Z] -
        "x" {}
        default {
            set msg "Invalid charactor : $now"
            error $msg
        }
    }
    switch $tie {
        "|" -
        " " {}
        default {
            set msg "Invalid charactor : $tie"
            error $msg
        }
    }

    set x(0) [expr $phy_x + 0]
    set x(1) [expr $x(0) + 10]
    set x(2) [expr $x(0) + 16]
    set x(3) [expr $x(0) + 50]

    # 頭
    putwave_sub $prev $x(0) $phy_y $x(1)

    # 尾っぽ
    putwave_sub $now  $x(2) $phy_y $x(3)

    # 中間
    if {$prev == $now} {
        # 同一の場合
        putwave_sub $prev $x(1) $phy_y $x(2)
    } \
    else {
        if {$prev == "x" || $now == "x"} {
            # どちらかがxの場合
            if {$prev == "x"} {
                .c create polygon $x(1)            [expr $phy_y +  0] \
                                  [expr $x(1) + 3] [expr $phy_y + 10] \
                                  $x(1)            [expr $phy_y + 20] \
                    -fill "gray"
            } \
            else {
                .c create polygon $x(2)            [expr $phy_y +  0] \
                                  [expr $x(1) + 3] [expr $phy_y + 10] \
                                  $x(2)            [expr $phy_y + 20] \
                    -fill "gray"
            }
            .c create line $x(1) [expr $phy_y +  0] \
                           $x(2) [expr $phy_y + 20] \
                -fill "black" -width 2
            .c create line $x(1) [expr $phy_y + 20] \
                           $x(2) [expr $phy_y +  0] \
                -fill "black" -width 2
        } \
        else {
            # 両方[A-Z]だけれど異なる場合

            # 余計な線を白で消す
            .c create line [expr $x(1) - 3] [expr $phy_y +  0] \
                           [expr $x(2) + 3] [expr $phy_y +  0] \
                -fill "white" -width 2
            .c create line [expr $x(1) - 3] [expr $phy_y + 20] \
                           [expr $x(2) + 3] [expr $phy_y + 20] \
                -fill "white" -width 2

            # 前の三角
            .c create line [expr $x(1) - 3] [expr $phy_y +  0] \
                           [expr $x(1) + 0] [expr $phy_y + 10] \
                -fill "black" -width 2
            .c create line [expr $x(1) + 0] [expr $phy_y + 10] \
                           [expr $x(1) - 3] [expr $phy_y + 20] \
                -fill "black" -width 2

            # 真ん中のひし形
            .c create polygon [expr $x(1) + 3] [expr $phy_y +  0] \
                              [expr $x(1) + 0] [expr $phy_y + 10] \
                              [expr $x(1) + 3] [expr $phy_y + 20] \
                              [expr $x(1) + 6] [expr $phy_y + 10] \
                -fill "gray" -width 2 -outline "black"

            # 後の三角
            .c create line [expr $x(2) + 3] [expr $phy_y +  0] \
                           [expr $x(2) + 0] [expr $phy_y + 10] \
                -fill "black" -width 2
            .c create line [expr $x(2) + 0] [expr $phy_y + 10] \
                           [expr $x(2) + 3] [expr $phy_y + 20] \
                -fill "black" -width 2
        }

        if {$now != "x"} {
            set tx  [expr $logical_x * 50 + 23]
            set ty  [expr $logical_y * 30 + 10]
            set index [string first $now "ABCDEFGHIJKLMNOPQRSTUVWXYZ"]
            .c create text $tx $ty -text [lindex $comment $index] -anchor w
        }
    }
}
proc puttext {str logical_x logical_y} {
    set phy_x  [expr $logical_x * 50 +  5]
    set phy_y  [expr $logical_y * 30 + 10]
    .c create text $phy_x $phy_y -text $str -anchor w
}

proc putgrid {logical_x logical_y count} {
    set phy_x  [expr $logical_x * 50]
    set phy_y  [expr $logical_y * 30]
    for {set i 0} {$i < $count} {incr i} {
        .c create line [expr $phy_x + $i * 50] $phy_y \
                       [expr $phy_x + $i * 50] [expr $phy_y + 480] \
            -fill "gray" -width 1
    }
}

proc putwaveline {name wave y comment} {
    puttext $name 0 $y
    for {set i 0} {1} {incr i} {
        set prev [string index $wave [expr $i * 2 + 0]]
        set tie  [string index $wave [expr $i * 2 + 1]]
        set now  [string index $wave [expr $i * 2 + 2]]
        if {$now == ""} {
            break
        }
        if {$comment != ""} {
            putbus  $prev $tie $now [expr $i + 2] $y $comment
        } else {
            putwave $prev $tie $now [expr $i + 2] $y
        }
    }
}

proc reload {filename} {
    set iChannel [open $filename]
    fconfigure $iChannel -encoding utf-8 -translation {auto lf}

    #set oChannel [open "debugout.txt" "w"]
    #fconfigure $oChannel -encoding utf-8 -translation {auto lf}

    set prevline ""
    set found 0
    set all   {}
    while {[gets $iChannel line] != -1} {
        if {[regexp "TimingChartEnd" $line]} {
            break
        } \
        elseif {[regexp "TimingChart" $line]} {
            set found 1
            continue
        }

        if {$found == 0} {
            continue
        }

        # 1行コメントのための文字列と、その前後の空白を削除
        regsub {^\s*--\s*} $line "" line
        regsub {^\s*//\s*} $line "" line
        regsub {^\s*#\s*}  $line "" line

        # 行末が \ だった場合は次の行と合体させる
        if {[regsub {\\$} $line "" line]} {
            set prevline $line
            continue
        }
        set line "$prevline$line"
        set prevline ""

        #puts $oChannel $line

        # パース開始
        if {[regexp {(\S+)\s+(\S[^:]*)(.*)} $line ignore name wave comment]} {

            # 末尾の余分な空白を削る
            regsub {\s$} $wave "" wave

            # 先頭の": "を削る
            regsub {^:\s} $comment "" comment

            lappend all $name $wave $comment
        }
    }
    close $iChannel

    set namelen_max 0
    set wavelen_max 0
    set number_of_waves 0
    foreach {name wave comment} $all {
        #puts $oChannel "name:$name wave:\"$wave\" comment:\"$comment\""
        set len [string length $name]
        if {$len > $namelen_max} {
            set namelen_max $len
        }
        set len [string length $wave]
        if {$len > $wavelen_max} {
            set wavelen_max $len
        }
        incr number_of_waves
    }

    # 波形の長さに応じてキャンバスの幅を変える
    set width [expr 50 + ($wavelen_max + 1) / 2 * 50 + 10]

    # 波形の数に応じてキャンバスの高さを変える
    set height [expr $number_of_waves * 30]

    pack [canvas .c -width $width -height $height -bg "white"]

    button .b(0) -text Reload -command {
        for {set i 0} {$i < 3} {incr i} {
            destroy .b($i)
        }
        destroy .c
        reload $filename
    } -takefocus 1 -anchor w
    pack .b(0) -side left

    button .b(1) -text PNG -command {
        set pngname [regsub {\.[^.]+$} $filename ".png"]
        set eps [.c postscript]

        # フォントがおかしいのと小さすぎるのを回避する
        #regsub -all {MsSansSerif} $eps "URWGothicL-Book" eps
        regsub -all {MsSansSerif} $eps "CenturySchL-Roma" eps
        regsub -all {findfont 8}  $eps "findfont 10"     eps

        # Cygwinのghostscriptにパイプでつないで、画像ファイルを作る
        set oCh [open "|gs -dQUIET -dSAFER -dBATCH -dNOPAUSE -sDEVICE=pngalpha -sOutputFile=$pngname -r95 -" "w"]
        puts -nonewline $oCh $eps
        close $oCh
    } -takefocus 1 -anchor w
    pack .b(1) -side left

    button .b(2) -text Exit -command exit -takefocus 1 -anchor w
    pack .b(2) -side left

    # 起動時に、ボタンにフォーカスが当たった状態にするための記述
    raise .
    focus .b(0)

    putgrid 2 0 [expr ($wavelen_max + 1) / 2]

    set y 0
    foreach {name wave comment} $all {
        #puts $oChannel "name:$name wave:\"$wave\" comment:\"$comment\""
        putwaveline $name $wave $y $comment
        incr y
    }
}

# main ------------------------------------------------------------------------

if {$argc == 0} {
    wm title . "Usage - tcv"
    pack [label .l -text "Usage : tcv.tcl <filename>"] -side top
    button .b -text Exit -command exit -takefocus 1 -anchor w
    pack .b

    # 起動時に、ボタンにフォーカスが当たった状態にするための記述
    raise .
    focus .b
} \
else {
    set filename [lindex $argv 0]
    wm title . "$filename - tcv"
    reload $filename
}

# テスト ----------------------------------------------------------
# putwave 0 " " 0 0 8
# putwave 0 " " 1 2 8
# putwave 0 " " z 4 8
# putwave 0 " " x 6 8
 
# putwave 1 " " 0  9 8
# putwave 1 " " 1 11 8
# putwave 1 " " z 13 8
# putwave 1 " " x 15 8
 
# putwave z " " 0 0 9
# putwave z " " 1 2 9
# putwave z " " z 4 9
# putwave z " " x 6 9
 
# putwave x " " 0  9 9
# putwave x " " 1 11 9
# putwave x " " z 13 9
# putwave x " " x 15 9
