// $Id: functions.h 26 2004-07-03 06:22:39Z takekawa $
// Copyright (C) 2004  Takashi Takekawa
// This file is part of the Nsim Library.

// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.

// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// long with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
// 02111-1307 USA.

#ifndef NSIM_FUNCTION_H
#define NSIM_FUNCTION_H

#include <nsim/config.h>
#include <cmath> // std::exp, std::tanh
#include <nsim/pow.h>

namespace nsim
{

struct exp
{
    static double apply(double x) { return std::exp(x); }
    static double deriv(double x) { return std::exp(x); }
};

struct lin
{
#if HAVE_LOG2
    static double log2(double x) { return ::log2(x); }
#else
    static double log2(double x) { return M_LOG2E*std::log(x); }
#endif
#if HAVE_EXP2
    static double exp2(double x) { return ::exp2(x); }
#else
    static double exp2(double x) { return std::exp(M_LN2*x); }
#endif
    static double apply(double x) { return log2(exp2(x)+1); }
    static double deriv(double x) { return .5*(std::tanh(.5*M_LN2*x)+1); }
};

struct sig
{
    static double apply(double x) { return 0.5*(std::tanh(0.5*x)+1.0); }
    static double deriv(double x)
    {
        double const tanh(std::tanh(0.5*x));
        return 0.25*(1.0-tanh*tanh);
    }
};

struct rcosh
{
    static double apply(double x) { return 1.0/std::cosh(x); }
    static double deriv(double x) { return -std::tanh(x)/std::cosh(x); }
};

struct constant
{
    double val;
    constant(double val = 0.0) : val(val) {}
    double operator()(double x) const { return val; }
    double dx(double x) const { return 0.0; }
};

struct ssig
{
    double g;
    double k;
    double pow;
    ssig(double g, double k, double pow = 1.0) : g(g), k(k), pow(pow) {}
    double operator()(double x) const
    {
        double x_(std::pow(x /= k, pow));
        return g*x_/(x_ + 1.0);
    }
    double dx(double x) const
    {
        double x_(std::pow(x /= k, pow-1.0));
        return g*pow*x_/(k*nsim::pow<2>(x_*x + 1.0));
    }
};

template <typename T>
struct scale
{
    double x0;
    double xs;
    double ys;
    scale(double x0 = 0.0, double xs = 1.0, double ys = 1.0)
        : x0(x0), xs(xs), ys(ys) {}
    double operator()(double x) const
    {
        return ys * T::apply((x - x0)/xs);
    }
    double dx(double x) const
    {
        return (ys / xs) * T::deriv((x - x0)/xs);
    }
};

}

#endif

