/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import java.io.IOException;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.TreeMap;
import java.util.TreeSet;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.protocol.BlockListAsLongs;
import org.apache.hadoop.hdfs.protocol.DatanodeInfo;
import org.apache.hadoop.hdfs.protocol.LocatedBlock;
import org.apache.hadoop.hdfs.server.common.HdfsConstants;
import org.apache.hadoop.hdfs.server.namenode.BlockInfo;
import org.apache.hadoop.hdfs.server.namenode.BlockInfoUnderConstruction;
import org.apache.hadoop.hdfs.server.namenode.BlockPlacementPolicy;
import org.apache.hadoop.hdfs.server.namenode.BlocksMap;
import org.apache.hadoop.hdfs.server.namenode.CorruptReplicasMap;
import org.apache.hadoop.hdfs.server.namenode.DatanodeDescriptor;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.INode;
import org.apache.hadoop.hdfs.server.namenode.INodeFile;
import org.apache.hadoop.hdfs.server.namenode.INodeFileUnderConstruction;
import org.apache.hadoop.hdfs.server.namenode.NameNode;
import org.apache.hadoop.hdfs.server.namenode.PendingReplicationBlocks;
import org.apache.hadoop.hdfs.server.namenode.UnderReplicatedBlocks;

@InterfaceAudience.Private
public class BlockManager {
    public static final int DEFAULT_INITIAL_MAP_CAPACITY = 16;
    public static final float DEFAULT_MAP_LOAD_FACTOR = 0.75f;
    public static final int DEFAULT_MAX_CORRUPT_FILES_RETURNED = 500;
    private final FSNamesystem namesystem;
    volatile long pendingReplicationBlocksCount = 0L;
    volatile long corruptReplicaBlocksCount = 0L;
    volatile long underReplicatedBlocksCount = 0L;
    volatile long scheduledReplicationBlocksCount = 0L;
    volatile long excessBlocksCount = 0L;
    volatile long pendingDeletionBlocksCount = 0L;
    final BlocksMap blocksMap;
    CorruptReplicasMap corruptReplicas = new CorruptReplicasMap();
    Map<String, Collection<Block>> recentInvalidateSets = new TreeMap<String, Collection<Block>>();
    Map<String, Collection<Block>> excessReplicateMap = new TreeMap<String, Collection<Block>>();
    UnderReplicatedBlocks neededReplications = new UnderReplicatedBlocks();
    private PendingReplicationBlocks pendingReplications;
    int maxReplication;
    int maxReplicationStreams;
    int minReplication;
    int defaultReplication;
    int maxCorruptFilesReturned;
    boolean shouldCheckForEnoughRacks = true;
    private int replIndex = 0;
    private long missingBlocksInCurIter = 0L;
    private long missingBlocksInPrevIter = 0L;
    Random r = new Random();
    BlockPlacementPolicy replicator;

    BlockManager(FSNamesystem fsn, Configuration conf) throws IOException {
        this(fsn, conf, 16);
    }

    BlockManager(FSNamesystem fsn, Configuration conf, int capacity) throws IOException {
        this.namesystem = fsn;
        this.pendingReplications = new PendingReplicationBlocks((long)conf.getInt("dfs.namenode.replication.pending.timeout-sec", -1) * 1000L);
        this.setConfigurationParameters(conf);
        this.blocksMap = new BlocksMap(capacity, 0.75f);
    }

    void setConfigurationParameters(Configuration conf) throws IOException {
        this.replicator = BlockPlacementPolicy.getInstance(conf, this.namesystem, this.namesystem.clusterMap);
        this.maxCorruptFilesReturned = conf.getInt("dfs.corruptfilesreturned.max", 500);
        this.defaultReplication = conf.getInt("dfs.replication", 3);
        this.maxReplication = conf.getInt("dfs.replication.max", 512);
        this.minReplication = conf.getInt("dfs.namenode.replication.min", 1);
        if (this.minReplication <= 0) {
            throw new IOException("Unexpected configuration parameters: dfs.namenode.replication.min = " + this.minReplication + " must be greater than 0");
        }
        if (this.maxReplication >= Short.MAX_VALUE) {
            throw new IOException("Unexpected configuration parameters: dfs.replication.max = " + this.maxReplication + " must be less than " + Short.MAX_VALUE);
        }
        if (this.maxReplication < this.minReplication) {
            throw new IOException("Unexpected configuration parameters: dfs.namenode.replication.min = " + this.minReplication + " must be less than dfs.replication.max = " + this.maxReplication);
        }
        this.maxReplicationStreams = conf.getInt("dfs.namenode.replication.max-streams", 2);
        this.shouldCheckForEnoughRacks = conf.get("net.topology.script.file.name") != null;
        FSNamesystem.LOG.info((Object)("defaultReplication = " + this.defaultReplication));
        FSNamesystem.LOG.info((Object)("maxReplication = " + this.maxReplication));
        FSNamesystem.LOG.info((Object)("minReplication = " + this.minReplication));
        FSNamesystem.LOG.info((Object)("maxReplicationStreams = " + this.maxReplicationStreams));
        FSNamesystem.LOG.info((Object)("shouldCheckForEnoughRacks = " + this.shouldCheckForEnoughRacks));
    }

    void activate() {
        this.pendingReplications.start();
    }

    void close() {
        if (this.pendingReplications != null) {
            this.pendingReplications.stop();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void metaSave(PrintWriter out) {
        UnderReplicatedBlocks underReplicatedBlocks = this.neededReplications;
        synchronized (underReplicatedBlocks) {
            out.println("Metasave: Blocks waiting for replication: " + this.neededReplications.size());
            for (Block block : this.neededReplications) {
                ArrayList<DatanodeDescriptor> containingNodes = new ArrayList<DatanodeDescriptor>();
                FSNamesystem.NumberReplicas numReplicas = new FSNamesystem.NumberReplicas();
                this.chooseSourceDatanode(block, containingNodes, numReplicas);
                int usableReplicas = numReplicas.liveReplicas() + numReplicas.decommissionedReplicas();
                if (block instanceof BlockInfo) {
                    String fileName = ((BlockInfo)block).getINode().getFullPathName();
                    out.print(fileName + ": ");
                }
                out.print(block + (usableReplicas > 0 ? "" : " MISSING") + " (replicas:" + " l: " + numReplicas.liveReplicas() + " d: " + numReplicas.decommissionedReplicas() + " c: " + numReplicas.corruptReplicas() + " e: " + numReplicas.excessReplicas() + ") ");
                Collection<DatanodeDescriptor> corruptNodes = this.corruptReplicas.getNodes(block);
                Iterator<DatanodeDescriptor> jt = this.blocksMap.nodeIterator(block);
                while (jt.hasNext()) {
                    DatanodeDescriptor node = jt.next();
                    String state = "";
                    if (corruptNodes != null && corruptNodes.contains(node)) {
                        state = "(corrupt)";
                    } else if (node.isDecommissioned() || node.isDecommissionInProgress()) {
                        state = "(decommissioned)";
                    }
                    out.print(" " + node + state + " : ");
                }
                out.println("");
            }
        }
        this.pendingReplications.metaSave(out);
        this.dumpRecentInvalidateSets(out);
    }

    boolean checkMinReplication(Block block) {
        return this.countNodes(block).liveReplicas() >= this.minReplication;
    }

    private void commitBlock(INodeFileUnderConstruction fileINode, BlockInfoUnderConstruction block, Block commitBlock) throws IOException {
        if (block.getBlockUCState() == HdfsConstants.BlockUCState.COMMITTED) {
            return;
        }
        assert (block.getNumBytes() <= commitBlock.getNumBytes()) : "commitBlock length is less than the stored one " + commitBlock.getNumBytes() + " vs. " + block.getNumBytes();
        block.commitBlock(commitBlock);
        long diff = fileINode.getPreferredBlockSize() - commitBlock.getNumBytes();
        if (diff > 0L) {
            try {
                String path = this.namesystem.leaseManager.findPath(fileINode);
                this.namesystem.dir.updateSpaceConsumed(path, 0L, -diff * (long)fileINode.getReplication());
            }
            catch (IOException e) {
                FSNamesystem.LOG.warn((Object)("Unexpected exception while updating disk space : " + e.getMessage()));
            }
        }
    }

    void commitOrCompleteLastBlock(INodeFileUnderConstruction fileINode, Block commitBlock) throws IOException {
        if (commitBlock == null) {
            return;
        }
        Object lastBlock = fileINode.getLastBlock();
        if (lastBlock == null) {
            return;
        }
        if (((BlockInfo)lastBlock).isComplete()) {
            return;
        }
        this.commitBlock(fileINode, (BlockInfoUnderConstruction)lastBlock, commitBlock);
        if (this.countNodes((Block)lastBlock).liveReplicas() >= this.minReplication) {
            this.completeBlock((INodeFile)fileINode, fileINode.numBlocks() - 1);
        }
    }

    BlockInfo completeBlock(INodeFile fileINode, int blkIndex) throws IOException {
        if (blkIndex < 0) {
            return null;
        }
        BlockInfo curBlock = fileINode.getBlocks()[blkIndex];
        if (curBlock.isComplete()) {
            return curBlock;
        }
        BlockInfoUnderConstruction ucBlock = (BlockInfoUnderConstruction)curBlock;
        if (ucBlock.numNodes() < this.minReplication) {
            throw new IOException("Cannot complete block: block does not satisfy minimal replication requirement.");
        }
        BlockInfo completeBlock = ucBlock.convertToCompleteBlock();
        fileINode.setBlock(blkIndex, completeBlock);
        return this.blocksMap.replaceBlock(completeBlock);
    }

    BlockInfo completeBlock(INodeFile fileINode, BlockInfo block) throws IOException {
        BlockInfo[] fileBlocks = fileINode.getBlocks();
        for (int idx = 0; idx < fileBlocks.length; ++idx) {
            if (fileBlocks[idx] != block) continue;
            return this.completeBlock(fileINode, idx);
        }
        return block;
    }

    void convertLastBlockToUnderConstruction(INodeFileUnderConstruction fileINode, DatanodeDescriptor[] targets) throws IOException {
        Object oldBlock = fileINode.getLastBlock();
        if (oldBlock == null) {
            return;
        }
        BlockInfoUnderConstruction ucBlock = fileINode.setLastBlock((BlockInfo)oldBlock, targets);
        this.blocksMap.replaceBlock(ucBlock);
    }

    ArrayList<String> getValidLocations(Block block) {
        ArrayList<String> machineSet = new ArrayList<String>(this.blocksMap.numNodes(block));
        Iterator<DatanodeDescriptor> it = this.blocksMap.nodeIterator(block);
        while (it.hasNext()) {
            String storageID = it.next().getStorageID();
            Collection<Block> blocks = this.recentInvalidateSets.get(storageID);
            if (blocks != null && blocks.contains(block)) continue;
            machineSet.add(storageID);
        }
        return machineSet;
    }

    List<LocatedBlock> getBlockLocations(BlockInfo[] blocks, long offset, long length, int nrBlocksToReturn) throws IOException {
        int curBlk = 0;
        long curPos = 0L;
        long blkSize = 0L;
        int nrBlocks = blocks[0].getNumBytes() == 0L ? 0 : blocks.length;
        for (curBlk = 0; curBlk < nrBlocks; ++curBlk) {
            blkSize = blocks[curBlk].getNumBytes();
            assert (blkSize > 0L) : "Block of size 0";
            if (curPos + blkSize > offset) break;
            curPos += blkSize;
        }
        if (nrBlocks > 0 && curBlk == nrBlocks) {
            return Collections.emptyList();
        }
        long endOff = offset + length;
        ArrayList<LocatedBlock> results = new ArrayList<LocatedBlock>(blocks.length);
        do {
            results.add(this.getBlockLocation(blocks[curBlk], curPos));
        } while ((curPos += blocks[++curBlk].getNumBytes()) < endOff && curBlk < blocks.length && results.size() < nrBlocksToReturn);
        return results;
    }

    LocatedBlock getBlockLocation(BlockInfo blk, long pos) throws IOException {
        int numNodes;
        int numCorruptReplicas;
        if (!blk.isComplete()) {
            BlockInfoUnderConstruction uc = (BlockInfoUnderConstruction)blk;
            DatanodeInfo[] locations = uc.getExpectedLocations();
            return this.namesystem.createLocatedBlock(uc, locations, pos, false);
        }
        int numCorruptNodes = this.countNodes(blk).corruptReplicas();
        if (numCorruptNodes != (numCorruptReplicas = this.corruptReplicas.numCorruptReplicas(blk))) {
            FSNamesystem.LOG.warn((Object)("Inconsistent number of corrupt replicas for " + blk + " blockMap has " + numCorruptNodes + " but corrupt replicas map has " + numCorruptReplicas));
        }
        boolean isCorrupt = numCorruptNodes == (numNodes = this.blocksMap.numNodes(blk));
        int numMachines = isCorrupt ? numNodes : numNodes - numCorruptNodes;
        DatanodeInfo[] machines = new DatanodeDescriptor[numMachines];
        if (numMachines > 0) {
            int j = 0;
            Iterator<DatanodeDescriptor> it = this.blocksMap.nodeIterator(blk);
            while (it.hasNext()) {
                DatanodeDescriptor d = it.next();
                boolean replicaCorrupt = this.corruptReplicas.isReplicaCorrupt(blk, d);
                if (!isCorrupt && (isCorrupt || replicaCorrupt)) continue;
                machines[j++] = d;
            }
        }
        return this.namesystem.createLocatedBlock(blk, machines, pos, isCorrupt);
    }

    void verifyReplication(String src, short replication, String clientName) throws IOException {
        if (replication >= this.minReplication && replication <= this.maxReplication) {
            return;
        }
        String text = "file " + src + (clientName != null ? " on client " + clientName : "") + ".\n" + "Requested replication " + replication;
        if (replication > this.maxReplication) {
            throw new IOException(text + " exceeds maximum " + this.maxReplication);
        }
        if (replication < this.minReplication) {
            throw new IOException(text + " is less than the required minimum " + this.minReplication);
        }
    }

    void removeFromInvalidates(String datanodeId, Block block) {
        Collection<Block> v = this.recentInvalidateSets.get(datanodeId);
        if (v != null && v.remove(block)) {
            --this.pendingDeletionBlocksCount;
            if (v.isEmpty()) {
                this.recentInvalidateSets.remove(datanodeId);
            }
        }
    }

    void addToInvalidates(Block b, DatanodeInfo dn, boolean log) {
        Collection<Block> invalidateSet = this.recentInvalidateSets.get(dn.getStorageID());
        if (invalidateSet == null) {
            invalidateSet = new HashSet<Block>();
            this.recentInvalidateSets.put(dn.getStorageID(), invalidateSet);
        }
        if (invalidateSet.add(b)) {
            ++this.pendingDeletionBlocksCount;
            if (log) {
                NameNode.stateChangeLog.info((Object)("BLOCK* NameSystem.addToInvalidates: " + b + " to " + dn.getName()));
            }
        }
    }

    void addToInvalidates(Block b, DatanodeInfo dn) {
        this.addToInvalidates(b, dn, true);
    }

    private void addToInvalidates(Block b) {
        StringBuilder datanodes = new StringBuilder();
        Iterator<DatanodeDescriptor> it = this.blocksMap.nodeIterator(b);
        while (it.hasNext()) {
            DatanodeDescriptor node = it.next();
            this.addToInvalidates(b, node, false);
            datanodes.append(node.getName()).append(" ");
        }
        if (datanodes.length() != 0) {
            NameNode.stateChangeLog.info((Object)("BLOCK* NameSystem.addToInvalidates: " + b + " to " + datanodes.toString()));
        }
    }

    private void dumpRecentInvalidateSets(PrintWriter out) {
        int size = this.recentInvalidateSets.values().size();
        out.println("Metasave: Blocks " + this.pendingDeletionBlocksCount + " waiting deletion from " + size + " datanodes.");
        if (size == 0) {
            return;
        }
        for (Map.Entry<String, Collection<Block>> entry : this.recentInvalidateSets.entrySet()) {
            Collection<Block> blocks = entry.getValue();
            if (blocks.size() <= 0) continue;
            out.println(this.namesystem.getDatanode(entry.getKey()).getName() + blocks);
        }
    }

    void findAndMarkBlockAsCorrupt(Block blk, DatanodeInfo dn) throws IOException {
        BlockInfo storedBlock = this.getStoredBlock(blk);
        if (storedBlock == null) {
            NameNode.stateChangeLog.info((Object)("BLOCK* NameSystem.markBlockAsCorrupt: block " + blk + " could not be marked as " + "corrupt as it does not exist in blocksMap"));
            return;
        }
        this.markBlockAsCorrupt(storedBlock, dn);
    }

    private void markBlockAsCorrupt(BlockInfo storedBlock, DatanodeInfo dn) throws IOException {
        assert (storedBlock != null) : "storedBlock should not be null";
        DatanodeDescriptor node = this.namesystem.getDatanode(dn);
        if (node == null) {
            throw new IOException("Cannot mark block " + storedBlock.getBlockName() + " as corrupt because datanode " + dn.getName() + " does not exist. ");
        }
        INodeFile inode = storedBlock.getINode();
        if (inode == null) {
            NameNode.stateChangeLog.info((Object)("BLOCK NameSystem.markBlockAsCorrupt: block " + storedBlock + " could not be marked as corrupt as it" + " does not belong to any file"));
            this.addToInvalidates(storedBlock, node);
            return;
        }
        node.addBlock(storedBlock);
        this.corruptReplicas.addToCorruptReplicasMap(storedBlock, node);
        if (this.countNodes(storedBlock).liveReplicas() > inode.getReplication()) {
            this.invalidateBlock(storedBlock, node);
        } else {
            this.updateNeededReplications(storedBlock, -1, 0);
        }
    }

    private void invalidateBlock(Block blk, DatanodeInfo dn) throws IOException {
        NameNode.stateChangeLog.info((Object)("DIR* NameSystem.invalidateBlock: " + blk + " on " + dn.getName()));
        DatanodeDescriptor node = this.namesystem.getDatanode(dn);
        if (node == null) {
            throw new IOException("Cannot invalidate block " + blk + " because datanode " + dn.getName() + " does not exist.");
        }
        int count = this.countNodes(blk).liveReplicas();
        if (count > 1) {
            this.addToInvalidates(blk, dn);
            this.removeStoredBlock(blk, node);
            NameNode.stateChangeLog.debug((Object)("BLOCK* NameSystem.invalidateBlocks: " + blk + " on " + dn.getName() + " listed for deletion."));
        } else {
            NameNode.stateChangeLog.info((Object)("BLOCK* NameSystem.invalidateBlocks: " + blk + " on " + dn.getName() + " is the only copy and was not deleted."));
        }
    }

    void updateState() {
        this.pendingReplicationBlocksCount = this.pendingReplications.size();
        this.underReplicatedBlocksCount = this.neededReplications.size();
        this.corruptReplicaBlocksCount = this.corruptReplicas.size();
    }

    int computeInvalidateWork(int nodesToProcess) {
        int keyIndex;
        int i;
        int numOfNodes = this.recentInvalidateSets.size();
        nodesToProcess = Math.min(numOfNodes, nodesToProcess);
        ArrayList<String> keyArray = new ArrayList<String>(this.recentInvalidateSets.keySet());
        int remainingNodes = numOfNodes - nodesToProcess;
        if (nodesToProcess < remainingNodes) {
            for (i = 0; i < nodesToProcess; ++i) {
                keyIndex = this.r.nextInt(numOfNodes - i) + i;
                Collections.swap(keyArray, keyIndex, i);
            }
        } else {
            for (i = 0; i < remainingNodes; ++i) {
                keyIndex = this.r.nextInt(numOfNodes - i);
                Collections.swap(keyArray, keyIndex, numOfNodes - i - 1);
            }
        }
        int blockCnt = 0;
        for (int nodeCnt = 0; nodeCnt < nodesToProcess; ++nodeCnt) {
            blockCnt += this.invalidateWorkForOneNode(keyArray.get(nodeCnt));
        }
        return blockCnt;
    }

    int computeReplicationWork(int blocksToProcess) throws IOException {
        List<List<Block>> blocksToReplicate = this.chooseUnderReplicatedBlocks(blocksToProcess);
        int scheduledReplicationCount = 0;
        for (int i = 0; i < blocksToReplicate.size(); ++i) {
            for (Block block : blocksToReplicate.get(i)) {
                if (!this.computeReplicationWorkForBlock(block, i)) continue;
                ++scheduledReplicationCount;
            }
        }
        return scheduledReplicationCount;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private List<List<Block>> chooseUnderReplicatedBlocks(int blocksToProcess) {
        ArrayList<List<Block>> blocksToReplicate = new ArrayList<List<Block>>(4);
        for (int i = 0; i < 4; ++i) {
            blocksToReplicate.add(new ArrayList());
        }
        FSNamesystem fSNamesystem = this.namesystem;
        synchronized (fSNamesystem) {
            UnderReplicatedBlocks underReplicatedBlocks = this.neededReplications;
            synchronized (underReplicatedBlocks) {
                if (this.neededReplications.size() == 0) {
                    this.missingBlocksInCurIter = 0L;
                    this.missingBlocksInPrevIter = 0L;
                    return blocksToReplicate;
                }
                UnderReplicatedBlocks.BlockIterator neededReplicationsIterator = this.neededReplications.iterator();
                for (int i = 0; i < this.replIndex && neededReplicationsIterator.hasNext(); ++i) {
                    neededReplicationsIterator.next();
                }
                blocksToProcess = Math.min(blocksToProcess, this.neededReplications.size());
                int blkCnt = 0;
                while (blkCnt < blocksToProcess) {
                    if (!neededReplicationsIterator.hasNext()) {
                        this.replIndex = 0;
                        this.missingBlocksInPrevIter = this.missingBlocksInCurIter;
                        this.missingBlocksInCurIter = 0L;
                        if (blkCnt >= (blocksToProcess = Math.min(blocksToProcess, this.neededReplications.size()))) break;
                        neededReplicationsIterator = this.neededReplications.iterator();
                        assert (neededReplicationsIterator.hasNext()) : "neededReplications should not be empty.";
                    }
                    Block block = neededReplicationsIterator.next();
                    int priority = neededReplicationsIterator.getPriority();
                    if (priority < 0 || priority >= blocksToReplicate.size()) {
                        FSNamesystem.LOG.warn((Object)("Unexpected replication priority: " + priority + " " + block));
                    } else {
                        ((List)blocksToReplicate.get(priority)).add(block);
                    }
                    ++blkCnt;
                    ++this.replIndex;
                }
            }
        }
        return blocksToReplicate;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean computeReplicationWorkForBlock(Block block, int priority) {
        int additionalReplRequired;
        int numEffectiveReplicas;
        DatanodeDescriptor srcNode;
        ArrayList<DatanodeDescriptor> containingNodes;
        short requiredReplication;
        Object object;
        INodeFile fileINode = null;
        FSNamesystem fSNamesystem = this.namesystem;
        synchronized (fSNamesystem) {
            object = this.neededReplications;
            synchronized (object) {
                fileINode = this.blocksMap.getINode(block);
                if (fileINode == null || fileINode.isUnderConstruction()) {
                    this.neededReplications.remove(block, priority);
                    --this.replIndex;
                    return false;
                }
                requiredReplication = fileINode.getReplication();
                containingNodes = new ArrayList<DatanodeDescriptor>();
                FSNamesystem.NumberReplicas numReplicas = new FSNamesystem.NumberReplicas();
                srcNode = this.chooseSourceDatanode(block, containingNodes, numReplicas);
                if (numReplicas.liveReplicas() + numReplicas.decommissionedReplicas() <= 0) {
                    ++this.missingBlocksInCurIter;
                }
                if (srcNode == null) {
                    return false;
                }
                numEffectiveReplicas = numReplicas.liveReplicas() + this.pendingReplications.getNumReplicas(block);
                if (numEffectiveReplicas >= requiredReplication && (this.pendingReplications.getNumReplicas(block) > 0 || this.blockHasEnoughRacks(block))) {
                    this.neededReplications.remove(block, priority);
                    --this.replIndex;
                    NameNode.stateChangeLog.info((Object)("BLOCK* Removing block " + block + " from neededReplications as it has enough replicas."));
                    return false;
                }
                additionalReplRequired = numReplicas.liveReplicas() < requiredReplication ? requiredReplication - numEffectiveReplicas : 1;
            }
        }
        DatanodeDescriptor[] targets = this.replicator.chooseTarget(fileINode, additionalReplRequired, srcNode, containingNodes, block.getNumBytes());
        if (targets.length == 0) {
            return false;
        }
        object = this.namesystem;
        synchronized (object) {
            UnderReplicatedBlocks underReplicatedBlocks = this.neededReplications;
            synchronized (underReplicatedBlocks) {
                fileINode = this.blocksMap.getINode(block);
                if (fileINode == null || fileINode.isUnderConstruction()) {
                    this.neededReplications.remove(block, priority);
                    --this.replIndex;
                    return false;
                }
                requiredReplication = fileINode.getReplication();
                FSNamesystem.NumberReplicas numReplicas = this.countNodes(block);
                numEffectiveReplicas = numReplicas.liveReplicas() + this.pendingReplications.getNumReplicas(block);
                if (numEffectiveReplicas >= requiredReplication && (this.pendingReplications.getNumReplicas(block) > 0 || this.blockHasEnoughRacks(block))) {
                    this.neededReplications.remove(block, priority);
                    --this.replIndex;
                    NameNode.stateChangeLog.info((Object)("BLOCK* Removing block " + block + " from neededReplications as it has enough replicas."));
                    return false;
                }
                if (numReplicas.liveReplicas() >= requiredReplication && !this.blockHasEnoughRacks(block) && srcNode.getNetworkLocation().equals(targets[0].getNetworkLocation())) {
                    return false;
                }
                srcNode.addBlockToBeReplicated(block, targets);
                for (DatanodeDescriptor dn : targets) {
                    dn.incBlocksScheduled();
                }
                this.pendingReplications.add(block, targets.length);
                NameNode.stateChangeLog.debug((Object)("BLOCK* block " + block + " is moved from neededReplications to pendingReplications"));
                if (numEffectiveReplicas + targets.length >= requiredReplication) {
                    this.neededReplications.remove(block, priority);
                    --this.replIndex;
                }
                if (NameNode.stateChangeLog.isInfoEnabled()) {
                    StringBuilder targetList = new StringBuilder("datanode(s)");
                    for (int k = 0; k < targets.length; ++k) {
                        targetList.append(' ');
                        targetList.append(targets[k].getName());
                    }
                    NameNode.stateChangeLog.info((Object)("BLOCK* ask " + srcNode.getName() + " to replicate " + block + " to " + targetList));
                    NameNode.stateChangeLog.debug((Object)("BLOCK* neededReplications = " + this.neededReplications.size() + " pendingReplications = " + this.pendingReplications.size()));
                }
            }
        }
        return true;
    }

    private DatanodeDescriptor chooseSourceDatanode(Block block, List<DatanodeDescriptor> containingNodes, FSNamesystem.NumberReplicas numReplicas) {
        containingNodes.clear();
        DatanodeDescriptor srcNode = null;
        int live = 0;
        int decommissioned = 0;
        int corrupt = 0;
        int excess = 0;
        Iterator<DatanodeDescriptor> it = this.blocksMap.nodeIterator(block);
        Collection<DatanodeDescriptor> nodesCorrupt = this.corruptReplicas.getNodes(block);
        while (it.hasNext()) {
            DatanodeDescriptor node = it.next();
            Collection<Block> excessBlocks = this.excessReplicateMap.get(node.getStorageID());
            if (nodesCorrupt != null && nodesCorrupt.contains(node)) {
                ++corrupt;
            } else if (node.isDecommissionInProgress() || node.isDecommissioned()) {
                ++decommissioned;
            } else if (excessBlocks != null && excessBlocks.contains(block)) {
                ++excess;
            } else {
                ++live;
            }
            containingNodes.add(node);
            if (nodesCorrupt != null && nodesCorrupt.contains(node) || node.getNumberOfBlocksToBeReplicated() >= this.maxReplicationStreams || excessBlocks != null && excessBlocks.contains(block) || node.isDecommissioned()) continue;
            if (node.isDecommissionInProgress() || srcNode == null) {
                srcNode = node;
                continue;
            }
            if (srcNode.isDecommissionInProgress() || !this.r.nextBoolean()) continue;
            srcNode = node;
        }
        if (numReplicas != null) {
            numReplicas.initialize(live, decommissioned, corrupt, excess);
        }
        return srcNode;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void processPendingReplications() {
        Block[] timedOutItems = this.pendingReplications.getTimedOutBlocks();
        if (timedOutItems != null) {
            FSNamesystem fSNamesystem = this.namesystem;
            synchronized (fSNamesystem) {
                for (int i = 0; i < timedOutItems.length; ++i) {
                    FSNamesystem.NumberReplicas num = this.countNodes(timedOutItems[i]);
                    if (!this.isNeededReplication(timedOutItems[i], this.getReplication(timedOutItems[i]), num.liveReplicas())) continue;
                    this.neededReplications.add(timedOutItems[i], num.liveReplicas(), num.decommissionedReplicas(), this.getReplication(timedOutItems[i]));
                }
            }
        }
    }

    public void processReport(DatanodeDescriptor node, BlockListAsLongs report) throws IOException {
        LinkedList<Block> toAdd = new LinkedList<Block>();
        LinkedList<Block> toRemove = new LinkedList<Block>();
        LinkedList<Block> toInvalidate = new LinkedList<Block>();
        LinkedList<BlockInfo> toCorrupt = new LinkedList<BlockInfo>();
        node.reportDiff(this, report, toAdd, toRemove, toInvalidate, toCorrupt);
        for (Block block : toRemove) {
            this.removeStoredBlock(block, node);
        }
        for (Block block : toAdd) {
            this.addStoredBlock(block, node, null);
        }
        for (Block block : toInvalidate) {
            NameNode.stateChangeLog.info((Object)("BLOCK* NameSystem.processReport: block " + block + " on " + node.getName() + " size " + block.getNumBytes() + " does not belong to any file."));
            this.addToInvalidates(block, node);
        }
        for (BlockInfo blockInfo : toCorrupt) {
            this.markBlockAsCorrupt(blockInfo, node);
        }
    }

    private Block addStoredBlock(Block block, DatanodeDescriptor node, DatanodeDescriptor delNodeHint) throws IOException {
        BlockInfo storedBlock = this.blocksMap.getStoredBlock(block);
        if (storedBlock == null || storedBlock.getINode() == null) {
            NameNode.stateChangeLog.info((Object)("BLOCK* NameSystem.addStoredBlock: addStoredBlock request received for " + block + " on " + node.getName() + " size " + block.getNumBytes() + " But it does not belong to any file."));
            return block;
        }
        assert (storedBlock != null) : "Block must be stored by now";
        INodeFile fileINode = storedBlock.getINode();
        assert (fileINode != null) : "Block must belong to a file";
        boolean added = node.addBlock(storedBlock);
        int curReplicaDelta = 0;
        if (added) {
            curReplicaDelta = 1;
            if (!this.namesystem.isInSafeMode()) {
                NameNode.stateChangeLog.info((Object)("BLOCK* NameSystem.addStoredBlock: blockMap updated: " + node.getName() + " is added to " + storedBlock + " size " + storedBlock.getNumBytes()));
            }
        } else {
            NameNode.stateChangeLog.warn((Object)("BLOCK* NameSystem.addStoredBlock: Redundant addStoredBlock request received for " + storedBlock + " on " + node.getName() + " size " + storedBlock.getNumBytes()));
        }
        FSNamesystem.NumberReplicas num = this.countNodes(storedBlock);
        int numLiveReplicas = num.liveReplicas();
        int numCurrentReplica = numLiveReplicas + this.pendingReplications.getNumReplicas(storedBlock);
        if (storedBlock.getBlockUCState() == HdfsConstants.BlockUCState.COMMITTED && numLiveReplicas >= this.minReplication) {
            storedBlock = this.completeBlock(fileINode, storedBlock);
        }
        if (storedBlock.isComplete()) {
            this.namesystem.incrementSafeBlockCount(numCurrentReplica);
        }
        if (fileINode.isUnderConstruction()) {
            return storedBlock;
        }
        if (this.namesystem.isInSafeMode()) {
            return storedBlock;
        }
        short fileReplication = fileINode.getReplication();
        if (!this.isNeededReplication(storedBlock, fileReplication, numCurrentReplica)) {
            this.neededReplications.remove(storedBlock, numCurrentReplica, num.decommissionedReplicas, fileReplication);
        } else {
            this.updateNeededReplications(storedBlock, curReplicaDelta, 0);
        }
        if (numCurrentReplica > fileReplication) {
            this.processOverReplicatedBlock(storedBlock, fileReplication, node, delNodeHint);
        }
        int corruptReplicasCount = this.corruptReplicas.numCorruptReplicas(storedBlock);
        int numCorruptNodes = num.corruptReplicas();
        if (numCorruptNodes != corruptReplicasCount) {
            FSNamesystem.LOG.warn((Object)("Inconsistent number of corrupt replicas for " + storedBlock + "blockMap has " + numCorruptNodes + " but corrupt replicas map has " + corruptReplicasCount));
        }
        if (corruptReplicasCount > 0 && numLiveReplicas >= fileReplication) {
            this.invalidateCorruptReplicas(storedBlock);
        }
        return storedBlock;
    }

    private void invalidateCorruptReplicas(Block blk) {
        DatanodeDescriptor[] nodesCopy;
        Collection<DatanodeDescriptor> nodes = this.corruptReplicas.getNodes(blk);
        boolean gotException = false;
        if (nodes == null) {
            return;
        }
        for (DatanodeDescriptor node : nodesCopy = nodes.toArray(new DatanodeDescriptor[0])) {
            try {
                this.invalidateBlock(blk, node);
            }
            catch (IOException e) {
                NameNode.stateChangeLog.info((Object)("NameNode.invalidateCorruptReplicas error in deleting bad block " + blk + " on " + node + e));
                gotException = true;
            }
        }
        if (!gotException) {
            this.corruptReplicas.removeFromCorruptReplicasMap(blk);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void processMisReplicatedBlocks() {
        long nrInvalid = 0L;
        long nrOverReplicated = 0L;
        long nrUnderReplicated = 0L;
        FSNamesystem fSNamesystem = this.namesystem;
        synchronized (fSNamesystem) {
            this.neededReplications.clear();
            for (BlockInfo block : this.blocksMap.getBlocks()) {
                FSNamesystem.NumberReplicas num;
                int numCurrentReplica;
                INodeFile fileINode = block.getINode();
                if (fileINode == null) {
                    ++nrInvalid;
                    this.addToInvalidates(block);
                    continue;
                }
                short expectedReplication = fileINode.getReplication();
                if (this.isNeededReplication(block, expectedReplication, numCurrentReplica = (num = this.countNodes(block)).liveReplicas()) && this.neededReplications.add(block, numCurrentReplica, num.decommissionedReplicas(), expectedReplication)) {
                    ++nrUnderReplicated;
                }
                if (numCurrentReplica <= expectedReplication) continue;
                ++nrOverReplicated;
                this.processOverReplicatedBlock(block, expectedReplication, null, null);
            }
        }
        FSNamesystem.LOG.info((Object)("Total number of blocks = " + this.blocksMap.size()));
        FSNamesystem.LOG.info((Object)("Number of invalid blocks = " + nrInvalid));
        FSNamesystem.LOG.info((Object)("Number of under-replicated blocks = " + nrUnderReplicated));
        FSNamesystem.LOG.info((Object)("Number of  over-replicated blocks = " + nrOverReplicated));
    }

    void processOverReplicatedBlock(Block block, short replication, DatanodeDescriptor addedNode, DatanodeDescriptor delNodeHint) {
        if (addedNode == delNodeHint) {
            delNodeHint = null;
        }
        ArrayList<DatanodeDescriptor> nonExcess = new ArrayList<DatanodeDescriptor>();
        Collection<DatanodeDescriptor> corruptNodes = this.corruptReplicas.getNodes(block);
        Iterator<DatanodeDescriptor> it = this.blocksMap.nodeIterator(block);
        while (it.hasNext()) {
            DatanodeDescriptor cur = it.next();
            Collection<Block> excessBlocks = this.excessReplicateMap.get(cur.getStorageID());
            if (excessBlocks != null && excessBlocks.contains(block) || cur.isDecommissionInProgress() || cur.isDecommissioned() || corruptNodes != null && corruptNodes.contains(cur)) continue;
            nonExcess.add(cur);
        }
        this.namesystem.chooseExcessReplicates(nonExcess, block, replication, addedNode, delNodeHint, this.replicator);
    }

    void addToExcessReplicate(DatanodeInfo dn, Block block) {
        Collection<Block> excessBlocks = this.excessReplicateMap.get(dn.getStorageID());
        if (excessBlocks == null) {
            excessBlocks = new TreeSet<Block>();
            this.excessReplicateMap.put(dn.getStorageID(), excessBlocks);
        }
        if (excessBlocks.add(block)) {
            ++this.excessBlocksCount;
            NameNode.stateChangeLog.debug((Object)("BLOCK* NameSystem.chooseExcessReplicates: (" + dn.getName() + ", " + block + ") is added to excessReplicateMap"));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void removeStoredBlock(Block block, DatanodeDescriptor node) {
        NameNode.stateChangeLog.debug((Object)("BLOCK* NameSystem.removeStoredBlock: " + block + " from " + node.getName()));
        FSNamesystem fSNamesystem = this.namesystem;
        synchronized (fSNamesystem) {
            Collection<Block> excessBlocks;
            if (!this.blocksMap.removeNode(block, node)) {
                NameNode.stateChangeLog.debug((Object)("BLOCK* NameSystem.removeStoredBlock: " + block + " has already been removed from node " + node));
                return;
            }
            INodeFile fileINode = this.blocksMap.getINode(block);
            if (fileINode != null) {
                this.namesystem.decrementSafeBlockCount(block);
                this.updateNeededReplications(block, -1, 0);
            }
            if ((excessBlocks = this.excessReplicateMap.get(node.getStorageID())) != null && excessBlocks.remove(block)) {
                --this.excessBlocksCount;
                NameNode.stateChangeLog.debug((Object)("BLOCK* NameSystem.removeStoredBlock: " + block + " is removed from excessBlocks"));
                if (excessBlocks.size() == 0) {
                    this.excessReplicateMap.remove(node.getStorageID());
                }
            }
            this.corruptReplicas.removeFromCorruptReplicasMap(block, node);
        }
    }

    void addBlock(DatanodeDescriptor node, Block block, String delHint) throws IOException {
        node.decBlocksScheduled();
        DatanodeDescriptor delHintNode = null;
        if (delHint != null && delHint.length() != 0 && (delHintNode = this.namesystem.getDatanode(delHint)) == null) {
            NameNode.stateChangeLog.warn((Object)("BLOCK* NameSystem.blockReceived: " + block + " is expected to be removed from an unrecorded node " + delHint));
        }
        this.pendingReplications.remove(block);
        LinkedList<Block> toAdd = new LinkedList<Block>();
        LinkedList<Block> toInvalidate = new LinkedList<Block>();
        LinkedList<BlockInfo> toCorrupt = new LinkedList<BlockInfo>();
        node.processReportedBlock(this, block, HdfsConstants.ReplicaState.FINALIZED, toAdd, toInvalidate, toCorrupt);
        assert (toAdd.size() + toInvalidate.size() <= 1) : "The block should be only in one of the lists.";
        for (Block block2 : toAdd) {
            this.addStoredBlock(block2, node, delHintNode);
        }
        for (Block block3 : toInvalidate) {
            NameNode.stateChangeLog.info((Object)("BLOCK* NameSystem.addBlock: block " + block3 + " on " + node.getName() + " size " + block3.getNumBytes() + " does not belong to any file."));
            this.addToInvalidates(block3, node);
        }
        for (BlockInfo blockInfo : toCorrupt) {
            this.markBlockAsCorrupt(blockInfo, node);
        }
    }

    FSNamesystem.NumberReplicas countNodes(Block b) {
        int count = 0;
        int live = 0;
        int corrupt = 0;
        int excess = 0;
        Iterator<DatanodeDescriptor> nodeIter = this.blocksMap.nodeIterator(b);
        Collection<DatanodeDescriptor> nodesCorrupt = this.corruptReplicas.getNodes(b);
        while (nodeIter.hasNext()) {
            DatanodeDescriptor node = nodeIter.next();
            if (nodesCorrupt != null && nodesCorrupt.contains(node)) {
                ++corrupt;
                continue;
            }
            if (node.isDecommissionInProgress() || node.isDecommissioned()) {
                ++count;
                continue;
            }
            Collection<Block> blocksExcess = this.excessReplicateMap.get(node.getStorageID());
            if (blocksExcess != null && blocksExcess.contains(b)) {
                ++excess;
                continue;
            }
            ++live;
        }
        return new FSNamesystem.NumberReplicas(live, count, corrupt, excess);
    }

    private void logBlockReplicationInfo(Block block, DatanodeDescriptor srcNode, FSNamesystem.NumberReplicas num) {
        int curReplicas = num.liveReplicas();
        int curExpectedReplicas = this.getReplication(block);
        INodeFile fileINode = this.blocksMap.getINode(block);
        Iterator<DatanodeDescriptor> nodeIter = this.blocksMap.nodeIterator(block);
        StringBuilder nodeList = new StringBuilder();
        while (nodeIter.hasNext()) {
            DatanodeDescriptor node = nodeIter.next();
            nodeList.append(node.name);
            nodeList.append(" ");
        }
        FSNamesystem.LOG.info((Object)("Block: " + block + ", Expected Replicas: " + curExpectedReplicas + ", live replicas: " + curReplicas + ", corrupt replicas: " + num.corruptReplicas() + ", decommissioned replicas: " + num.decommissionedReplicas() + ", excess replicas: " + num.excessReplicas() + ", Is Open File: " + fileINode.isUnderConstruction() + ", Datanodes having this block: " + nodeList + ", Current Datanode: " + srcNode.name + ", Is current datanode decommissioning: " + srcNode.isDecommissionInProgress()));
    }

    boolean isReplicationInProgress(DatanodeDescriptor srcNode) {
        boolean status = false;
        int underReplicatedBlocks = 0;
        int decommissionOnlyReplicas = 0;
        int underReplicatedInOpenFiles = 0;
        Iterator<BlockInfo> it = srcNode.getBlockIterator();
        while (it.hasNext()) {
            Block block = it.next();
            INodeFile fileINode = this.blocksMap.getINode(block);
            if (fileINode == null) continue;
            FSNamesystem.NumberReplicas num = this.countNodes(block);
            int curReplicas = num.liveReplicas();
            int curExpectedReplicas = this.getReplication(block);
            if (!this.isNeededReplication(block, curExpectedReplicas, curReplicas)) continue;
            if (curExpectedReplicas > curReplicas) {
                if (!status) {
                    status = true;
                    this.logBlockReplicationInfo(block, srcNode, num);
                }
                ++underReplicatedBlocks;
                if (curReplicas == 0 && num.decommissionedReplicas() > 0) {
                    ++decommissionOnlyReplicas;
                }
                if (fileINode.isUnderConstruction()) {
                    ++underReplicatedInOpenFiles;
                }
            }
            if (this.neededReplications.contains(block) || this.pendingReplications.getNumReplicas(block) != 0) continue;
            this.neededReplications.add(block, curReplicas, num.decommissionedReplicas(), curExpectedReplicas);
        }
        srcNode.decommissioningStatus.set(underReplicatedBlocks, decommissionOnlyReplicas, underReplicatedInOpenFiles);
        return status;
    }

    int getActiveBlockCount() {
        return this.blocksMap.size() - (int)this.pendingDeletionBlocksCount;
    }

    DatanodeDescriptor[] getNodes(BlockInfo block) {
        DatanodeDescriptor[] nodes = new DatanodeDescriptor[block.numNodes()];
        Iterator<DatanodeDescriptor> it = this.blocksMap.nodeIterator(block);
        int i = 0;
        while (it != null && it.hasNext()) {
            nodes[i] = it.next();
            ++i;
        }
        return nodes;
    }

    int getTotalBlocks() {
        return this.blocksMap.size();
    }

    void removeBlock(Block block) {
        this.addToInvalidates(block);
        this.corruptReplicas.removeFromCorruptReplicasMap(block);
        this.blocksMap.removeBlock(block);
    }

    BlockInfo getStoredBlock(Block block) {
        return this.blocksMap.getStoredBlock(block);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void updateNeededReplications(Block block, int curReplicasDelta, int expectedReplicasDelta) {
        FSNamesystem fSNamesystem = this.namesystem;
        synchronized (fSNamesystem) {
            FSNamesystem.NumberReplicas repl = this.countNodes(block);
            int curExpectedReplicas = this.getReplication(block);
            if (this.isNeededReplication(block, curExpectedReplicas, repl.liveReplicas())) {
                this.neededReplications.update(block, repl.liveReplicas(), repl.decommissionedReplicas(), curExpectedReplicas, curReplicasDelta, expectedReplicasDelta);
            } else {
                int oldReplicas = repl.liveReplicas() - curReplicasDelta;
                int oldExpectedReplicas = curExpectedReplicas - expectedReplicasDelta;
                this.neededReplications.remove(block, oldReplicas, repl.decommissionedReplicas(), oldExpectedReplicas);
            }
        }
    }

    void checkReplication(Block block, int numExpectedReplicas) {
        FSNamesystem.NumberReplicas number = this.countNodes(block);
        if (this.isNeededReplication(block, numExpectedReplicas, number.liveReplicas())) {
            this.neededReplications.add(block, number.liveReplicas(), number.decommissionedReplicas, numExpectedReplicas);
        }
    }

    private int getReplication(Block block) {
        INodeFile fileINode = this.blocksMap.getINode(block);
        if (fileINode == null) {
            return 0;
        }
        assert (!fileINode.isDirectory()) : "Block cannot belong to a directory.";
        return fileINode.getReplication();
    }

    void removeFromInvalidates(String storageID) {
        Collection<Block> blocks = this.recentInvalidateSets.remove(storageID);
        if (blocks != null) {
            this.pendingDeletionBlocksCount -= (long)blocks.size();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private int invalidateWorkForOneNode(String nodeId) {
        FSNamesystem fSNamesystem = this.namesystem;
        synchronized (fSNamesystem) {
            if (this.namesystem.isInSafeMode()) {
                return 0;
            }
            assert (nodeId != null);
            DatanodeDescriptor dn = this.namesystem.getDatanode(nodeId);
            if (dn == null) {
                this.removeFromInvalidates(nodeId);
                return 0;
            }
            Collection<Block> invalidateSet = this.recentInvalidateSets.get(nodeId);
            if (invalidateSet == null) {
                return 0;
            }
            ArrayList<Block> blocksToInvalidate = new ArrayList<Block>(this.namesystem.blockInvalidateLimit);
            Iterator<Block> it = invalidateSet.iterator();
            for (int blkCount = 0; blkCount < this.namesystem.blockInvalidateLimit && it.hasNext(); ++blkCount) {
                blocksToInvalidate.add(it.next());
                it.remove();
            }
            if (!it.hasNext()) {
                this.removeFromInvalidates(nodeId);
            }
            dn.addBlocksToBeInvalidated(blocksToInvalidate);
            if (NameNode.stateChangeLog.isInfoEnabled()) {
                StringBuilder blockList = new StringBuilder();
                for (Block blk : blocksToInvalidate) {
                    blockList.append(' ');
                    blockList.append(blk);
                }
                NameNode.stateChangeLog.info((Object)("BLOCK* ask " + dn.getName() + " to delete " + blockList));
            }
            this.pendingDeletionBlocksCount -= (long)blocksToInvalidate.size();
            return blocksToInvalidate.size();
        }
    }

    int getNumberOfRacks(Block b) {
        HashSet<String> rackSet = new HashSet<String>(0);
        Collection<DatanodeDescriptor> corruptNodes = this.corruptReplicas.getNodes(b);
        Iterator<DatanodeDescriptor> it = this.blocksMap.nodeIterator(b);
        while (it.hasNext()) {
            String rackName;
            DatanodeDescriptor cur = it.next();
            if (cur.isDecommissionInProgress() || cur.isDecommissioned() || corruptNodes != null && corruptNodes.contains(cur) || rackSet.contains(rackName = cur.getNetworkLocation())) continue;
            rackSet.add(rackName);
        }
        return rackSet.size();
    }

    boolean blockHasEnoughRacks(Block b) {
        if (!this.shouldCheckForEnoughRacks) {
            return true;
        }
        boolean enoughRacks = false;
        Collection<DatanodeDescriptor> corruptNodes = this.corruptReplicas.getNodes(b);
        int numExpectedReplicas = this.getReplication(b);
        String rackName = null;
        Iterator<DatanodeDescriptor> it = this.blocksMap.nodeIterator(b);
        while (it.hasNext()) {
            DatanodeDescriptor cur = it.next();
            if (cur.isDecommissionInProgress() || cur.isDecommissioned() || corruptNodes != null && corruptNodes.contains(cur)) continue;
            if (numExpectedReplicas == 1) {
                enoughRacks = true;
                break;
            }
            String rackNameNew = cur.getNetworkLocation();
            if (rackName == null) {
                rackName = rackNameNew;
                continue;
            }
            if (rackName.equals(rackNameNew)) continue;
            enoughRacks = true;
            break;
        }
        return enoughRacks;
    }

    boolean isNeededReplication(Block b, int expectedReplication, int curReplicas) {
        return curReplicas < expectedReplication || !this.blockHasEnoughRacks(b);
    }

    long getMissingBlocksCount() {
        return Math.max(this.missingBlocksInPrevIter, this.missingBlocksInCurIter);
    }

    BlockInfo addINode(BlockInfo block, INodeFile iNode) {
        return this.blocksMap.addINode(block, iNode);
    }

    INodeFile getINode(Block b) {
        return this.blocksMap.getINode(b);
    }

    void removeFromCorruptReplicasMap(Block block) {
        this.corruptReplicas.removeFromCorruptReplicasMap(block);
    }

    int numCorruptReplicas(Block block) {
        return this.corruptReplicas.numCorruptReplicas(block);
    }

    void removeBlockFromMap(Block block) {
        this.blocksMap.removeBlock(block);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    int getCapacity() {
        FSNamesystem fSNamesystem = this.namesystem;
        synchronized (fSNamesystem) {
            return this.blocksMap.getCapacity();
        }
    }

    float getLoadFactor() {
        return this.blocksMap.getLoadFactor();
    }

    long[] getCorruptReplicaBlockIds(int numExpectedBlocks, Long startingBlockId) {
        return this.corruptReplicas.getCorruptReplicaBlockIds(numExpectedBlocks, startingBlockId);
    }

    INode[] getCorruptInodes() {
        LinkedHashSet<INodeFile> set = new LinkedHashSet<INodeFile>();
        for (Block blk : this.neededReplications.getQueue(2)) {
            INodeFile inode = this.blocksMap.getINode(blk);
            if (inode == null || this.countNodes(blk).liveReplicas() != 0) continue;
            set.add(inode);
            if (set.size() < this.maxCorruptFilesReturned) continue;
            break;
        }
        return set.toArray(new INode[set.size()]);
    }
}

