/**************************************************
OpengateM - a MAC address authentication system
   module for local work database

Copyright (C) 2011 Opengate Project Team
Written by Yoshiaki Watanabe

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Email: watanaby@is.saga-u.ac.jp
**************************************************/
#include "opengatemd.h"
#include <sqlite3.h>

static int sqliteBusyTimeout=100;  /* value used in sqite3_busy_timeout() */
static sqlite3 *dbMd; /* handle for opengatemd.db */

/*******************************************************************
 read sqlite busy timeout value from conf and set it to static variable
*******************************************************************/
int setupSqliteBusyTimeoutValue(void){

  char *str;
  
  /* if set in conf, use the value. if not, use the above default. */
  str=GetConfValue("SqliteBusyTimeout");
  if(str!=NULL) sqliteBusyTimeout=atoi(str);

  return sqliteBusyTimeout;
}

/********************************************
initialize work db implemented with sqlite
********************************************/
int initWorkDb(void){

  char *pErrMsg=NULL;

  /* SQL CREATE TABLE COMMANDs */
  char *createCmd1="CREATE TABLE IF NOT EXISTS sessionmd "
    "(macAddress TEXT PRIMARY KEY, "
    "userId TEXT, extraId TEXT, openTime INTEGER, checkTime INTEGER, "
    "ruleNumber INTEGER)";
  char *createCmd2="CREATE TABLE IF NOT EXISTS macinfo "
    "(macAddress TEXT PRIMARY KEY ON CONFLICT REPLACE, "
    "detectTime INTEGER, ttl INTEGER, isNat INTEGER)";
  char *createCmd3="CREATE TABLE IF NOT EXISTS macippair "
    "(macAddress TEXT, "
    "ipAddress TEXT, findTime INTEGER)";

  /* setup SqLite3_busy_timeout read from conf */
  SetupSqliteBusyTimeoutValue();

  /* Open sqlite */
  if(sqlite3_open(GetConfValue("SqliteDbMd"),&dbMd)!=SQLITE_OK){
    err_msg("ERR at %s#%d: sqlite3_open",__FILE__,__LINE__);
    terminateProg(0);
  }
  sqlite3_busy_timeout(dbMd, sqliteBusyTimeout);

  /* create table1 */
  if(sqlite3_exec(dbMd, createCmd1, NULL, NULL, &pErrMsg)!=SQLITE_OK){
    err_msg("ERR at %s#%d: sqlite3_exec: %s",__FILE__,__LINE__,pErrMsg);
    terminateProg(0);
  }

  /* create table2 */
  if(sqlite3_exec(dbMd, createCmd2, NULL, NULL, &pErrMsg)!=SQLITE_OK){
    err_msg("ERR at %s#%d: sqlite3_exec: %s",__FILE__,__LINE__,pErrMsg);
    terminateProg(0);
  }

  /* create table3 */
  if(sqlite3_exec(dbMd, createCmd3, NULL, NULL, &pErrMsg)!=SQLITE_OK){
    err_msg("ERR at %s#%d: sqlite3_exec: %s",__FILE__,__LINE__,pErrMsg);
    terminateProg(0);
  }

  return TRUE;
}

/********************************************
finalize work db implemented with sqlite
********************************************/
int finalizeWorkDb(void){

  /* Close db for opengatemd */
  sqlite3_close(dbMd);

  return TRUE;
}

/************************************************************
 insert session-info to work db at starting session
************************************************************/
int insertSessionToWorkDb(char* macAddress, char* userId, char* extraId, 
			int ruleNumber){

  int rc;
  char *pErrMsg=NULL;

  /* SQL INSERT COMMAND, where %x is replaced in snprintf */
  char *insertFormat="INSERT INTO sessionmd "
    "(macAddress, userId, extraId, openTime, checkTime, ruleNumber) "
    "values ('%s','%s','%s', %d, %d, %d)";
  char *insertCmd;
  int resultFlag=TRUE;

  /* Prepare insert command */
  insertCmd=sqlite3_mprintf(insertFormat, macAddress, userId,extraId, 
			    time(NULL), time(NULL), ruleNumber);

  /* Execute insert to sqlite */
  if((rc=sqlite3_exec(dbMd, insertCmd, NULL, NULL, &pErrMsg))!=SQLITE_OK){
    resultFlag=FALSE;
    err_msg("ERR at %s#%d: sqlite3_exec: %s",__FILE__,__LINE__,pErrMsg);
  }

  /*Memory free for sqlite3 string */
  sqlite3_free(insertCmd);

  return resultFlag;
}

/*************************************************************
 update checkTime to now
*************************************************************/
int updateCheckTimeInWorkDb(char* macAddress){

  char *pErrMsg=NULL;

  /* SQL UPDATE COMMAND, where %x is replaced in mprintf */
  char *updateFormat="UPDATE sessionmd "
    "SET checkTime=%d WHERE macAddress='%s'";
  char *updateCmd;
  int resultFlag=TRUE;

  /* prepare command */
  updateCmd=sqlite3_mprintf(updateFormat, time(NULL), macAddress);

  /* execute update to sqlite */
  if(sqlite3_exec(dbMd, updateCmd, NULL, NULL, &pErrMsg)!=SQLITE_OK){
    resultFlag=FALSE;
    err_msg("ERR at %s#%d: sqlite3_exec: %s",__FILE__,__LINE__,pErrMsg);
  }

  /*memory free for sqlite3 string */
  sqlite3_free(updateCmd);

  return resultFlag;
}

/*************************************************************
 delete session-info in work db at stop session
*************************************************************/
int delSessionFromWorkDb(char* macAddress){

  char *pErrMsg=NULL;

  /* SQL DELETE COMMAND, where %x is replaced in mprintf */
  char *deleteFormat="DELETE FROM sessionmd WHERE macAddress='%s'";
  char *deleteCmd;
  int resultFlag=TRUE;

  /* prepare command */
  deleteCmd=sqlite3_mprintf(deleteFormat, macAddress);

  /* execute delete */
  if(sqlite3_exec(dbMd, deleteCmd, NULL, NULL, &pErrMsg)!=SQLITE_OK){
    resultFlag=FALSE;
    err_msg("ERR at %s#%d: sqlite3_exec: %s",__FILE__,__LINE__,pErrMsg);
  }

  /*memory free for sqlite3 string */
  sqlite3_free(deleteCmd);

  return resultFlag;
}

/************************************************
 get session-info from work db  
input = macAddress, output = others
*************************************************/
int getSessionFromWorkDb(char* macAddress, char* userId, char* extraId, 
			 int* openTime, int* checkTime, 
			 int* ruleNumber){

  sqlite3_stmt *stmt;
 
  /* SQL SELECT COMMAND, where %x is replaced in snprintf */
  char *selectFormat="SELECT userId, extraId, openTime, checkTime, "
    "ruleNumber FROM sessionmd WHERE macAddress='%s'";
  char *selectCmd;
  int resultFlag=TRUE;

  /* prepare command string */
  selectCmd=sqlite3_mprintf(selectFormat, macAddress);
  
  /* compile to internal statement */
  if(sqlite3_prepare(dbMd, selectCmd, BUFFMAXLN, &stmt, NULL)!=SQLITE_OK){
    resultFlag=FALSE;
    err_msg("ERR at %s#%d: sqlite3_prepare",__FILE__,__LINE__);

    /* finalize */
    sqlite3_free(selectCmd);
    sqlite3_finalize(stmt);

    return FALSE;
  }

  /* get first record */
  if(sqlite3_step(stmt)==SQLITE_ROW){
    strlcpy(userId, (char*)sqlite3_column_text(stmt, 0), USERMAXLN);
    strlcpy(extraId, (char*)sqlite3_column_text(stmt, 1), USERMAXLN);
    *openTime=(int)sqlite3_column_int(stmt, 2);
    *checkTime=(int)sqlite3_column_int(stmt, 3);
    *ruleNumber=(int)sqlite3_column_int(stmt, 4);
    resultFlag=TRUE;
  }else{
    userId[0]='\0';
    extraId[0]='\0';
    *openTime=0;
    *checkTime=0;
    *ruleNumber=0;
    resultFlag=FALSE;
  }

  /* finalize */
  sqlite3_free(selectCmd);
  sqlite3_finalize(stmt);
  
  return resultFlag;
}


/************************************************
 close sessions that exceed time limit  
 1. select timeover records and close the firewall 
 2. delete the records
if delayed=FALSE, close all sessions without delay
if delayed=TRUE, close sessions that exceed time limit 
*************************************************/
int delUselessSessionsInWorkDb(int delayed){

  char *pErrMsg=NULL;
  int uselessLimitTime;

  /* the session is useless, if it doesn't update after this time */
  uselessLimitTime = time(NULL)-atoi(GetConfValue("UselessTimeout"));
 
  /* if delayed is false, all sessions(before now) are deleted */
  if(!delayed) uselessLimitTime = time(NULL);

  /* SQL SELECT COMMAND, where %x is replaced in snprintf */
  char *selectFormat="SELECT ruleNumber, userId, extraId, "
    "macAddress, openTime FROM sessionmd WHERE checkTime<%d";
  char *deleteFormat="DELETE FROM sessionmd WHERE checkTime<%d";
  char *selectCmd;
  char *deleteCmd;
  int resultFlag=TRUE;

  /* prepare command string for select */
  selectCmd=sqlite3_mprintf(selectFormat, uselessLimitTime);

  /* exec command, callback function = CloseSession() */
  if(sqlite3_exec(dbMd, selectCmd, CloseSession, NULL, &pErrMsg)!=SQLITE_OK){
    resultFlag=FALSE;
    err_msg("ERR at %s#%d: sqlite_exec:%s",__FILE__,__LINE__, pErrMsg);
  }

  /* prepare command string for update */
  deleteCmd=sqlite3_mprintf(deleteFormat, uselessLimitTime);

  /* exec command  */
  if(sqlite3_exec(dbMd, deleteCmd, NULL, NULL, &pErrMsg)!=SQLITE_OK){
    resultFlag=FALSE;
    err_msg("ERR at %s#%d: sqlite_exec:%s",__FILE__,__LINE__, pErrMsg);
  }
    
  /* finalize */
  sqlite3_free(selectCmd);
  sqlite3_free(deleteCmd);

  return resultFlag;
}

/************************************************
get list of sessions from work DB and create hash table of sessions
for opengatemd 
 key=macAddress, value=0 
*************************************************/
int getSessionTableFromWorkDb(DB* sessionTable){
  DBT hashKey;
  DBT hashVal;
  sqlite3_stmt *stmt=NULL;
  int resultFlag=FALSE;
  char macAddress[ADDRMAXLN];
  int zero=0;
 
  /* SQL SELECT COMMAND to get all mac address in session table */
  char *selectCmd="SELECT macAddress FROM sessionmd";

  /* compile to internal statement */
  if(sqlite3_prepare(dbMd, selectCmd, BUFFMAXLN, &stmt, NULL)!=SQLITE_OK){
    err_msg("ERR at %s#%d: sqlite3_prepare",__FILE__,__LINE__);

    /* finalize */
    sqlite3_finalize(stmt);
    return FALSE;
  }

  /* get result rows */
  while(sqlite3_step(stmt)==SQLITE_ROW){
    resultFlag=TRUE;
    strlcpy(macAddress,(char*)sqlite3_column_text(stmt, 0), ADDRMAXLN);

    /* put to hash table */
    hashVal.data = &zero;
    hashVal.size = sizeof(int);    
    hashKey.data = macAddress;
    hashKey.size = strlen(macAddress)+1;
    if(sessionTable->put(sessionTable, &hashKey, &hashVal, 0) == -1) {
      err_msg("ERR at %s#%d: fail to put into hash table",__FILE__,__LINE__);
    }
  }

  /* finalize */
  sqlite3_finalize(stmt);

  return resultFlag;
}


/**********************************
 put out detected mac related info to macinfo table in work db
 it is used at checking nat 
**********************************/
int putMacInfoToWorkDb(char* macAddress, int ttl, int isNat){

  int rc;
  char *pErrMsg=NULL;

  /* SQL INSERT COMMAND, where %x is replaced in snprintf */
  char *insertFormat="INSERT INTO macinfo "
    "(macAddress, detectTime, ttl, isNat) "
    "values ('%s', %d, %d, %d)";
  char *insertCmd;
  int resultFlag=TRUE;

  /* Prepare insert command */
  insertCmd=sqlite3_mprintf(insertFormat, macAddress, time(NULL), ttl, isNat);

  /* Execute insert to sqlite */
  if((rc=sqlite3_exec(dbMd, insertCmd, NULL, NULL, &pErrMsg))!=SQLITE_OK){
    resultFlag=FALSE;
    err_msg("ERR at %s#%d: sqlite3_exec: %s",__FILE__,__LINE__,pErrMsg);
  }

  /*Memory free for sqlite3 string */
  sqlite3_free(insertCmd);

  return resultFlag;
}


/**********************************
 get mac related info from macinfo table in work db 
**********************************/
int getMacInfoFromWorkDb(char* macAddress, char* detectTimeStr, int* pTtl){

  sqlite3_stmt *stmt;

  /* SQL SELECT COMMAND, where %x is replaced in snprintf */
  char *selectFormat="SELECT datetime(detectTime,'unixepoch','localtime'), "
    " ttl FROM macinfo WHERE macAddress='%s' ";
  char *selectCmd;
  int resultFlag=TRUE;

  /* set default value */
  *pTtl=0;
  detectTimeStr[0]='?';
  detectTimeStr[1]='\0';

  /* Prepare select command */
  selectCmd=sqlite3_mprintf(selectFormat, macAddress);

  /* compile to internal statement */
  if(sqlite3_prepare(dbMd, selectCmd, BUFFMAXLN, &stmt, NULL)!=SQLITE_OK){
    resultFlag=FALSE;
    err_msg("ERR at %s#%d: sqlite3_prepare",__FILE__,__LINE__);

    /* finalize */
    sqlite3_free(selectCmd);
    sqlite3_finalize(stmt);
    return FALSE;
  }

  /* get first record */
  if(sqlite3_step(stmt)==SQLITE_ROW){
    strlcpy(detectTimeStr, (char*)sqlite3_column_text(stmt, 0), WORDMAXLN);
    *pTtl=(int)sqlite3_column_int(stmt, 1);
    resultFlag=TRUE;
  }else{
    resultFlag=FALSE;
  }

  /* finalize */
  sqlite3_free(selectCmd);
  sqlite3_finalize(stmt);
  return resultFlag;
}

/************************************
is the rule number active in opengatemd session table
************************************/
int isActiveRuleInWorkDb(int ruleNumber){

  sqlite3_stmt *stmt;
 
  /* SQL SELECT COMMAND, where %x is replaced in snprintf */
  char *selectFormat="SELECT * FROM sessionmd "
    " WHERE ruleNumber=%d";
  char *selectCmd;
  int resultFlag=TRUE;

  /* prepare command string */
  selectCmd=sqlite3_mprintf(selectFormat, ruleNumber);
  
  /* compile to internal statement */
  if(sqlite3_prepare(dbMd, selectCmd, BUFFMAXLN, &stmt, NULL)!=SQLITE_OK){
    resultFlag=FALSE;
    err_msg("ERR at %s#%d: sqlite3_prepare",__FILE__,__LINE__);

    /* finalize */
    sqlite3_free(selectCmd);
    sqlite3_finalize(stmt);
    return FALSE;
  }

  /* get first record */
  if(sqlite3_step(stmt)==SQLITE_ROW) resultFlag=TRUE;
  else resultFlag=FALSE;

  /* finalize */
  sqlite3_free(selectCmd);
  sqlite3_finalize(stmt);
  return resultFlag;
}

/********************************************
Is the MAC-IP pair found in work db
********************************************/
int isFoundMacIpPairInWorkDb(char* macAddress, char* ipAddress){
  sqlite3_stmt *stmt;
 
  /* SQL SELECT COMMAND, where %x is replaced in snprintf */
  char *selectFormat="SELECT * FROM macippair "
    " WHERE macAddress='%s' AND ipAddress='%s'";
  char *selectCmd;
  int resultFlag=TRUE;

  /* prepare command string */
  selectCmd=sqlite3_mprintf(selectFormat, macAddress, ipAddress);
  
  /* compile to internal statement */
  if(sqlite3_prepare(dbMd, selectCmd, BUFFMAXLN, &stmt, NULL)!=SQLITE_OK){
    resultFlag=FALSE;
    err_msg("ERR at %s#%d: sqlite3_prepare",__FILE__,__LINE__);

    /* finalize */
    sqlite3_free(selectCmd);
    sqlite3_finalize(stmt);
    return FALSE;
  }

  /* get first record */
  if(sqlite3_step(stmt)==SQLITE_ROW) resultFlag=TRUE;
  else resultFlag=FALSE;

  /* finalize */
  sqlite3_free(selectCmd);
  sqlite3_finalize(stmt);
  return resultFlag;
}


/********************************************
Insert MAC-IP pair to work db
********************************************/
int putMacIpPairToWorkDb(char* macAddress, char* ipAddress){
  int rc;
  char *pErrMsg=NULL;

  /* SQL INSERT COMMAND, where %x is replaced in snprintf */
  char *insertFormat="INSERT INTO macippair "
    "(macAddress, ipAddress, findTime) "
    "values ('%s','%s', %d)";
  char *insertCmd;
  int resultFlag=TRUE;

  /* Prepare insert command */
  insertCmd=sqlite3_mprintf(insertFormat, macAddress, ipAddress, time(NULL));

  /* Execute insert to sqlite */
  if((rc=sqlite3_exec(dbMd, insertCmd, NULL, NULL, &pErrMsg))!=SQLITE_OK){
    resultFlag=FALSE;
    err_msg("ERR at %s#%d: sqlite3_exec: %s",__FILE__,__LINE__,pErrMsg);
  }

  /*Memory free for sqlite3 string */
  sqlite3_free(insertCmd);

  return resultFlag;
}

/********************************************
Delete the mac-ip pairs in work db
input=macAddress only
********************************************/
int delMacIpPairsInWorkDb(char* macAddress){
  char *pErrMsg=NULL;

  /* SQL DELETE COMMAND, where %x is replaced in mprintf */
  char *deleteFormat="DELETE FROM macippair WHERE macAddress='%s'";
  char *deleteCmd;
  int resultFlag=TRUE;

  /* prepare command */
  deleteCmd=sqlite3_mprintf(deleteFormat, macAddress);

  /* execute delete */
  if(sqlite3_exec(dbMd, deleteCmd, NULL, NULL, &pErrMsg)!=SQLITE_OK){
    resultFlag=FALSE;
    err_msg("ERR at %s#%d: sqlite3_exec: %s",__FILE__,__LINE__,pErrMsg);
  }

  /*memory free for sqlite3 string */
  sqlite3_free(deleteCmd);

  return resultFlag;

}

/********************************************
Delete old mac info in work db
 (detected before 1 month)
********************************************/
int delOldMacInfoInWorkDb(void){
  char *pErrMsg=NULL;

  /* SQL DELETE COMMAND */
  char *deleteCmd="DELETE FROM macinfo WHERE detectTime<strftime('%s','now','-1 month')";
  int resultFlag=TRUE;

  /* execute delete */
  if(sqlite3_exec(dbMd, deleteCmd, NULL, NULL, &pErrMsg)!=SQLITE_OK){
    resultFlag=FALSE;
    err_msg("ERR at %s#%d: sqlite3_exec: %s",__FILE__,__LINE__,pErrMsg);
  }

  return resultFlag;
}

/*********************************************************
 routines for debugging output
*********************************************************/
int SetupSqliteBusyTimeoutValue(void){
  int ret;
  if(debug>1) err_msg("DEBUG:=>setupSqliteBusyTimeoutValue()");
  ret=setupSqliteBusyTimeoutValue();
  if(debug>1) err_msg("DEBUG:(%d)<=setupSqliteBusyTimeoutValue()",ret);
  return ret;
}

int InitWorkDb(void){
  int ret;
  if(debug>1) err_msg("DEBUG:=>initWorkDb( )");
  ret = initWorkDb();
  if(debug>1) err_msg("DEBUG:(%d)<=initWorkDb( )",ret);
  return ret;
}

int FinalizeWorkDb(void){
  int ret;
  if(debug>1) err_msg("DEBUG:=>finalizeWorkDb( )");
  ret = finalizeWorkDb();
  if(debug>1) err_msg("DEBUG:(%d)<=finalizeWorkDb( )",ret);
  return ret;
}

int InsertSessionToWorkDb(char* macAddress, char* userId, char* extraId, 
			 int ruleNumber){
  int ret;
  if(debug>1) err_msg("DEBUG:=>insertSessionToWorkDb(%s,%s,%s,%d)",
		      macAddress, userId, extraId, ruleNumber);
  ret = insertSessionToWorkDb(macAddress, userId, extraId, ruleNumber);
  if(debug>1) err_msg("DEBUG:(%d)<=insertSessionToWorkDb( )",ret);
  return ret;
}

int UpdateCheckTimeInWorkDb(char* macAddress){
  int ret;
  if(debug>1) err_msg("DEBUG:=>updateCheckTimeInWorkDb(%s)", macAddress);
  ret = updateCheckTimeInWorkDb(macAddress);
  if(debug>1) err_msg("DEBUG:(%d)<=updateCheckTimeInWorkDb( )",ret);
  return ret;
}

int DelSessionFromWorkDb(char* macAddress){
  int ret;
  if(debug>1) err_msg("DEBUG:=>delSessionFromWorkDb(%s)", macAddress);
  ret = delSessionFromWorkDb(macAddress);
  if(debug>1) err_msg("DEBUG:(%d)<=delSessionFromWorkDb( )",ret);
  return ret;
}

int GetSessionFromWorkDb(char* macAddress, char* userId, char* extraId, 
			 int* openTime, int* checkTime, int* ruleNumber){
  int ret;
  if(debug>1) err_msg("DEBUG:=>getSessionFromWorkDb(%s)", macAddress);
  ret = getSessionFromWorkDb(macAddress, userId, extraId, openTime, 
			       checkTime, ruleNumber);
  if(debug>1) err_msg("DEBUG:(%d)<=getSessionFromWorkDb(,%s,%s,%d,%d,%d)",
		      ret,userId,extraId,*openTime,*checkTime, *ruleNumber);
  return ret;
}

int DelUselessSessionsInWorkDb(int delayed){
  int ret;
  if(debug>1) err_msg("DEBUG:=>delUselessSessionsInWorkDb(%d)", delayed);
  ret=delUselessSessionsInWorkDb(delayed);
  if(debug>1) err_msg("DEBUG:(%d)<=delUselessSessionsInWorkDb( )",ret);
  return ret;
}

int GetSessionTableFromWorkDb(DB* sessionTable){
  int ret;
  if(debug>1) err_msg("DEBUG:=>getSessionTableFromWorkDb( )");
  ret=getSessionTableFromWorkDb(sessionTable);
  if(debug>1) err_msg("DEBUG:(%d)<=getSessionTableFromWorkDb( )", ret);
  return ret;
}

int PutMacInfoToWorkDb(char* macAddress, int ttl, int isNat){
  int ret;
  if(debug>1) err_msg("DEBUG:=>putMacInfoToWorkDb(%s,%d,%d)",macAddress,ttl,isNat);
  ret=putMacInfoToWorkDb(macAddress,ttl,isNat);
  if(debug>1) err_msg("DEBUG:(%d)<=putMacInfoToWorkDb( )", ret);
  return ret;
}

int IsActiveRuleInWorkDb(int ruleNumber){
  int ret;
  if(debug>1) err_msg("DEBUG:=>isActiveRuleInWorkDb(%d)",ruleNumber);
  ret=isActiveRuleInWorkDb(ruleNumber);
  if(debug>1) err_msg("DEBUG:(%d)<=isActiveRuleInWorkDb( )", ret);
  return ret;
}

int GetMacInfoFromWorkDb(char* macAddress, char* detectTimeStr, int* pTtl){
  int ret;
  if(debug>1) err_msg("DEBUG:=>getMacInfoFromWorkDb(%s)", macAddress);
  ret = getMacInfoFromWorkDb(macAddress, detectTimeStr, pTtl);
  if(debug>1) err_msg("DEBUG:(%d)<=getMacInfoFromWorkDb(,%s,%d)",
		      ret, detectTimeStr, *pTtl);
  return ret;
}

int IsFoundMacIpPairInWorkDb(char* macAddress, char* ipAddress){
  int ret;
  if(debug>1) err_msg("DEBUG:=>isFoundMacIpPairInWorkDb(%s,%s)",
		      macAddress,ipAddress);
  ret=isFoundMacIpPairInWorkDb(macAddress, ipAddress);
  if(debug>1) err_msg("DEBUG:(%d)<=isfoundMacIpPairInWorkDb( )", ret);
  return ret;
}

int PutMacIpPairToWorkDb(char* macAddress, char* ipAddress){
  int ret;
  if(debug>1) err_msg("DEBUG:=>putMacIpPairToWorkDb(%s,%s)",
		      macAddress,ipAddress);
  ret=putMacIpPairToWorkDb(macAddress, ipAddress);
  if(debug>1) err_msg("DEBUG:(%d)<=putMacIpPairtoWorkDb( )", ret);
  return ret;
}


int DelMacIpPairsInWorkDb(char* macAddress){
  int ret;
  if(debug>1) err_msg("DEBUG:=>delMacIpPairsInWorkDb(%s)",
		      macAddress);
  ret=delMacIpPairsInWorkDb(macAddress);
  if(debug>1) err_msg("DEBUG:(%d)<=delMacIpPairsInWorkDb( )", ret);
  return ret;
}

int DelOldMacInfoInWorkDb(void){
  int ret;
  if(debug>1) err_msg("DEBUG:=>delOldMacInfoInWorkDb()");
  ret=delOldMacInfoInWorkDb();
  if(debug>1) err_msg("DEBUG:(%d)<=delOldMacInfoInWorkDb()", ret);
  return ret;
}
