/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

/**
 * ファイルダウンロードアイコン処理に必要な情報を格納しておく
 * データ管理クラスです。
 * fileUD タグから、common/fileDownload.jsp に処理が遷移しますが、
 * その間、DBTableModel が指定の画面で作成されたか、また、view で
 * 指定されたカラムのみを抜き出しているか、スコープは、などの
 * チェックを行います。
 *
 * @og.rev 4.3.0.0 (2008/07/04) 新規追加
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DBLastSql  {
	private String  scope	= null;
	private final String  guikey ;		// 4.3.1.1 (2008/08/23) final化
	private final boolean overflow ;	// 4.3.1.1 (2008/08/23) final化
	private String  tableId	= null;
	private String  clmNames= null;

	/**
	 * 初期情報を含んだ新規オブジェクトを作成します。
	 *
	 * @param    scope     スコープ(session/request)
	 * @param    guikey    画面ID
	 * @param    overflow  検索時にオーバーフローしたかどうか
	 * @param    tableId   テーブルID(DBTableModelの格納キー)
	 */
	public DBLastSql( final String scope,
						final String guikey,
						final boolean overflow,
						final String tableId ) {
		this.scope		= scope;
		this.guikey		= guikey;
		this.overflow	= overflow;
		this.tableId	= tableId;
	}

	/**
	 * DBTableModel を出力するときのカラム名(CVS形式)をセットします。
	 *
	 * ファイルダウンロード時に、view で表示した分だけ抜き出す場合は、
	 * このカラム名を指定します。
	 *
	 * @param    clmNames  画面ID
	 */
	public void setClmNames( final String clmNames ) {
		this.clmNames = clmNames;
	}

	/**
	 *  DBTableModel を出力するときのカラム名(CVS形式)を返します。
	 *
	 * ファイルダウンロード時に、view で表示した分だけ抜き出す場合は、
	 * このカラム名を指定します。
	 *
	 * @return clmNames カラム名(CVS形式)
	 */
	public String getClmNames() { return clmNames; }

	/**
	 * スコープ(session/request)をセットします。
	 *
	 * @param scope スコープ(session/request)
	 */
	public void setScope( final String scope ) { this.scope = scope; }

	/**
	 * スコープ(session/request)が、requestかどうかを返します。
	 *
	 * scope=="request" の場合は、DBTableModel は
	 * メモリに残っていませんので、
	 * １．抜出アイコンを表示しない。
	 * ２．lastSql を利用してフルのDBTableModelを作成しなおす。
	 * 方法が考えられます。
	 *
	 * @return boolean スコープが、requestなら、true
	 */
	public boolean isRequest() { return "request".equals( scope ); }

	/**
	 * 画面IDを返します。
	 *
	 * この画面IDは、ファイルダウンロードアイコンの存在している
	 * 画面と同じ箇所で、作成されたかをチェックする為に使用されます。
	 *
	 * @return guikey 画面ID
	 */
	public String getGuiKey() { return guikey; }

	/**
	 * 内部画面IDと等しいか判定します。
	 *
	 * gamenId != null &amp;&amp; gamenId.equals( lastSql.getGuikey() )
	 * 処理と同等です。
	 *
	 * @param  gamenId 画面ID
	 * @return boolean 引数が null でなく、且つ内部画面キーと同じ場合は、true
	 */
	public boolean isGuiMatch( final String gamenId ) {
		return ( gamenId != null && gamenId.equals( guikey ) );
	}

	/**
	 * 検索時にオーバーフローしたかどうかを返します。
	 *
	 * 検索時にオーバーフローした場合、ファイルダウンロードとして、
	 * １．そのまま、DBTableModel の分だけを抜き出す。
	 * ２．lastSql を利用してフルのDBTableModelを作成しなおす。
	 * 方法が考えられます。
	 *
	 * @return  オーバーフローしたかどうか
	 */
	public boolean isOverflow() { return overflow; }

	/**
	 *  テーブルID(DBTableModelの格納キー)をセットします。
	 *
	 * DBTableModel を取り出すときに使用します。
	 *
	 * @param tableId String テーブルID(DBTableModelの格納キー)
	 */
	public void setTableId( final String tableId ) { this.tableId = tableId; }

	/**
	 *  テーブルID(DBTableModelの格納キー)を返します。
	 *
	 * DBTableModel を取り出すときに使用します。
	 *
	 * @return tableId テーブルID(DBTableModelの格納キー)
	 */
	public String getTableId() { return tableId; }
}
