/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.html;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.fukurou.util.StringUtil;
// import org.opengion.fukurou.util.CSVTokenizer ;
import org.opengion.fukurou.model.Formatter;

import java.util.regex.Pattern;
import java.util.regex.Matcher;

/**
 * [PN],[OYA] などの [] で指定されたカラムで表されたフォーマットデータに対して、
 * DBTableModel オブジェクトを適用して 各カラムに実データを割り当てるオブジェクトです。
 *
 * 特に、[XXXX]に対して、[#XXXX]、[$XXXX]、[$XXXX]などの特殊記号が使用できます。
 * 特殊記号の解釈は、HTMLFormatTextField系とHTMLFormatTable系で異なりますので
 * ご注意ください。
 *
 * @og.rev 3.5.4.0 (2003/11/25) 新規追加
 * @og.group 画面表示
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class TableFormatter {
	/** フォーマットタイプの指定  ヘッダータイプ {@value}  */
//	public static final String TYPE_HEAD  = "head" ;
	/** フォーマットタイプの指定  ボディタイプ {@value}  */
//	public static final String TYPE_BODY  = "body" ;
	/** フォーマットタイプの指定  フッタータイプ {@value}  */
//	public static final String TYPE_FOOT = "foot";

	/** フォーマットタイプの指定の特殊なマーク  */
	public static final String HYBS_ITD_MARKER = "h_itd_marker";
	private static final Pattern ptnKey = Pattern.compile( "[ \t]+</td" ); // 4.3.2.0 (2008/09/10)

	// 3.5.5.2 (2004/04/02) 定数化しておきます。
//	private static final int SYSFORM_KEY_CLM = -1;		// [KEY.カラム名]
//	private static final int SYSFORM_ROWNUM  = -2;		// [I]
//	private static final int SYSFORM_ROW_ID  = -3;		// [ROW.ID]  3.6.0.0 (2004/09/17)

//	private	String			formatType	= null;
	private	FormatterType	formatType	= null;
	private int[]			location	= null;
	private String[]		format		= null;
//	private String[]		systemFormat= null;		// 3.5.5.0 (2004/03/12) [KEY.カラム名] 機能追加
	private String			formatTag	= null;
	private String			rowspan		= " rowspan=\"2\"";
	private String			trTag		= null;
	private boolean			noClass 	= false;
	// 3.5.6.0 (2004/06/18) '!' 値のみ 追加 既存の '$' は、レンデラー
	private char[]			type		= null;		// '#':ラベルのみ  '$':レンデラー '!':値のみ  その他:通常
	private String			usableKey	= null;		// キー情報のカラム文字列
	private int				usableKeyNo	= -1;		// キー情報のカラム番号
	private String			usableList	= "1" ;

	private String			itdBody		= "";		// 3.5.6.0 (2004/06/18) 追加
	private Formatter		formatter	= null;

	/**
	 * フォーマットをセットします。
	 * フォーマットに、&lt;table&gt;を含む場合、TextField扱いなので、フォーマット分割
	 * しません。table を含まず、tr を含む場合は、１行分のデータとして扱う為、
	 * trTag を求めます。
	 * trTag と format との間に、行ヘッダーが入ります。
	 * Tomcat6では、JSPのパース時に、tabやspaceはそのままパースされるため、</td>前
	 * のスペース削除処理も行います。
	 *
	 * @og.rev 4.3.2.0 (2008/09/10) </td>前のスペースを取り消します。
	 *
	 * @param	fmt  [カラム名] 形式のフォーマットデータ
	 */
	public void setFormat( final String fmt ) {
		int tbl = fmt.indexOf( "<table" );
		int str = fmt.indexOf( "<tr" );

		// tr を含み、かつ、tableを含まないか、含んでも tr の後ろにtableがある場合。
		if( str >= 0 && ( tbl < 0 || str < tbl ) ) {
			int end = fmt.indexOf( '>',str );
			formatTag = fmt.substring(end+1);
			trTag = fmt.substring(0,end+1) ;
		}
		else {
			formatTag = fmt;
			trTag     = null;
		}
		// 4.3.2.0 (2008/09/10) </td>前のスペースを取り消す。
		Matcher matcher = ptnKey.matcher( formatTag );
		formatTag = matcher.replaceAll( "</td" );

	}

	/**
	 * フォーマットを取得します。
	 *
	 * @og.rev 3.5.5.8 (2004/05/20) 新規追加
	 *
	 * @return	フォーマットデータ
	 */
	public String getFormat() {
		if( trTag != null ) {
			return trTag + formatTag ;
		}
		else {
			return formatTag ;
		}
	}

	/**
	 * DBTableModelを利用して、フォーマットデータを初期化します。
	 *
	 * @og.rev 3.5.5.0 (2004/03/12) [KEY.カラム名] 機能追加
	 * @og.rev 3.5.5.2 (2004/04/02) [I] で、行番号を作成します。
	 * @og.rev 3.5.6.0 (2004/06/18) '!' 値のみ 追加 既存の '$' は、レンデラー
	 * @og.rev 3.6.0.0 (2004/09/17) [ROW.ID] で、行毎のチェックボックスのIDを返します。
	 *
	 * @param  table DBTableModel
	 */
	public void makeFormat( final DBTableModel table ) {
//		String fmt2 = formatTag.replace( '[',']' );
//		CSVTokenizer token = new CSVTokenizer( fmt2,']',false );
//		int count = token.countTokens() / 2 ;
//		format   = new String[ count+1 ];
//		location = new int[ count ];
//		systemFormat = new String[ count ];	// 3.5.5.0 (2004/03/12)
//		type     = new char[ count ];
//		for( int i=0; i<count; i++ ) {
//			format[i]  = token.nextToken();
//			String clm = token.nextToken();
//			type[i] = clm.charAt(0);
//			if( type[i] == '#' || type[i] == '$' || type[i] == '!' ) {
//				location[i] = table.getColumnNo( clm.substring(1) );
//			}
//			// 3.5.5.0 (2004/03/12) [KEY.カラム名] 機能追加
//			else if( clm.startsWith( "KEY." ) ) {
//				location[i] = SYSFORM_KEY_CLM ;
//				systemFormat[i] = clm.substring(4) ;
//			}
//			// 3.5.5.0 (2004/03/12) [I] 機能追加
//			else if("I".equals( clm ) ) {
//				location[i] = SYSFORM_ROWNUM;
//				systemFormat[i] = null ;
//			}
//			// 3.6.0.0 (2004/09/17) [ROW.ID] 機能追加
//			else if( "ROW.ID".equals( clm ) ) {
//				location[i] = SYSFORM_ROW_ID;
//				systemFormat[i] = null ;
//			}
//			else {
//				location[i] = table.getColumnNo( clm );
//			}
//		}
//		format[count] = token.nextToken();

		formatter = new Formatter( table );
		formatter.setFormat( formatTag );
		location = formatter.getClmNos();
		format   = formatter.getFormat();
		type     = formatter.getType();

		// このフォーマットを使用するかどうかを指定する判定条件の初期設定です。
		if( usableKey != null ) {
			usableKeyNo = table.getColumnNo( usableKey );
		}
	}

	/**
	 * テーブルフォーマットのタイプを指定します。
	 * enum FormatterType で、指定します。
	 *
	 * @og.rev 4.0.0 (2007/05/02) enum 定義に変更
	 *
	 * @param  ftype フォーマットのタイプ
	 */
	public void setFormatType( final FormatterType ftype ) {
		formatType = ftype;
	}

	/**
	 * テーブルフォーマットのタイプを指定します。
	 * head/body/foot/<del>ith/itd</del> で、指定します。
	 *
	 * @og.rev 3.5.4.8 (2004/02/23) フォーマットのタイプに、ith/itd を追加
	 * @og.rev 3.5.6.0 (2004/06/18) フォーマットのタイプの、ith/itd を削除
	 *
	 * @param  ftype フォーマットのタイプ（head/body/foot/<del>ith/itd</del>）
	 */
//	public void setFormatType( final String ftype ) {
//		if( ftype == null || "head/body/foot/".indexOf( ftype ) < 0 ) {
//			String errMsg = "テーブルフォーマットのタイプは、head/body/foot 以外指定できません。";
//			errMsg = errMsg + "[" + ftype + "]";
//			throw new HybsSystemException( errMsg );
//		}
//		formatType = ftype;
//	}

	/**
	 * 【TAG】このフォーマットのタイプを返します。
	 *
	 * @og.tag このフォーマットのタイプを返します。
	 *
	 * @og.rev 4.0.0 (2007/05/02) enum 定義に変更
	 *
	 * @return	FormatterType このフォーマットのタイプを返します。
	 */
	public FormatterType getFormatType() {
		return formatType;
	}

	/**
	 * 【TAG】このフォーマットのタイプを返します。
	 *
	 * @og.tag
	 * タイプは、"head/body/foot/<del>ith/itd</del>" の中から、指定します。
	 *
	 * @return	このフォーマットのタイプを返します。
	 */
//	public String getFormatType() {
//		return formatType;
//	}

	/**
	 * テーブルの rowspan 属性をセットします。
	 * rowspan は、ヘッダー部のフォーマットの行数です。デフォルトは ２行 です。
	 * 設定は、"2" などの、数字部のみをセットします。
	 *
	 * @param  rowspan 属性
	 */
	public void setRowspan( final String rowspan ) {
		if( rowspan == null || rowspan.length() == 0 || rowspan.equals( "1" ) ) {
			this.rowspan = "";
		}
		else {
			this.rowspan = " rowspan=\"" + rowspan + "\"";
		}
	}

	/**
	 * 設定された rowspan を返します。
	 * これは、フォーマットの段組の数を取り出します。
	 * 文字列としては、rowspan="2" という形で取り出します。
	 *
	 * @return フォーマット文字列
	 */
	public String getRowspan() {
		return rowspan;
	}

	/**
	 * ロケーション番号のサイズを返します。
	 * フォーム位置番号は、0 から getLocationSize()-1 までの数字を指定します。
	 * ロケーションサイズは、aaa[ABC]bbb[DEF]ccc[GHI]ddd となっている場合、
	 * aaa , bbb , ccc , ddd は、フォーマットで、サイズは４。
	 * ABC , DEF , GHI に対応するカラム番号がロケーションで、サイズは３。
	 * このメソッドで返すのは、ロケーション番号（３）の方です。
	 *
	 * @return  ロケーション番号のサイズ
	 */
	public int getLocationSize() {
		return location.length;
	}

	/**
	 * カラムのロケーション番号を返します。
	 * 引数は、0 から、getLocationSize()-1 までの数で指定します。
	 * 指定の位置の、フォーマットのカラム名に対応するロケーション番号
	 * を返します。
	 *
	 * @param no フォーム位置番号
	 * @return ロケーション番号
	 */
	public int getLocation( final int no ) {
		return location[no];
	}

	/**
	 * フォーマット文字列を返します。
	 * 引数は、0 から、getLocationSize() までの数で指定します。
	 * 指定のフォーマットが、aaa[ABC]bbb[DEF]ccc[GHI]ddd となっている場合、
	 * aaa , bbb , ccc , ddd を引数 0 , 1 , 2 , 3 で返します。
	 *
	 * @param no フォーム位置番号
	 * @return フォーマット文字列
	 */
	public String getFormat( final int no ) {
		return format[no];
	}

	/**
	 * システムフォーマット文字列を返します。
	 * システムフォーマット文字列は、[KEY.カラム名] などの特殊記号で指定された
	 * カラム名の事で、location には、マイナスの値が設定されます。
	 * マイナスの値に応じて、処理を変えることが出来ます。
	 *
	 * [KEY.カラム名] : 行番号付きカラム名
	 * [I]            : 行番号
	 * [ROW.ID]       : 行毎のチェックボックスのID
	 * [ROW.JSON]     : 行毎の全データのJavaScriptオブジェクト形式
	 *
	 * @og.rev 3.5.5.0 (2004/03/12) [KEY.カラム名] 機能追加
	 * @og.rev 3.5.5.2 (2004/04/02) [I] で、行番号を作成します。
	 * @og.rev 3.6.0.0 (2004/09/17) [ROW.ID] で、行毎のチェックボックスのIDを返します。
	 * @og.rev 4.0.0 (2007/05/02) Formatter を使用するように変更
	 *
	 * @param row  int 行番号
	 * @param loc  int 位置番号
	 * @return フォーマット文字列
	 */
	public String getSystemFormat( final int row,final int loc ) {
		if( loc == Formatter.SYS_ROWNUM ) {
			return String.valueOf( row );
		}
		else if( loc == Formatter.SYS_JSON ) {
			return formatter.getJson( row );
		}

		String errMsg = "システムフォーマットは、下記の形式しか使用できません。[" + loc + "]" + HybsSystem.CR
				+ "  : [KEY.カラム名] : 行番号付きカラム名" + HybsSystem.CR
				+ "  : [I]            : 行番号" + HybsSystem.CR
				+ "  : [ROW.ID]       : 行毎のチェックボックスのID" + HybsSystem.CR
				+ "  : [ROW.JSON]     : 行毎の全データのJavaScriptオブジェクト形式" ;
		throw new HybsSystemException( errMsg );
	}

//	public String getSystemFormat( final int row,final int no ) {
//		if( location[no] == SYSFORM_KEY_CLM ) {
//			return systemFormat[no] + HybsSystem.JOINT_STRING + row ;
//		}
//		else if( location[no] == SYSFORM_ROWNUM ) {
//			return String.valueOf( row );
//		}
//		else if( location[no] == SYSFORM_ROW_ID ) {
//			return HybsSystem.ROW_ID_KEY + row;
//		}
//
//		String errMsg = "システムフォーマットは、下記の形式しか使用できません。[" + location[no] + "]" + HybsSystem.CR
//				+ "  " + SYSFORM_KEY_CLM + ": [KEY.カラム名] カラムID(接続記号)行番号 というフィールドキーを作成"
//				+ "  " + SYSFORM_ROWNUM  + ": [I] 行番号を作成" ;
//		throw new HybsSystemException( errMsg );
//	}

	/**
	 * タイプ文字列を返します。
	 * タイプとは、[XXX] の記述で、[#XXX] は、XXXカラムのラベルを、[$XXX]は、XXXカラムの
	 * レンデラーを、[!XXX} は、値のみ取り出す指定を行います。
	 * 主に、TextField系のフォーマットとTable系では、意味合いが異なりますので、
	 * ご注意ください。
	 *
	 * @param no フォーム位置番号
	 * @return タイプ文字列 '#':ラベルのみ  '$':レンデラー '!':値のみ  その他:通常
	 */
	public char getType( final int no ) {
		return type[no];
	}

	/**
	 * 設定された フォーマットの trタグを返します。
	 * これは、trタグにclass属性他の設定がされていた場合に、変換後の
	 * 文字列にも反映させる為に必要です。
	 *
	 * @return フォーマットの trタグ
	 */
	public String getTrTag() {
		if( trTag == null ) { return ""; }

		return trTag ;
	}

	/**
	 * カラムのクラス名（<de>VERCHAR2 , NUMBER</del>X,S9 など）のセットを行うかどうか指定します。
	 *
	 * "true" で、クラス属性を設定しません。これは、ＣＳＳファイルに書かれている属性を
	 * 使用しないことを意味します。
	 * デフォルトは、"false" です。
	 *
	 * @param	flag クラス名使用の有無（true:使用しない/false:使用する。）
	 */
	public void setNoClass( final String flag ) {
		noClass = StringUtil.nval( flag,noClass );
	}

	/**
	 * カラムのクラス名（<de>VERCHAR2 , NUMBER</del>X,S9 など）のセットを行うかどうか取得します。
	 *
	 * "true" で、クラス属性を設定しません。これは、ＣＳＳファイルに書かれている属性を
	 * 使用しないことを意味します。
	 * デフォルトは、"false" です。
	 *
	 * @return	クラス名使用の有無（true:使用しない/false:使用する。）
	 */
	public boolean isNoClass() {
		return noClass;
	}

	/**
	 * フォーマットの使用可否を判断するキーとなるカラム名を指定します。
	 *
	 * キーが、usableList に含まれる場合は、このフォームを使用できます。
	 * キー（カラム名）が指定されない場合は、常に使用されます。
	 * ※ 現時点では、BODYタイプのみ使用しています。
	 *
	 * @param  key フォーマットの使用可否を判断するカラム名
	 */
	public void setUsableKey( final String key ) {
		usableKey = key;
	}

	/**
	 *  フォーマットの使用可否を判断する文字列リストを指定します。
	 *
	 * キーが、この文字列リスト中に存在する場合は、このフォームを使用できます。
	 * この文字列リストは、固定な文字列です。{&64;XXXX}は使用できますが、[XXXX]は
	 * 使用できません。
	 * 初期値は、"1" です。
	 * ※ 現時点では、BODYタイプのみ使用しています。
	 *
	 * @param  list フォーマットの使用可否を判断する文字列リスト
	 * @see TableFormatter#isUse( int,DBTableModel )
	 */
	public void setUsableList( final String list ) {
		if( list != null ) {
			usableList = list;
		}
	}

	/**
	 * このフォーマットを使用するかどうかの問い合わせを返します。
	 *
	 * "true" で、使用します。setUsableKey( String ) で、指定された
	 * カラム名の値が、setUsableList( String ) で指定された文字列に含まれていれば、
	 * 使用します。カラム名がセットされない場合は、デフォルト値（"true")が使用されます。
	 * ※ 現時点では、BODYタイプのみ使用しています。
	 * カラムのデータに、不正なスペースが入る場合を想定して、trim() しています。
	 * よって、usableList の値にスペースは使用できません。
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) 判定評価用カラムの値を trim() します。
	 *
	 * @param  row 行番号
	 * @param  table DBTableModel
	 * @return	このフォームを使用するかどうか（true:使用する/false:使用しない）
	 * @see TableFormatter#setUsableKey( String )
	 * @see TableFormatter#setUsableList( String )
	 */
	public boolean isUse( final int row, final DBTableModel table ) {
		if( usableKeyNo < 0 ) { return true; }
		String val = table.getValue( row,usableKeyNo ).trim();
		return ( val.length() == 0 ? false : usableList.indexOf( val ) >= 0 );
	}

	/**
	 *  itdフォーマット文字列を設定します。
	 *
	 * itd ボディ部の文字列を指定します。
	 * itd ボディは、繰り返し処理を行います。これを、上位のボディ文字列の中の
	 * HYBS_ITD_MARKER 文字列 と置き換えます。
	 *
	 * @og.rev 3.5.6.0 (2004/06/18) itdフォーマット文字列の取り込み
	 *
	 * @param  itd itdフォーマットの文字列
	 */
	public void setItdBody( final String itd ) {
		if( itd != null ) {
			itdBody = itd;
		}
	}

	/**
	 *  itdフォーマット文字列を取得します。
	 *
	 * itd ボディ部の文字列を取得します。
	 * itd ボディは、繰り返し処理を行います。これを、上位のボディ文字列の中の
	 * HYBS_ITD_MARKER 文字列 と置き換えます。
	 *
	 * @og.rev 3.5.6.0 (2004/06/18) itdフォーマット文字列の取り込み
	 *
	 * @return  itd itdフォーマットの文字列
	 */
	public String getItdBody() {
		return itdBody;
	}
}
