/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.report;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.fukurou.util.LogWriter;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.ApplicationInfo;
import org.opengion.fukurou.db.DBUtil;

import org.opengion.fukurou.mail.MailTX ;

/**
 * 帳票要求テーブル（GE50)と、エラーテーブル(GE56)に対して、データ書込みを行います。
 *
 * このクラスでは、４つの機能を実装しています。
 *
 * ●要求番号採番機能： makeYkno()
 *   新たな要求番号を採番します(シーケンスより取得)
 *   採番後、内部の要求番号(ykno)もこの値に再セットされます。
 * ●帳票要求テーブルデータセット機能： insertGE50( final String fgkan )
 *   内部変数にセットされている値を利用して、データ登録を行います。
 *   繰返し登録したい場合は、変更する値のみを、内部変数にセットしてから、
 *   このメソッドを呼び出してください。
 *   なお、完成フラグ(fgkan)は、内部変数に持っていないため、外部から指定してください。
 * ●完成フラグ設定機能： updateGE50( final String fgkan )
 *   内部変数にセットされている、systemId と ykno を利用して、データ登録を行います。
 *   なお、完成フラグ(fgkan)は、内部変数に持っていないため、外部から指定してください。
 * ●エラーメッセージ登録機能： insertErrorGE56( final String inErrMsg )
 *   エラー発生時に、エラーメッセージを登録します。
 *   内部変数にセットされている、systemId と ykno を利用して、データ登録を行います。
 *   なお、このメソッドは、要求テーブルの完成フラグは、なにもしませんので、
 *   完成フラグ設定機能と併用してください。
 *   また、システムパラメータ の COMMON_MAIL_SERVER(メールサーバー)と
 *   ERROR_MAIL_FROM_USER(エラーメール発信元)と、ERROR_MAIL_TO_USERS(エラーメール受信者)
 *   がすべて設定されている場合に、エラー情報のメール送信を行います。
 *
 * @og.rev 3.8.0.0 (2005/06/07) 新規追加
 * @og.group 帳票システム
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class GE50Access {
	// 完成フラグのフラグ定義
	public static final String FG_SET  = "1";	// 登録
	public static final String FG_OK   = "2";	// 済み
	public static final String FG_RUN  = "3";	// 実行中
	public static final String FG_HAND = "4";	// 手動
	public static final String FG_WAIT = "5";	// 印刷待ち
	public static final String FG_DBIN = "6";	// 取込済	Ver 3.8.0.0
	public static final String FG_ERR1 = "7";	// デーモンエラー
	public static final String FG_ERR2 = "8";	// アプリエラー

	// GE50 の要求番号のシーケンスを取得します。
	private static final String GE50_SEQ = "select GE50_SEQUENCE.NEXTVAL from DUAL" ;

	// GE50 に要求データを設定します。
	private static final String GE50_INSERT	=
		"INSERT INTO GE50" +
		" (SYSTEM_ID,YKNO,LISTID,JOKEN,FGKAN,OUTDIR,OUTFILE,COMMENTS," +
		"  FGJ,DYSET,DYUPD,USRSET,USRUPD,PRGUPD,DMN_NAME,DMN_HOST )" +
		" VALUES" +
		"  (?,?,?,?,?,?,?,SUBSTRB(?,1,100)," +
		"  '1',?,?,?,?,?,?,? )" ;

	// GE50 の完成フラグを設定するSQL文です。
	// 3.8.0.4 (2005/08/08) PRGUPD も更新するように変更
	private static final String GE50_UPDATE	=
		"UPDATE GE50 SET FGKAN = ? , DYUPD = ? , DMN_NAME = ? , DMN_HOST = ? " +
		" WHERE FGJ = '1'" +
		" AND   SYSTEM_ID = ?" +
		" AND   YKNO  = ?" ;

	// GE56 のエラー情報を書き込むためのSQL文です。
	private static final String GE56_ERRMSG	=
		"INSERT INTO GE56" +
		"        ( FGJ,SYSTEM_ID,YKNO,ERRMSG,DYSET,DYUPD,USRSET,USRUPD,PRGUPD )" +
		" VALUES ( '1',?        ,?   ,?     ,?    ,?    ,?     ,?     ,?      )" ;

	/** コネクションにアプリケーション情報を追記するかどうか指定 */
	public static final boolean USE_DB_APPLICATION_INFO  = HybsSystem.sysBool( "USE_DB_APPLICATION_INFO" ) ;

	// 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfo オブジェクトを設定
	private final ApplicationInfo appInfo;

	private final String USRSET ;
	private final String PRGUPD ;
	private final String DMN_NAME ;		// 3.8.5.0 (2006/03/02)

	private String systemId	= null;
	private String ykno		= null;
	private String listid	= null;
	private String joken	= null;
	private String outdir	= null;
	private String outfile	= null;
	private String comments	= null;

	/**
	 * ユーザーとプログラムIDを指定して、オブジェクトを構築します。
	 * このオブジェクトを構築した時刻を、DYSET、DYUPD にセットします。
	 * つまり、このオブジェクトで登録する一連のデータは、すべて同一時刻での登録になります。
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfo オブジェクトを設定
	 *
	 * @param usrset  String ユーザー
	 * @param prgupd  String プログラムID
	 * @param dmnName String 更新デーモン名
	 */
	public GE50Access( final String usrset, final String prgupd ,final String dmnName) {
		USRSET   = substr( usrset, 10, "UNNONE" ) ;
		PRGUPD   = substr( prgupd, 10, "UNNONE" ) ;
		DMN_NAME = substr( dmnName,50, "UNNONE" ) ;

		// 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfo オブジェクトを設定
		if( USE_DB_APPLICATION_INFO ) {
			appInfo = new ApplicationInfo();
			// ユーザーID,IPアドレス,ホスト名
			appInfo.setClientInfo( USRSET,HybsSystem.HOST_ADRS,HybsSystem.HOST_NAME );
			// 画面ID,操作,プログラムID
			appInfo.setModuleInfo( "GE50Access",null,DMN_NAME );
		}
		else {
			appInfo = null;
		}
	}

	/**
	 * systemId をセットします。
	 *
	 * @param systemId システムID
	 */
	public void setSystemId	( final String systemId ) { this.systemId = systemId; }

	/**
	 * systemId を取得します。
	 *
	 * @return systemId システムID
	 */
	public String getSystemId() { return systemId; }

	/**
	 * listid をセットします。
	 *
	 * @param listid 帳票ID
	 */
	public void setListId( final String listid ) { this.listid = listid; }

	/**
	 * joken をセットします。
	 *
	 * @param joken 振分条件
	 */
	public void setJoken( final String joken ) { this.joken = joken; }

	/**
	 * outdir をセットします。
	 *
	 * @param outdir 出力DIR
	 */
	public void setOutDir( final String outdir ) { this.outdir = outdir; }

	/**
	 * outfile をセットします。
	 *
	 * @param outfile 出力ファイル
	 */
	public void setOutFile( final String outfile ) { this.outfile = outfile; }

	/**
	 * comments をセットします。
	 *
	 * @param comments コメント
	 */
	public void setComments( final String comments ) { this.comments = comments; }

	/**
	 * ykno をセットします。
	 *
	 * @param ykno 要求番号
	 */
	public void setYkno( final String ykno ) { this.ykno = ykno; }

	/**
	 * ykno を取得します。
	 *
	 * @return ykno 要求番号
	 */
	public String getYkno() { return ykno; }

	/**
	 * 新たな要求番号を採番します(シーケンスより取得)
	 * 採番後、内部の要求番号(ykno)もこの値に再セットされます。
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfo オブジェクトを設定
	 *
	 * @return String 要求番号
	 */
	public String makeYkno() {
		try {
			String[][] vals = DBUtil.dbExecute( GE50_SEQ,null,appInfo );	// 3.8.7.0 (2006/12/15)
			ykno = vals[0][0].trim();
		}
		catch( RuntimeException ex ) {
			String errMsg = "要求番号の採番に失敗しました。"
						+ ex.getMessage();
			LogWriter.log( errMsg );
			LogWriter.log( ex );
		}
		return ykno ;
	}

	/**
	 * 帳票要求テーブルにデータをセットします。
	 * 内部変数にセットされている値を利用して、データ登録を行います。
	 * 繰返し登録したい場合は、変更する値のみを、内部変数にセットしてから、
	 * このメソッドを呼び出してください。
	 * なお、完成フラグ(fgkan)は、内部変数に持っていないため、外部から指定してください。
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfo オブジェクトを設定
	 *
	 * @param fgkan 完成フラグ
	 */
	public void insertGE50( final String fgkan ) {
		String DYSET = HybsSystem.getDate( "yyyyMMddHHmmss" ) ;	// 3.8.5.0 (2006/03/02)

		// GE50_INSERT の引数
		String[] args = new String[] {
							systemId,	// SYSTEM_ID
							ykno ,		// YKNO
							StringUtil.nval( listid,"NO_LIST" ),		// LISTID
							StringUtil.nval( joken,"NO_JOKEN" ),		// JOKEN
							fgkan,		// FGKAN
							StringUtil.nval( outdir,"" ),		// OUTDIR
							StringUtil.nval( outfile,"" ),		// OUTFILE
							StringUtil.nval( comments,"" ),		// COMMENTS
							DYSET,					// DYSET
							DYSET,					// DYUPD
							USRSET,					// USRSET
							USRSET,					// USRUPD
							PRGUPD,					// PRGUPD
							DMN_NAME,				// DMN_NAME
							HybsSystem.HOST_NAME	// DMN_HOST
						} ;

		try {
			DBUtil.dbExecute( GE50_INSERT,args,appInfo );	// 3.8.7.0 (2006/12/15)
		}
		catch( RuntimeException ex ) {
			String errMsg = "帳票要求テーブルデータセットに失敗しました。"
						+ ex.getMessage();
			LogWriter.log( errMsg );
			LogWriter.log( ex );
		}
	}

	/**
	 * 処理終了後に完成フラグを設定します。
	 * 内部変数にセットされている、systemId と ykno を利用して、データ登録を行います。
	 * なお、完成フラグ(fgkan)は、内部変数に持っていないため、外部から指定してください。
	 *
	 * @og.rev 3.8.0.4 (2005/08/08) PRGUPDを追加
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfo オブジェクトを設定
	 *
	 * @param fgkan 完成フラグ
	 */
	public void updateGE50( final String fgkan ) {
		String DYSET = HybsSystem.getDate( "yyyyMMddHHmmss" ) ;	// 3.8.5.0 (2006/03/02)

		String[] args = new String[] {
							fgkan,						// FGKAN
							DYSET,						// DYUPD
							DMN_NAME,					// DMN_NAME
							HybsSystem.HOST_NAME,		// DMN_HOST
							systemId,					// SYSTEM_ID
							ykno						// YKNO
						} ;

		try {
			DBUtil.dbExecute( GE50_UPDATE,args,appInfo );	// 3.8.7.0 (2006/12/15)
		}
		catch( RuntimeException ex ) {
			String errMsg = "完成フラグ設定に失敗しました。"
						+ ex.getMessage();
			LogWriter.log( errMsg );
			LogWriter.log( ex );
		}
	}

	/**
	 * エラー発生時に、エラーメッセージを登録します。
	 * 内部変数にセットされている、systemId と ykno を利用して、データ登録を行います。
	 *
	 * @og.rev 3.8.5.3 (2006/06/30) エラーメッセージを 4000Byte以下にする。
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfo オブジェクトを設定
	 *
	 * @param inErrMsg String エラーメッセージ
	 */
	public void insertErrorGE56( final String inErrMsg ) {

		String outErrMsg = (inErrMsg == null) ? "" : inErrMsg.trim();
		if( outErrMsg.length() > 2000 ) {
			String code = HybsSystem.sys( "DB_ENCODE" );
			byte[] byteValue = StringUtil.makeByte( outErrMsg,code );
			if( byteValue.length > 4000 ) {
				outErrMsg = StringUtil.makeString( byteValue,0,4000,code );
			}
		}
		String DYSET = HybsSystem.getDate( "yyyyMMddHHmmss" ) ;	// 3.8.5.0 (2006/03/02)

		String[] args = new String[] {
								systemId,
								ykno,
								outErrMsg ,		// ERRMSG
								DYSET,			// DYSET
								DYSET,			// DYUPD
								USRSET,			// USRSET
								USRSET,			// USRUPD
								PRGUPD };		// PRGUPD

		try {
			DBUtil.dbExecute( GE56_ERRMSG,args,appInfo );	// 3.8.7.0 (2006/12/15)
		}
		catch( RuntimeException ex ) {
			String errMsg = "エラーメッセージテーブルセットに失敗しました。" + HybsSystem.CR
						+ "SYSTEM_ID=[" + systemId + "] , YKNO=[" + ykno + "]" + HybsSystem.CR
						+ ex.getMessage();
			LogWriter.log( errMsg );
			LogWriter.log( ex );
		}
		sendMail( outErrMsg );		// 3.8.0.4 (2005/08/08)
	}

	/**
	 * substring を補完する簡易メソッド
	 *
	 * substring で、文字をカットする場合、文字列長が、カットする文字数より
	 * 長い必要があります。ここでは、最大長に制限をかけることが目的なので
	 * 入力文字長が指定文字数より小さいケースがあります。そのチェックを
	 * 簡易的に実行できるように、このメソッドで処理します。
	 *
	 * @param in String 入力文字
	 * @param len int 最大文字数
	 * @param defVal String NULL 時の初期値
	 * @return 指定の文字数でカットされた新しい文字列
	 */
	private String substr( final String in, final int len, final String defVal ) {
		String rtn = in;
		if( in == null ) {
			rtn = defVal;
		}
		else if( in.length() > len ) {
			rtn = in.substring( 1,len );
		}

		return rtn ;
	}

	/**
	 * エラー情報のメール送信を行います。
	 * エラーメールは、システムパラメータ の COMMON_MAIL_SERVER(メールサーバー)と
	 * ERROR_MAIL_FROM_USER(エラーメール発信元)と、ERROR_MAIL_TO_USERS(エラーメール受信者)
	 * がすべて設定されている場合に、送信されます。
	 *
	 * @og.rev 3.8.0.4 (2005/08/08) 新規追加
	 *
	 * @param inErrMsg String エラーメッセージ
	 */
	private void sendMail( final String inErrMsg ) {

		String   host = HybsSystem.sys( "COMMON_MAIL_SERVER" );
		String   from = HybsSystem.sys( "ERROR_MAIL_FROM_USER" );
		String[] to = StringUtil.csv2Array( HybsSystem.sys( "ERROR_MAIL_TO_USERS" ) );

		if( host != null && from != null && to.length > 0 ) {
			// 3.8.5.0 (2006/03/02) DMN_NAME、DMN_HOST 追加
			String subject = "SYSTEM_ID=[" + systemId + "] , YKNO=[" + ykno + "] , "
						   + "DMN_NAME=[" + DMN_NAME + "] , DMN_HOST=[" + HybsSystem.HOST_NAME + "]" ;
			try {
				MailTX tx = new MailTX( host );
//				tx.setHost( host );
				tx.setFrom( from );
				tx.setTo( to );
				tx.setSubject( "帳票エラー：" + subject );
				tx.setMessage( inErrMsg );
				tx.sendmail();
			}
			catch( Throwable ex ) {
				String errMsg = "エラー時メール送信に失敗しました。" + HybsSystem.CR
							+ subject + HybsSystem.CR
							+ ex.getMessage();
				LogWriter.log( errMsg );
				LogWriter.log( ex );
			}
		}
	}
}
