/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.resource;

import java.text.MessageFormat;

import org.opengion.hayabusa.common.HybsSystem ;
import org.opengion.fukurou.util.StringUtil ;

/**
 * systemId , lang に対応したラベルデータを作成します。
 *
 * ラベルデータは、項目(CLM)に対して、各種ラベル情報を持っています。
 * 従来のラベルは、表示名称として、一種類しか持っていませんでしたが、
 * ラベルデータは、３種類の名称と、１種類の概要を持っています。
 *
 *   label       : 名称(長)      従来のラベルと同じで定義された文字そのものです。
 *   shortLabel  : 名称(HTML短)  概要説明をバルーン表示する短い名称です。
 *   longLabel   : 名称(HTML長)  概要説明をバルーン表示する長い名称です。
 *   description : 概要説明      カラムの説明やバルーンに使用します。
 *
 * 名称(HTML長)は、従来の表示名称にあたります。これは、一般的なラベルとして
 * 使用されます。名称(HTML短)は、テーブル一覧のヘッダーの様に、特殊なケースで、
 * 簡略化された名称を使用するときに利用されます。この切り替えは、自動で判断されます。
 * 名称(HTML短)に、なにも設定されていない場合は、名称(HTML長)が自動的に使用されますので
 * 初期データ移行時には、そのまま、通常時もテーブルヘッダー時も同じ文字列が
 * 使用されます。
 * ただし、一覧表示のうち、EXCEL出力などのデータには、名称(長)が使用されます。
 * これは、名称(HTML短)や名称(HTML長)は、Tips表示を行う為のHTML構文を採用している為
 * テキスト等に出力するには不適切だからです。また、EXCEL等のツールでは、ラベル名が
 * 長くてもセル幅等で調整できる為、簡略化された名称よりも正式名称で出力します。
 *
 * ラベルデータを作成する場合は、同一ラベルで、作成区分(KBSAKU)違いの場合は、
 * 最も大きな作成区分を持つコードを使用します。
 * 作成区分(KBSAKU)は、0:システム予約、1:アプリ設定、2:ユーザー設定 という具合に
 * カスタマイズの度合いに応じて大きな数字を割り当てることで、キー情報を上書き修正
 * することが可能になります。（削除することは出来ません。）
 *
 * @og.rev 4.0.0 (2004/12/31) 新規作成
 * @og.group リソース管理
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class LabelData implements LabelInterface {

	/** 内部データのカラム番号 {@value}	*/
	public static final int CLM			= 0 ;
	/** 内部データのカラム番号 {@value}	*/
	public static final int SNAME		= 1 ;
	/** 内部データのカラム番号 {@value}	*/
	public static final int LNAME		= 2 ;
	/** 内部データのカラム番号 {@value}	*/
	public static final int DESCRIPTION	= 3 ;
	/** 内部データのカラム数 {@value}	*/
	public static final int DATA_SIZE	= 4 ;
	/** リソース読み込みのために一時利用 4.3.5.7 (2009/03/22) */
	public static final int FG_LOAD		= 4 ;

	private final String	key			;		// 項目
	private final String	label		;		// 名称（長）
	private final String	shortLabel	;		// 名称（HTML短）
	private final String	longLabel	;		// 名称（HTML長）
	private final String	description	;		// 概要説明
	private final boolean	official	;		// リソースＤＢから作成されたかどうか
	private final boolean	isFormat	;		// メッセージフォーマット対象かどうか 4.0.0.0 (2007/10/17)
	private final String	rawShortLabel;		// 名称(未変換短) 4.3.8.0 (2009/08/01)
	private final boolean	isFormatDesc;		// 概要がフォーマット対象か 4.3.7.6 (2009/07/15)

	/**
	 * null LabelData オブジェクトを作成します。
	 * このオブジェクトは、ＤＢリソース上に存在しない場合に使用される
	 * null 情報を表す、LabelData オブジェクトです。
	 *
	 * @param inkey String キー情報
	 */
	LabelData( final String inkey ) {
		key			= inkey.intern() ;
		label		= key ;		// 名称（長）
		shortLabel	= key ;		// 名称（HTML短）
		longLabel	= key ;		// 名称（HTML長）
		description	= ""  ;		// 概要説明
		official	= false;	// 非正式
		isFormat 	= false;	// 非フォーマット 4.0.0.0 (2007/10/17)
		rawShortLabel = key;	// 名称(未変換短) 4.3.8.0 (2009/08/01)
		isFormatDesc = false;	// 概要フォーマット 4.3.7.6 (2009/07/15)
	}

	/**
	 * 配列文字列のデータを元に、LabelDataオブジェクトを構築します。
	 * このコンストラクタは、他のパッケージから呼び出せないように、
	 * パッケージプライベートにしておきます。
	 * このコンストラクタは、ＤＢリソースファイルを想定しています。
	 *
	 * @param data String[] CLM,SNAME,LNAME,DESCRIPTION
	 *
	 */
	LabelData( final String[] data ) {
		key			= data[CLM].intern() ;				// 項目
		label		= StringUtil.nval2( data[LNAME],"" ) ;			// 名称（HTML長）
		description	= data[DESCRIPTION] ;	// 概要説明
		official	= true;					// 正式
		isFormat = ( label.indexOf( '{' ) >= 0 ) ;	// 4.0.0.0 (2007/10/17)
		isFormatDesc = ( description.indexOf( '{' ) >= 0 ); // 4.3.7.6 (2009/07/15)

		if( description == null || description.length() == 0 ) {
			// 概要説明がない場合は、そのままラベルを使用する。
			longLabel = label;
		}
		else {
			// 概要説明がある場合は、ツールチップにDESCRIPTIONを表示する。
			longLabel = "<span title=\""
							+ StringUtil.htmlFilter( description )
							+ "\">"
							+ label
							+ "</span>" ;
		}

		String sname = data[SNAME];		// 名称（HTML短）
		if( sname == null || sname.length() == 0 ) {
			// SNAME がない場合は、longLabel を使用する。
			shortLabel = longLabel;
			rawShortLabel = longLabel; // 4.3.8.0 (2009/08/01)
		}
		else {
			// SNAME が存在する場合、ツールチップにDESCRIPTIONかlabelを使用する。
			String title = ( description == null || description.length() == 0 ) ? label : description ;
			shortLabel = "<span title=\""
							+ StringUtil.htmlFilter( title )
							+ "\">"
							+ sname
							+ "</span>" ;
			rawShortLabel = sname; // 4.3.8.0 (2009/08/01)
		}
	}

	/**
	 * ラベルオブジェクトのキーを返します。
	 *
	 * @return key ラベルオブジェクトのキー
	 *
	 */
	public String getKey() { return key; }

	/**
	 * ラベルオブジェクトの名称を返します。
	 * これは、ＤＢ上の LNAME(名称(長))に該当します。
	 *
	 * @return label ラベルオブジェクトの名称(短)
	 *
	 */
	public String getLabel() { return label; }

	/**
	 * ラベルオブジェクトの名称(短)を返します。
	 * 概要説明がない場合でかつＤＢ上のSNAMEが未設定の場合は、
	 * LNAME が返されます。SNAMEが設定されている場合は、
	 * ツールチップにLNAME が表示されます。
	 * 概要説明が存在する場合は、ツールチップに概要説明が
	 * 表示されます。
	 *
	 * @return shortLabel ラベルオブジェクトの名称(短)
	 *
	 */
	public String getShortLabel() { return shortLabel; }

	/**
	 * ラベルオブジェクトの名称(長)を返します。
	 * 概要説明が存在する場合は、ツールチップに概要説明が
	 * 表示されます。
	 *
	 * @return longLabel ラベルオブジェクトの名称(長)
	 * @see #getLongLabel( String )
	 */
	public String getLongLabel() { return longLabel; }

	/**
	 * ラベルインターフェースの名称(長)を返します。
	 * ツールチップに表示するタイトル属性(概要説明)を置き換えます。
	 * null の場合は、既存のgetLongLabel()を返します。
	 *
	 * @param  title String ツールチップに表示するタイトル属性
	 * @return shortLabel ラベルインターフェースの名称(長)
	 * @see #getLongLabel()
	 */
	public String getLongLabel( final String title ) {
		final String tipsLabel ;
		if( title == null ) {
			tipsLabel = longLabel;
		}
		else {
			tipsLabel = "<span title=\""
							+ StringUtil.htmlFilter( title )
							+ "\">"
							+ label
							+ "</span>" ;
		}
		return tipsLabel ;
	}

	/**
	 * ラベルインターフェースの引数付きメッセージを返します。
	 * メッセージの引数部分に、文字列配列を適用して、MessageFormat
	 * で変換した結果を返します。(MessageData でのみ有効です。)
	 *
	 * @og.rev 4.0.0 (2007/10/17) メッセージリソース統合に伴い、MessageDataより移行
	 * @og.rev 4.3.8.0 (2009/08/01) 引数にHTMLサニタイジング処理
	 *
	 * @return label ラベルインターフェースの引数付きメッセージ
	 *
	 */
	public String getMessage( final String[] vals ) {
//		return label ;
		final String rtn ;
		
		// 4.3.8.0 valsに対してサニタイジングフィルタをかける
		if( vals != null && vals.length > 0){
			for( int i=0; i<vals.length; i++ ){
				vals[i] = StringUtil.htmlFilter( vals[i] );
			}
		}

		String[] args = ( vals == null ) ? new String[0] : vals ;
		if( isFormat ) {
			rtn = MessageFormat.format( label,(Object[])args );
		}
		else {
			StringBuilder buf = new StringBuilder();
			buf.append( label );
			for( int i=0; i<args.length; i++ ) {
				if( args[i] != null && ! args[i].equals( label ) ) {
					buf.append( " " ).append( args[i] );
				}
			}
			rtn = buf.toString();
		}
		return rtn ;
	}

	/**
	 * ラベルオブジェクトの概要説明を返します。
	 * 概要説明が存在する場合は、ラベルのツールチップに
	 * 概要説明が表示されます。
	 *
	 * @return description ラベルオブジェクトの概要説明
	 *
	 */
	public String getDescription() { return description; }
	
	/**
	 * ラベルオブジェクトの概要説明を返します。
	 * このメソッドでは{0},{1}...をパラメータで置換します。
	 * 
	 * @og.rev 4.3.7.6 (2009/07/15) 新規作成
	 * @og.rev 4.3.8.0 (2009/08/01) 引数にHTMLサニタイジング処理
	 *
	 * @return description ラベルオブジェクトの概要説明
	 *
	 */
	public String getDescription( final String[] vals ) {
		final String rtn ;
		
		// 4.3.8.0 valsに対してサニタイジングフィルタをかける
		if( vals != null && vals.length > 0){
			for( int i=0; i<vals.length; i++ ){
				vals[i] = StringUtil.htmlFilter( vals[i] );
			}
		}

		String[] args = ( vals == null ) ? new String[0] : vals ;
		if( isFormatDesc ) {
			rtn = MessageFormat.format( description,(Object[])args );
		}
		else {
			StringBuilder buf = new StringBuilder();
			buf.append( description );
			// Descriptionでは{}が存在しない場合は単に概要を出力
			// for( int i=0; i<args.length; i++ ) {
			// 	if( args[i] != null && ! args[i].equals( description ) ) {
			// 		buf.append( " " ).append( args[i] );
			// 	}
			// }
			rtn = buf.toString();
		}
		return rtn ;
	}

	/**
	 * リソースＤＢから作成されたかどうかを返します。
	 * 正式な場合は、true / リソースになく、独自に作成された場合は、false になります。
	 *
	 * @return official リソースＤＢから作成されたかどうか
	 *
	 */
	public boolean isOfficial() { return official; }

	/**
	 * ラベルオブジェクトの名称(短)をspanタグを付けない状態で返します。
	 * SNAMEが未設定の場合は、LNAME が返されます。
	 * 
	 * @og.rev 4.3.8.0 (2009/08/01) 追加
	 *
	 * @return rawShortLabel ラベルオブジェクトの名称(短)にspanタグを付けない状態
	 *
	 */
	public String getRawShortLabel() { return rawShortLabel; }

	/**
	 * オブジェクトの識別子として，詳細なユーザー情報を返します。
	 *
	 * @return  詳細なユーザー情報
	 */
	public String toString() {
		StringBuilder rtn = new StringBuilder( HybsSystem.BUFFER_MIDDLE );
		rtn.append( "CLM :" ).append( key );
		rtn.append( " SNAME :" ).append( shortLabel );
		rtn.append( " LNAME :" ).append( longLabel );
		rtn.append( " DESCRIPTION :" ).append( description ).append( HybsSystem.CR );
		return rtn.toString();
	}
}
