/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import static org.opengion.fukurou.util.StringUtil.nval;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.Map;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.DBColumnConfig;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.resource.CodeData;
import org.opengion.hayabusa.resource.LabelData;

/**
 * 検索結果のカラムに対して様々な属性を付加するタグです。
 *
 * このタグでは、カラムリソースの属性を再設定できます。検索結果（DBTableModel)
 * には、カラムオブジェクト（DBColumn)を持っています。このオブジェクトは、
 * カラムリソースで設定した値を持っています。その個々の値を再設定することで、
 * カラムオブジェクトの属性をその都度設定しなおすことができます。
 * 最もよく使用するのは、must 属性の設定と、DBMENU エディターの設定です。
 * must属性は、予め必須項目として指定することが出来ない為、画面毎に再設定が
 * 必要です。ただし、ここでのmust属性は、カラムの表示色を変更するだけで、
 * 実際のnullチェックは、columnCheck タグのnullCheck属性で指定します。
 * DBMENU エディターは、データベースの属性から選択リストを作成する機能ですが、
 * これも、マスタ登録時は、テキストフィールドで入力し、使用時は、選択するケース
 * の様に、画面毎に異なる形式で使用したい場合があるためです。
 * このタグでは、カラムオブジェクトの属性以外に、HTMLの属性を付加する事も
 * できます。
 *
 * @og.formSample
 * ●形式：&lt;og:ColumnEditor column=&quot;･･･&quot; ･･･ &gt; ･･･ &lt;/og:ColumnEditor &gt;
 * ●body：あり
 *
 * ●使用例：
 *     ・必須表示の場合
 *         &lt;og:columnEditor column=&quot;NOSYN&quot; must=&quot;true&quot; /&gt;
 *
 *     ・DBからのコンボボックスの場合
 *         &lt;og:columnEditor column=&quot;CDBK&quot; editor=&quot;DBMENU&quot;&gt;
 *             &lt;jsp:text&gt;
 *                 select CDBK,CDBK||&#39;:&#39;||NMBK from DB02 order by 1
 *             &lt;/jsp:text&gt;
 *         &lt;/og:columnEditor&gt;
 *         
 *     ・親子関係のプルダウンを作る場合
 *       ({&#64;xxx.value}で同じ行のxxxカラムで選択した値を取得します)
 *         &lt;og:columnEditor column=&quot;CDBK&quot; editor=&quot;DBMENU&quot;
 *              eventColumn="OYA"&gt;
 *             &lt;jsp:text&gt;
 *                 select CDBK,CDBK||&#39;:&#39;||NMBK from DB02
 *                   where CLM = {&#64;OYA.value} order by 1
 *             &lt;/jsp:text&gt;
 *         &lt;/og:columnEditor&gt;
 *     
 *
 * @og.rev 3.5.1.0 (2003/10/03) 親クラスをCommonTagSupportからHTMLTagSupportに変更。
 * @og.group 画面表示
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class ColumnEditorTag extends HTMLTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/01/31)

	private transient LabelData	labelData		= null;
	private String		tableId			= HybsSystem.TBL_MDL_KEY;
	private String[]	columns			= null;		// 3.7.0.5 (2005/04/11)
	private boolean		allColumns		= false;	// 3.8.8.5 (2007/03/09)
	private String		className		= null;
	private String		size			= null;
	private String		maxlength		= null;		// 3.5.5.5 (2004/04/23)
	private String		writable		= null;
	private String		renderer		= null;
	private String		editor			= null;
	private String		dbType			= null;
	private String		codeName		= null;		// 3.5.5.7 (2004/05/10)
	private String		defaultVal		= null;
	private String		parameter		= null;
	private String		dbid			= null;
	private String		addNoValue		= null;		// 3.5.5.9 (2004/06/07)
	private final static String DEFAULT_ADD_NO_VALUE	= "true"; // 3.7.0.5 (2005/04/11)
	
	private String 	eventColumn		= null; // 4.3.6.0 (2009/04/01) イベントカラム
	private String	rawParameter	= null; // 4.3.6.0 (2009/04/01) 生パラメータ
	private String	eventURL		= null; // 4.3.6.0 (2009/04/01) イベントURL

	// 4.1.2.1 (2008/03/13) must , mustAny 属性を自動化します。
	private String		mustType		= null;		// 4.1.2.1 (2008/03/13)
	
	// 4.3.4.0 (2008/12/01) ラベルデータの動的取得に対応します。
	private boolean		useLabelMap		= false;		// 4.3.4.0 (2008/12/01)
	private Map<String,LabelData>		labelMap		= null;

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 3.1.0.0 (2003/03/20) DBColumnConfig#getColumnConfig() メソッド廃止
	 * @og.rev 3.1.1.0 (2003/03/28) ボディの内容を取得する処理を、CommonTagSupport で行う。
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。ボディが存在する場合のみボディが呼ばれる対応。
	 * @og.rev 3.5.1.0 (2003/10/03) 親クラスをCommonTagSupportからHTMLTagSupportに変更。
	 * @og.rev 3.5.5.5 (2004/04/23) 入力欄の大きさを指定する viewSize 属性を追加。
	 * @og.rev 3.5.5.7 (2004/05/10) CodeSelection を、codeName を元に構築できるように設定します。
	 * @og.rev 3.5.5.9 (2004/06/07) addNoValue が、設定された場合のみ、config に設定します
	 * @og.rev 3.7.0.5 (2005/04/11) 複数カラムを同時設定できる columns 属性を追加
	 *
	 * @return	int
	 */
	@Override
	public int doStartTag() {
		if( "DBMENU".equalsIgnoreCase(editor) ) {
			className = "VARCHAR2";
			renderer  = "DBMENU";
			editor    = "DBMENU";

			if( addNoValue == null ) {
				addNoValue = DEFAULT_ADD_NO_VALUE ;
			}
		}

		return( EVAL_BODY_BUFFERED );
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @og.rev 2.3.1.3 (2003/02/08) ラベル、サイズのみでも変更できるように修正
	 * @og.rev 3.0.0.1 (2003/02/14) DBMENU作成時に、ラベルの最後のスペースを除去する。
	 * @og.rev 3.1.0.0 (2003/03/20) DBColumnConfig#getColumnConfig() メソッド廃止
	 * @og.rev 3.1.1.0 (2003/03/28) ボディの内容を取得する処理を、CommonTagSupport で行う。
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。ボディが存在する場合のみボディが呼ばれる対応。
	 * @og.rev 3.5.0.0 (2003/09/17) キー、ラベルともに、""（ダブルコーテーション）で囲うように変更。
	 * @og.rev 3.5.5.7 (2004/05/10) このタグでSQLを発行するのではなく、DBMENU の DBColumnを作成する。
	 * @og.rev 3.5.5.9 (2004/06/07) addNoValue が null の場合は、初期値を設定します。
	 * @og.rev 3.7.0.5 (2005/04/11) 複数カラムを同時設定できる columns 属性を追加
	 * @og.rev 3.8.6.3 (2006/11/30) SQL 文の前後のスペースを取り除きます。
	 * @og.rev 4.3.4.0 (2008/12/01) ラベルデータの動的取得に対応します。
	 * @og.rev 4.3.6.0 (2009/04/01) イベントカラム対応
	 *
	 * @return	int
	 */
	@Override
	public int doAfterBody() {
		String sql = getBodyString();
		String rawsql = getBodyRawString(); // 4.3.6.0 (2009/04/01) 生のままのSQL文を持つ
		if( sql != null && sql.length() > 0 ) {
			if( useLabelMap )	{ labelMap = getResource().getLabelMap( sql.trim() ); } // 4.3.4.0 (2008/12/01)
			else				{
				if( eventColumn != null && eventColumn.length() >0 ){  // 4.3.6.0 (2009/04/01)
					rawParameter = rawsql.trim();
				}
				parameter = sql.trim() ;
			}
		}

		return(SKIP_BODY);
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。ボディが存在する場合のみボディが呼ばれる対応。
	 * @og.rev 3.7.0.5 (2005/04/11) 複数カラムを同時設定できる columns 属性を追加
	 * @og.rev 3.8.8.5 (2007/03/09) 全カラム が指定されたことにする allColumns 属性を追加
	 * @og.rev 4.1.2.1 (2008/03/13) must , mustAny 属性を自動化します。
	 * @og.rev 4.3.4.0 (2008/12/01) カラム指定のワイルドカード対応
	 *
	 * @return  int
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)

		DBTableModel table = (DBTableModel)getObject( tableId );
		if( table == null ) { return EVAL_PAGE; }

		if( columns != null ) {
			// 3.8.8.5 (2007/03/09) 全カラム が指定
			if( allColumns ) {
				int len = table.getColumnCount();
				for( int i=0; i<len; i++ ) {

					// 4.3.4.0 (2008/12/01)
					DBColumn dbColumn = table.getDBColumn( i );
					for( int j=0; j<columns.length; j++ ) {
						String prefix = columns[j].replace( "*", "" );
						if( dbColumn.getName().indexOf( prefix ) == 0 ) { // 前方一致のみ対応
							DBColumnConfig config = getModifyConfig( dbColumn );
							table.setDBColumn( i,new DBColumn( config ) );
							
							// 4.1.2.1 (2008/03/13)
							if( mustType != null ) {
								table.addMustType( i,mustType );
							}
							break;
						}
					}
				}
			}
			else {
				for( int i=0; i<columns.length; i++ ) {
					int clmNo = table.getColumnNo( columns[i] );
					DBColumn dbColumn = table.getDBColumn( clmNo );
					DBColumnConfig config = getModifyConfig( dbColumn );
					table.setDBColumn( clmNo,new DBColumn( config ) );

					// 4.1.2.1 (2008/03/13)
					if( mustType != null ) {
						table.addMustType( clmNo,mustType );
					}
				}
			}
		}

		return(EVAL_PAGE);
	}

	/**
	 * 検索条件の入力用のテキストフィールドを作成します。
	 *
	 * テーブルタグによりフィールドは分割されます。
	 * 使用時は、テーブルタグを前後に使用して下さい。
	 * このクラスでは、このメソッドは、使われておりません。
	 *
	 * @og.rev 3.5.1.0 (2003/10/03) 新規追加
	 *
	 * @return  入力用のテキストフィールドタグ
	 */
	protected String makeTag() {
		return null;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 *
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.5.1.0 (2003/10/03) 親クラスをCommonTagSupportからHTMLTagSupportに変更。
	 * @og.rev 3.5.5.5 (2004/04/23) 入力欄の大きさを指定する viewSize 属性を追加。
	 * @og.rev 3.5.5.7 (2004/05/10) codeName 属性を追加。
	 * @og.rev 3.5.5.9 (2004/06/07) addNoValue 属性を修正。
	 * @og.rev 3.7.0.5 (2005/04/11) 複数カラムを同時設定できる columns 属性を追加
	 * @og.rev 3.8.8.5 (2007/03/09) 全カラム が指定されたことにする allColumns 属性を追加
	 * @og.rev 4.1.2.1 (2008/03/13) must , mustAny 属性を自動化します。
	 * @og.rev 4.3.4.0 (2008/12/01) ラベルデータの動的取得に対応します。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		tableId			= HybsSystem.TBL_MDL_KEY;
		columns			= null;		// 3.7.0.5 (2005/04/11)
		allColumns		= false;	// 3.8.8.5 (2007/03/09)
		labelData		= null;		// 4.0.0 (2005/01/31)
		className		= null;
		size			= null;
		maxlength		= null;		// 3.5.5.5 (2004/04/23)
		writable		= null;
		renderer		= null;
		editor			= null;
		dbType			= null;
		codeName		= null;		// 3.5.5.7 (2004/05/10)
		defaultVal		= null;
		parameter		= null;
		dbid			= null;
		addNoValue		= null;		// 3.5.5.9 (2004/06/07)
		mustType		= null;		// 4.1.2.1 (2008/03/13)
		useLabelMap		= false;	// 4.3.4.0 (2008/12/01)
		labelMap		= null;		// 4.3.4.0 (2008/12/01)
		eventColumn		= null; 	// 4.3.6.0 (2009/04/01)
		rawParameter	= null;		// 4.3.6.0 (2009/04/01)
		eventURL		= null;		// 4.3.6.0 (2009/04/01)
	}

	/**
	 * カスタマイズされた DBColumnConfig を作成します。
	 *
	 * DBColumnConfig は、DBColumn より取得します。この DBColumnConfig に、
	 * 属性で指定された値をセットすることで、カスタマイズを行います。
	 *
	 * @og.rev 3.7.0.5 (2005/04/11) 新規追加
	 * @og.rev 4.1.2.1 (2008/03/13) must , mustAny 属性を自動化します。
	 * @og.rev 4.3.4.0 (2008/12/01) ラベルデータの動的取得に対応します。
	 * @og.rev 4.3.6.0 (2009/04/01) evnetColumn対応
	 *
	 * @return  カスタマイズされた DBColumnConfig
	 */
	private DBColumnConfig getModifyConfig( final DBColumn dbColumn ) {
		DBColumnConfig config = dbColumn.getConfig();

		String lang = get( "lang" );	// 4.0.0 (2005/01/31) super クラスで受けます。

		if( lang		!= null ) { config.setLang(		 lang		); }
		if( labelData	!= null ) { config.setLabelData( labelData	); }	// 4.0.0 (2005/01/31)
		if( className	!= null ) { config.setClassName( className	); }
		if( size		!= null ) { config.setViewLength(size		); }
		if( maxlength	!= null ) { config.setMaxlength(  maxlength ); }	// 3.5.5.5 (2004/04/23)
		if( writable	!= null ) { config.setWritable(  writable	); }
		if( renderer	!= null ) { config.setRenderer(  renderer	); }
		if( editor		!= null ) { config.setEditor(	 editor 	); }
		if( dbType		!= null ) { config.setDbType(	 dbType 	); }
		if( defaultVal	!= null ) { config.setDefValue(  defaultVal ); }
		if( parameter	!= null ) { config.setParameter( parameter	); }
		if( dbid		!= null ) { config.setDbid(		 dbid		); }
		if( eventColumn != null ) { config.setEventColumn(eventColumn);}	// 4.3.6.0 (2009/04/01)
		if( eventURL	!= null ) { config.setEventURL( eventURL );	   }	// 4.3.6.0 (2009/04/01)
		if( rawParameter != null ){ config.setRawParameter( rawParameter ); } // 4.3.6.0 (2009/04/01)

		// 4.0.0 (2005/11/30) ロールを指定
		String roles = get( "roles" );
		if( roles		!= null ) { config.setRoles(     roles      ); }

		// 4.0.0 (2005/01/31)
		if( codeName   != null ) {
			CodeData codeData = getResource().getCodeData( codeName ) ;
			config.setCodeData( codeData );
		}

		// 3.5.5.9 (2004/06/07)
		// 注意：addNoValue は、互換性の関係で、値が指定された場合のみ、
		// config に設定します。
		// editor="DBMENU" が、指定された場合は、addNoValue が指定されていない場合のみ、
		// 従来との互換性の関係より、このクラスのデフォルト（true）を、config に設定します。
		if( addNoValue != null ) {
			boolean anv = Boolean.valueOf( addNoValue ).booleanValue() ;
			config.setAddNoValue( anv );
		}

		// 3.5.1.0 (2003/10/03) 親クラスをCommonTagSupportからHTMLTagSupportに変更。
		config.setEditorAttributes( getAttributes() );

		// 4.3.4.0 (2008/12/01) ラベルデータの動的取得に対応します。
		if( useLabelMap && labelMap != null ) {
			LabelData ldata = labelMap.get( dbColumn.getName() );
			if( ldata != null ) {
				config.setLabelData( ldata	);
			}
		}

		return config ;
	}

	/**
	 * 【TAG】カラム名をセットします。
	 *
	 * @og.tag
	 * カラム名をセットします。
	 * columns と同時にはセットできません。
	 * 値に"*"を指定することで、全カラムを対象とすることができます。
	 * また、"PN*,CDK*"等のように前方一致でのワイルドカード指定も可能です。
	 *
	 * @og.rev 3.7.0.5 (2005/04/11) 複数カラムを同時設定できる columns 属性を追加
	 * @og.rev 3.8.8.5 (2007/03/09) 全カラム が指定されたことにする allColumns 属性を追加
	 *
	 * @param	clm ネーム
	 */
	public void setColumn( final String clm ) {
		if( columns != null ) {
			String errMsg = "columns との同時セットは出来ません。: " + HybsSystem.CR
						+ "column=[" + clm + "] , "
						+ "columns=[" + StringUtil.array2csv( columns ) + "]";
			throw new HybsSystemException( errMsg );
		}
		String column = getRequestParameter( clm );
		if( column != null && column.length() > 0 ) {
			columns = new String[] { column };
			allColumns = column.indexOf( '*' ) >= 0 ;	// 3.8.8.5 (2007/03/09)
		}
	}

	/**
	 * 【TAG】カラム名をカンマ区切りで複数セットします。
	 *
	 * @og.tag
	 * カラム名をセットします。
	 * column と同時にはセットできません。
	 * 分解方法は、通常のパラメータ取得後に、CSV分解します。
	 * 値に"*"を指定することで、全カラムを対象とすることができます。
	 * また、"PN*,CDK*"等のように前方一致でのワイルドカード指定も可能です。
	 *
	 * @og.rev 3.7.0.5 (2005/04/11) 複数カラムを同時設定できる columns 属性を追加
	 * @og.rev 3.8.8.5 (2007/03/09) 通常のパラメータ取得後に、CSV分解に戻します。
	 * @og.rev 3.8.8.5 (2007/03/09) 全カラム が指定されたことにする allColumns 属性を追加
	 *
	 * @param	clms CSV形式のカラム名
	 */
	public void setColumns( final String clms ) {
		if( columns != null ) {
			String errMsg = "column との同時セットは出来ません。: " + HybsSystem.CR
						+ "column=[" + StringUtil.array2csv( columns ) + "] , "
						+ "columns=[" + clms + "]" ;
			throw new HybsSystemException( errMsg );
		}
		String column = nval( getRequestParameter( clms ),null );
		if( column != null ) {
			allColumns = column.indexOf( '*' ) >= 0 ;		// 3.8.8.5 (2007/03/09)
			columns = StringUtil.csv2Array( column );
			if( columns.length == 0 ) { columns = null; }
		}
	}

	/**
	 * 【TAG】ラベルリソースに定義されたラベルＩＤをセットします。
	 *
	 * @og.tag
	 * ラベルをセットします。
	 *
	 * @og.rev 2.3.1.2 (2003/01/28) ラベルリソースからラベル名を取得するように修正
	 * @og.rev 4.0.0 (2005/01/31) キーより、ラベルデータオブジェクトを取得します。
	 *
	 * @param	lbl ラベル
	 */
	public void setLabel( final String lbl ) {
		labelData = getResource().getLabelData( getRequestParameter( lbl ) ) ;	// 4.0.0 (2005/01/31)
	}

	/**
	 * 【TAG】カラムのクラス名(VARCHAR2,NUMBER など)をセットします。
	 *
	 * @og.tag
	 * カラムのクラス名をセットします。
	 *
	 * @param	cls カラムのクラス
	 */
	public void setClassName ( final String cls ) {
		className = nval( getRequestParameter( cls ),className );
	}

	/**
	 * 【TAG】カラムの入力文字制限数をセットします。
	 *
	 * @og.tag
	 * カラムの入力文字制限数をセットします。
	 *
	 * @param	len カラムの入力文字制限数
	 */
	public void setMaxlength( final String len ) {
		maxlength = nval( getRequestParameter( len ),maxlength );
	}

	/**
	 * 【TAG】カラムの表示サイズをセットします(大きさだけで入力文字制限ではありません)。
	 *
	 * @og.tag
	 * カラムのサイズをセットします。
	 *
	 * @og.rev 3.5.5.5 (2004/04/23) 入力欄の大きさを指定する viewSize 属性を追加。
	 *
	 * @param	siz カラムのサイズ
	 */
	public void setSize( final String siz ) {
		size = nval( getRequestParameter( siz ),size );
	}

	/**
	 * 【TAG】カラムが書き込み可能かどうか(true/false)をセットします。
	 *
	 * @og.tag
	 * カラムの書き込み可能をセットします。
	 *
	 * @param	flag カラムの書き込み可能かどうか（可能:true/付加:false)
	 */
	public void setWritable ( final String flag ) {
		writable = nval( getRequestParameter( flag ),writable );
	}

	/**
	 * 【TAG】カラムの表示用レンデラー(Renderer_****)をセットします。
	 *
	 * @og.tag
	 * これは、plugin.column 以下の Renderer_**** クラスの **** を
	 * 与えます。これらは、CellRenderer インターフェースを継承したサブクラスです。
	 * 属性クラス定義の
	 * <a href="/ge/jsp/DOC03/index.jsp?command=NEW&SORT=VERSION+DESC%2CSEQ%2CATTKEY&VALUENAME=renderer" target="CONTENTS">renderer</a>
	 * を参照願います。
	 *
	 * @param	rdr カラムの表示用レンデラー(Renderer_**** の ****)
	 * @see		org.opengion.hayabusa.db.CellRenderer
	 */
	public void setRenderer ( final String rdr ) {
		renderer = nval( getRequestParameter( rdr ),renderer );
	}

	/**
	 * 【TAG】カラムの編集用エディター(Editor_****)をセットします。
	 *
	 * @og.tag
	 * これは、plugin.column 以下の Editor_**** クラスの **** を
	 * 与えます。これらは、CellEditor インターフェースを継承したサブクラスです。
	 * 属性クラス定義の
	 * <a href="/ge/jsp/DOC03/index.jsp?command=NEW&SORT=VERSION+DESC%2CSEQ%2CATTKEY&VALUENAME=editor" target="CONTENTS">editor</a>
	 * を参照願います。
	 *
	 * @param	ed カラムの編集用エディター(Editor_**** の ****)
	 * @see		org.opengion.hayabusa.db.CellEditor
	 */
	public void setEditor ( final String ed ) {
		editor = nval( getRequestParameter( ed ),editor );
	}

	/**
	 * 【TAG】ＤＢタイプ(DBType_****)をセットします。
	 *
	 * @og.tag
	 * これは、org.opengion.hayabusa.db.column 以下の DBType_**** クラスの **** を
	 * 与えます。これらは、DBType インターフェースを継承したサブクラスです。
	 * 属性クラス定義の
	 * <a href="/ge/jsp/DOC03/index.jsp?command=NEW&SORT=VERSION+DESC%2CSEQ%2CATTKEY&VALUENAME=dbType" target="CONTENTS">dbType</a>
	 * を参照願います。
	 *
	 * @param	dt パラメータ
	 * @see		org.opengion.hayabusa.db.DBType  DBType_**** の ****
	 */
	public void setDbType( final String dt ) {
		dbType = nval( getRequestParameter( dt ),dbType );
	}

	/**
	 * 【TAG】コードセレクトのユニークキー(コード名称)をセットします。
	 *
	 * @og.tag
	 * このキーを元に、CodeSelection オブジェクトを構築します。
	 *
	 * @og.rev 3.5.5.7 (2004/05/10)  新規作成
	 *
	 * @param  name String メニューのユニークキー(コード名称)
	 */
	public void setCodeName( final String name ) {
		codeName = nval( getRequestParameter( name ),codeName );
	}

	/**
	 * 【TAG】パラメータをセットします。
	 *
	 * @og.tag
	 * ここで指定したパラメータは、renderer、editor、dbType すべてのパラメータに
	 * セットされます。
	 * パラメータをセットします。
	 *
	 * @param	prm パラメータ
	 */
	public void setParameter( final String prm ) {
		parameter = nval( getRequestParameter( prm ),parameter );
	}

	/**
	 * 【TAG】初期値をセットします。
	 *
	 * @og.tag
	 * 初期値をセットします。
	 *
	 * @param	dv 初期値
	 */
	public void setDefaultVal( final String dv ) {
		defaultVal = nval( getRequestParameter( dv ),defaultVal );
	}

	/**
	 * 【TAG】必須入力を表す色に変えるかどうか(true/false)指定します(初期値:false)。
	 *
	 * @og.tag
	 * 初期値は、必須でない("false") です。
	 * なお、実際の必須入力チェックは、ここではなく、columnCheck タグで指定が必要です。
	 * <del>(注意)必須チェックとは連動していません。</del>
	 * requert 変数に属性をセットし、writeCheck タグで hidden 出力します。
	 * よって、必ず、writeCheck タグより前方に記述する必要があります。
	 *
	 * @og.rev 3.5.1.0 (2003/10/03) 親クラスをCommonTagSupportからHTMLTagSupportに変更。
	 * @og.rev 4.0.0 (2005/02/28) must属性の登録を、id属性から、class 属性に変更します。
	 * @og.rev 4.1.2.1 (2008/03/13) must属性を自動化します。
	 *
	 * @param	flag 必須入力 ("true")／必須でない (それ以外)
	 */
	public void setMust( final String flag ) {
		boolean must = nval( getRequestParameter( flag ),false );
//		mustType = nval( getRequestParameter( flag ),null );
//		if( "true".equalsIgnoreCase( mustType ) ) {
		if( must ) {
			mustType = "must";
			add( "class","must" );
		}
	}

	/**
	 * 【TAG】選択必須入力(どれかひとつ必須)を表す色(true|mustAny|その他)を指定します(初期値:無指定)。
	 *
	 * @og.tag
	 * 複数のカラムのうち、どれかひとつを必須とする選択必須入力を示す色を指定します。
	 * true または、mustAny を設定すると、class属性に、mustAny がセットされます。
	 * mustAny は、CSSファイルに初期設定されています。true または、mustAny 以外の値をセット
	 * すると、その値がそのまま、class属性にセットされますので、選択必須のグループ化が
	 * 可能です。
	 * なお、実際の選択必須入力チェックは、ここではなく、columnCheck タグで指定が必要です。
	 * must 属性と同時に設定した場合は、must 属性 が優先されます。
	 * 初期値は、無指定です。
	 * requert 変数に属性をセットし、writeCheck タグで hidden 出力します。
	 * よって、必ず、writeCheck タグより前方に記述する必要があります。
	 *
	 * @og.rev 3.8.0.9 (2005/10/17) 新規追加
	 * @og.rev 4.1.2.1 (2008/03/13) mustAny属性を自動化します。
	 *
	 * @param	flag 選択必須入力(どれかひとつ必須)を表す色を指定(true|mustAny|その他)
	 */
	public void setMustAny( final String flag ) {
		if( mustType == null ) {	// must 属性と同時設定時には、must 属性を優先します。
			mustType = nval( getRequestParameter( flag ),null );
			if( "true".equalsIgnoreCase( mustType ) ) {
				mustType = "mustAny";
			}
			add( "class",mustType );	// mustType == null の場合は、add されません。
		}

//		String str = nval( getRequestParameter( flag ),null );
//		if( str != null ) {
//			if( str.equalsIgnoreCase( "true" ) ) {
//				add( "class","mustAny" );
//			}
//			else {
//				add( "class",str );
//			}
//		}
	}

	/**
	 * 【TAG】コンボボックスの値なしのOptionの有無(true/false)を指定します(初期値:falseと同等)。
	 *
	 * @og.tag
	 * コンボボックスの場合、値なしのOptionをSelectに含めるかどうかを指定します。
	 * ここでは、検索結果の登録画面に使用されるケースが多いため、なにも指定されない
	 * 場合は、なにもセットしません。(Coderesourceの指定どおり)
	 * 動作としては、初期値は、含めない("false") と同等です。
	 *
	 * &lt;og:columnEditor column="CDBK" addNoValue="false" /&gt;
	 *
	 * @param	flag 含める ("true")／含めない (それ以外)
	 */
	public void setAddNoValue( final String flag ) {
		addNoValue = nval( getRequestParameter( flag ),addNoValue );
	}

	/**
	 * 【TAG】SQL文を指定してラベル情報を動的に読み込みます。
	 *
	 * @og.tag
	 * trueを指定すると、body部分に記述されたSQL文よりラベル情報を生成して、
	 * 動的にラベル表示を切り替えます。
	 * 
	 * ワイルドカードを使用した、カラムの複数指定と組み合わせることで、外部から設定
	 * されたラベル情報で一括で変更することができます。
	 * 
	 * 発行するクエリでは、第1カラムにラベルキーを、第2カラムにラベル名称を設定します。
	 * 第3カラムが存在する場合は、名称(短)として使用されます。(必須ではありません)
	 * 
	 * ここで作成されたラベル情報は、カラムエディターを適用したDBTableModelのみで
	 * 有効であり、内部的にもキャッシュされません。
	 * 
	 * 初期値は、false(動的読込しない)です。
	 * 
	 * @og.rev 4.3.4.0 (2008/12/01) 新規作成
	 *
	 * @param	flag true(動的読込する)／false(動的読込しない)
	 */
	public void setUseLabelMap( final String flag ) {
		useLabelMap = nval( getRequestParameter( flag ),useLabelMap );
	}

	/**
	 * 【TAG】(通常使いません)sessionから所得する DBTableModel オブジェクトの ID。
	 *
	 * @og.tag
	 * 表示処理後に，（内部ポインタを書き換えた）DBTableModel オブジェクトを
	 * 同じキーで、sessionに登録します。
	 * 初期値は、HybsSystem.TBL_MDL_KEY です。
	 *
	 * @og.rev 3.5.0.0 (2003/09/17) セッターメソッドの追加
	 *
	 * @param	id sessionに登録する時の ID
	 */
	public void setTableId( final String id ) {
		tableId   = nval( getRequestParameter( id ),tableId );
	}

	/**
	 * 【TAG】データベースへ接続する場合のID。
	 *
	 * @og.tag
	 * データベース接続を使用するエディター、レンデラーを使用する場合に使用する接続先IDを指定します。
	 * カラムオブジェクトは、query処理ではなく、view処理時に適用されるため、データベース
	 * 接続先を指定する必要があります。
	 * 初期値は、null(DEFAULT 接続) です。
	 *
	 * @og.rev 4.0.0 (2006/04/02) 新規追加
	 *
	 * @param	id 接続先ID
	 */
	public void setDbid( final String id ) {
		dbid = nval( getRequestParameter( id ),dbid );
	}
	
	/**
	 * 【TAG】イベントカラム（親カラム）のセット
	 *
	 * @og.tag
	 * 親カラムをセットします。
	 * 親子関係のプルダウンを作成する場合等に利用します。
	 * （editorによっては利用できません。）
	 * {@xxx}は一覧の同じ行のxxxカラムの値です。
	 * columnEditorでセットしたサイズ等は入替え時には無視されます。
	 * 
	 * columnタグの説明も参照して下さい。
	 *
	 * @og.rev 4.3.6.0 (2009/04/01)
	 *
	 * @param col	
	 */
	public void setEventColumn( final String col ) {
		eventColumn = nval( getRequestParameter( col ), eventColumn );
	}

	/**
	 * 【TAG】eventColumn利用時に呼ばれるJSPのURL
	 *
	 * @og.tag
	 * イベントカラム指定時に部品を作成するJSPを指定します。
	 * 初期値はシステムリソースのEVENT_COLUMN_URLです。
	 * (例：eventURL="makeColumn_custom.jsp")
	 *
	 * @og.rev 4.3.6.0 (2009/04/01)
	 *
	 * @param url	
	 */
	public void setEventURL( final String url ) {
		eventURL = nval( getRequestParameter( url ), eventURL );
	}

	/**
	 * シリアライズ用のカスタムシリアライズ書き込みメソッド
	 *
	 * @og.rev 4.0.0 (2006/09/31) 新規追加
	 * @serialData
	 *
	 * @param strm ObjectOutputStream
	 */
	private void writeObject( final ObjectOutputStream strm ) throws IOException {
		strm.defaultWriteObject();
	}

	/**
	 * シリアライズ用のカスタムシリアライズ読み込みメソッド
	 *
	 * ここでは、transient 宣言された内部変数の内、初期化が必要なフィールドのみ設定します。
	 *
	 * @og.rev 4.0.0 (2006/09/31) 新規追加
	 * @serialData
	 *
	 * @param strm ObjectInputStream
	 * @see #release2()
	 */
	private void readObject( final ObjectInputStream strm ) throws IOException , ClassNotFoundException {
		strm.defaultReadObject();
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "tableId"		,tableId	)
				.println( "columns"		,columns	)
				.println( "allColumns"	,allColumns	)
				.println( "className"	,className	)
				.println( "size"		,size		)
				.println( "maxlength"	,maxlength	)
				.println( "writable"	,writable	)
				.println( "renderer"	,renderer	)
				.println( "editor"		,editor		)
				.println( "codeName"	,codeName	)
				.println( "dbType"		,dbType		)
				.println( "defaultVal"	,defaultVal	)
				.println( "parameter"	,parameter	)
				.println( "dbid"		,dbid		)
				.println( "addNoValue"	,addNoValue	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
