/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import static org.opengion.fukurou.util.StringUtil.nval;

import java.util.ArrayList;
import java.util.List;

import org.opengion.hayabusa.common.ErrorMessage;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.db.Query;
import org.opengion.hayabusa.resource.GUIInfo;

/**
 * SQL文を直接指定して、データベースに追加/更新/削除を行います(queryType="JDBCTableUpdate")。
 *
 * 存在チェックを行う場合は、tableExist タグと併用してください。
 * 複雑な処理が必要な場合は、従来より使用しています、PLSQLをCALLする、
 * plsqlUpdateタグを使用してください。
 * また、tableUpdateParam タグを使用する事で、テーブル名とsqlTypeの指定で動的に
 * SQL文を自動生成できます。これにより、追加、更新、削除やテーブルに関して、
 * 単一のJSP画面ですべて対応できるようになります。
 *
 * @og.formSample
 * ●形式：&lt;og:tableUpdate command="…" names="…" queryType="JDBCTableUpdate" &gt;
 *             &lt;jsp:text&gt;{&#064;SQL}&lt;/jsp:text&gt;
 *         &lt;/og:update&gt;
 * ●body：あり
 *
 * ●使用例
 *    ・QUERYを他のJSPから渡す場合
 *    【copy.jsp】
 *        &lt;og:hidden name=&quot;SQL&quot; &gt;
 *          &lt;jsp:text&gt;
 *          INSERT INTO GE41
 *               (CLM,NAME_JA,LABEL_NAME,KBSAKU,SYSTEM_ID,LANG,
 *               FGJ,DYSET,DYUPD,USRSET,USRUPD,PRGUPD)
 *          VALUES
 *               ([CLM],[NAME_JA],[LABEL_NAME],[KBSAKU],[SYSTEM_ID],[LANG],
 *               &#39;1&#39;,&#39;{&#064;USER.YMDH}&#39;,&#39;{&#064;USER.YMDH}&#39;,&#39;{&#064;USER.ID}&#39;,&#39;{&#064;USER.ID}&#39;,&#39;{&#064;GUI.KEY}&#39;)
 *          &lt;/jsp:text&gt;
 *        &lt;/og:value&gt;
 *
 *    【entry.jsp】
 *        &lt;og:tableUpdate
 *            command   = &quot;{&#064;command}&quot;
 *            queryType = &quot;JDBCTableUpdate&quot;
 *        &lt;jsp:text&gt;{&#064;SQL}&lt;/jsp:text&gt;
 *        &lt;/og:tableUpdate&gt;
 *
 *    ・tableUpdateParamを使用する場合
 *    【entry.jsp】
 *        &lt;og:tableUpdate
 *            command   = &quot;{&#064;command}&quot;
 *            queryType = &quot;JDBCTableUpdate&quot;
 *            sqlType   = &quot;{&#064;sqlType}&quot;        // tableUpdateParam の sqlType と一致
 *        &gt;
 *            &lt;og:tableUpdateParam
 *                sqlType     = &quot;{&#064;sqlType}"       // INSERT,COPY,UPDATE,MODIFY,DELETE
 *                table       = &quot;{&#064;TABLE_NAME}"    // 処理対象のテーブル名
 *                names       = &quot;{&#064;names}"         // 処理対象のカラム名
 *                omitNames   = &quot;{&#064;omitNames}"     // 処理対象外のカラム名
 *                where       = &quot;{&#064;where}"         // 処理対象を特定するキー
 *                constKeys   = &quot;{&#064;constKeys}"     // 処理カラム名の中の固定情報カラム名
 *                constVals   = &quot;{&#064;constVals}"     // 処理カラム名の中の固定情報設定値
 *            /&gt;
 *        &lt;/og:tableUpdate&gt;
 *
 *    ・処理の可否を指定する場合
 *    【entry.jsp】
 *        &lt;og:tableUpdate
 *            command   = &quot;{&#064;command}&quot;
 *            queryType = &quot;JDBCTableUpdate&quot;
 *            conditionKey  ="…"      : 条件判定するカラムＩＤを指定(初期値は columnId )
 *            conditionList ="…"      : 条件判定する値のリストを、"|"で区切って登録(初期値は、無条件)
 *        &lt;jsp:text&gt;{&#064;SQL}&lt;/jsp:text&gt;
 *        &lt;/og:tableUpdate&gt;
 *
 * @og.rev 3.8.8.0 (2007/12/22) 新規作成
 * @og.group ＤＢ登録
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class TableUpdateTag extends QueryTag {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/01/31)

	/** command 引数に渡す事の出来る コマンド  登録{@value} */
	public static final String CMD_ENTRY  = "ENTRY" ;
	/** command 引数に渡す事の出来る コマンド リスト  */
	private static final String COMMAND_LIST = CMD_ENTRY;

	// 処理を行う、リソースの種類を指定します。(GEA03,GEA04,GEA08 のどれか)
	private String	sqlType			= null;		// INSERT,COPY,UPDATE,MODIFY,DELETE
	private String	resourceType	= null;
	private int		resTypeColNo	= -1;
	private String	conditionKey	= null;		// 条件判定するカラムＩＤを指定(初期値は columnId )
	private String	conditionList	= null;		// 条件判定する値のリストを、"|"で区切って登録(初期値は、無条件)
	private boolean selectedAll		= false;
	private boolean commitTableModel= true;		// 4.0.2.0 (2007/12/25)
	private boolean followCdkh		= false;	// 4.3.2.0 (2008/09/09).

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 4.0.0.0 (2007/11/14) 0件の場合でもstartQueryTransactionを通すように変更
	 *
	 * @return  int
	 */
	@Override
	public int doStartTag() {
		dyStart = System.currentTimeMillis();
		table = (DBTableModel)getObject( tableId );
		startQueryTransaction( tableId );		// 4.0.0.0 (2007/11/14) 0件の場合でもdoEndでPAGE_SKIPしないように位置変更。
		if( table == null || table.getRowCount() == 0 ||
			! check( command, COMMAND_LIST ) ) { return(SKIP_BODY); }
//		startQueryTransaction( tableId );		// 3.6.0.8 (2004/11/19)
		quotCheck = false;		// このタグでは、クオートチェックは行いません。
		return( EVAL_BODY_BUFFERED );	// Body を評価する。（ extends BodyTagSupport 時）
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 4.0.0.0 (2007/10/18) メッセージリソース統合( getResource().getMessage > getResource().getLabel )
	 *
	 * @return	int
	 */
	@Override
	public int doEndTag() {
		debugPrint();

//		String label = HybsSystem.BR;				// 検索しなかった場合。
		String label  = "";				// 4.0.0 (2005/11/30) 検索しなかった場合。
		if( check( command, COMMAND_LIST ) ) {
			StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_SMALL );
			if( executeCount > 0 && displayMsg != null && displayMsg.length() > 0 ) {
				buf.append( executeCount );
//				buf.append( getResource().getMessage( displayMsg ) );
				buf.append( getResource().getLabel( displayMsg ) );
				buf.append( HybsSystem.BR );
			}

			String err = TaglibUtil.makeHTMLErrorTable( errMessage,getResource() );
			if( err != null && err.length() > 0 ) {
//				if( errCode >= ErrorMessage.NG ) {		// 異常の場合
//					label = err ;
//				}
				buf.append( err );
				setSessionAttribute( errMsgId,errMessage );
			}
			else {
				removeSessionAttribute( errMsgId );
			}
			label = buf.toString();

			if( table != null && ! commitTableObject( tableId, table )  ) {
				// 3.6.0.8 (2004/11/19) トランザクションチェックを行います。
				jspPrint( "TableUpdateTag Query処理が割り込まれました。DBTableModel は登録しません。" );
				return (SKIP_PAGE);
			}
		}

		jspPrint( label );

		// 警告時に停止していましたが、継続処理させます。
//		int rtnCode = EVAL_PAGE;
//		if( errCode >= ErrorMessage.NG )  { 	// 異常
//			rtnCode = SKIP_PAGE;
//		}
//		else {
//			rtnCode = EVAL_PAGE;
//		}

		long dyTime = System.currentTimeMillis()-dyStart;
		jspPrint( "<div id=\"queryTime\" value=\"" + (dyTime) + "\"></div>" );	// 3.5.6.3 (2004/07/12)

		// セキュリティチェック(データアクセス件数登録)
		GUIInfo guiInfo = (GUIInfo)getSessionAttribute( HybsSystem.GUIINFO_KEY );
		if( guiInfo != null ) { guiInfo.addWriteCount( executeCount,dyTime,sql ); }

		int rtnCode = ( errCode >= ErrorMessage.NG ) ? SKIP_PAGE : EVAL_PAGE ;
		return( rtnCode );
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 4.0.2.0 (2007/12/25) commitTableModel追加
	 * @og.rev 4.1.2.0 (2008/03/12) sqlType追加
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		sqlType			= null;		// INSERT,COPY,UPDATE,MODIFY,DELETE
		resourceType	= null;
		resTypeColNo	= -1;
		conditionKey	= null;		// 条件判定するカラムＩＤを指定(初期値は columnId )
		conditionList	= null;		// 条件判定する値のリストを、"|"で区切って登録(初期値は、無条件)
		selectedAll		= false;
		commitTableModel= true;		// 4.0.2.0 (2007/12/25)
		followCdkh		= false;	// 4.3.2.0 (2008/09/09)
	}

	/**
	 * Query を実行します。
	 *
	 * @og.rev 4.0.2.0 (2007/12/25) commitTableModel追加
	 *
	 * @param   query オブジェクト
	 */
	protected void execute( final Query query ) {

		try {
			int[] rowNo = getParameterRows();		// 4.0.0 (2005/01/31)
			if( rowNo.length > 0 ) {
				query.execute( rowNo,table );

				errCode = query.getErrorCode();
				errMessage = query.getErrorMessage();

				// リソースクリア処理
				if( resourceType != null ) {
//					if( "GEA09".equals( resourceType ) ) {
//						resTypeColNo = table.getColumnNo( "MSGCD" );	// キーは、MSGCD
//					}
//					else {
					resTypeColNo = table.getColumnNo( "CLM" );		// キーは、CLM
//					}
				}

				// 逆順にDELETEしないと、行番号がずれてしまう。
				int row;
				for( int j=rowNo.length-1; j>=0; j-- ) {
					row = rowNo[j];
					if( resTypeColNo >= 0 ) {
						clearResourceData( table.getValue( row,resTypeColNo ) );		// リソースのクリア
					}

					if( commitTableModel ) { // 4.0.2.0 (2007/12/25)
						if( DBTableModel.DELETE_TYPE.equals( table.getModifyType( row ) ) ) {
							table.removeValue( row );
						}
						else {
							table.resetModify( row );
						}
					}
				}
//				if( query.getUpdateFlag() ) { query.commit(); }
				if( query.isUpdate() ) { query.commit(); }
			}
		}
		finally {
			if( query != null ) { query.close(); }
		}
	}

	/**
	 * 表示データの HybsSystem.ROW_SEL_KEY を元に、選ばれた 行番号の
	 * 配列を返します。
	 * ここでは、conditionKey に値が設定されている場合は、そのカラムの値が
	 * conditionList にマッチする場合のみ対象選択行として返します。
	 * 値がセットされていない場合は、通常のCommonTagSupport#getParameterRows()
	 * が呼ばれます。
	 * なにも選ばれていない場合は、サイズ０の配列を返します。
	 * 
	 * @og.rev 4.3.2.0 (2008/09/09) followCdkh属性対応
	 *
	 * return int[] (選ばれていない場合は、サイズ０の配列を返す)
	 */
	protected int[] getParameterRows() {
		int[] rowNo ;
		if( selectedAll ) {
			int rowCnt = table.getRowCount();		// 3.5.5.7 (2004/05/10)
			rowNo = new int[ rowCnt ];
			for( int i=0; i<rowCnt; i++ ) {
				rowNo[i] = i;
			}
		} else {
			rowNo = super.getParameterRows();		// 4.0.0 (2005/01/31)
		}

		if( conditionKey != null ) {
			int col = table.getColumnNo( conditionKey );
			List<Integer> list = new ArrayList<Integer>();
			for( int i=0; i<rowNo.length; i++ ) {
				String val = "|" + table.getValue( rowNo[i],col ) + "|";
				if( conditionList.indexOf( val ) >= 0 ) {
					list.add( Integer.valueOf( rowNo[i] ) );
				}
			}

			int size = list.size();
			rowNo = new int[size];
			for( int i=0; i<size; i++ ) {
				rowNo[i] = (list.get(i)).intValue();
			}
		}

		// 4.3.2.0 (2008/09/09)
		if( sqlType != null && sqlType.length() > 0 && followCdkh ) {
			List<Integer> flist = new ArrayList<Integer>();
			for( int i=0; i<rowNo.length; i++ ) {
				String cdkh = table.getModifyType( rowNo[i] );
				if( ( ( "INSERT".equals( sqlType ) || "COPY".equals( sqlType ) )
							&& DBTableModel.INSERT_TYPE.equals( cdkh ) )
					||
					( ( "UPDATE".equals( sqlType ) || "CHANGE".equals( sqlType ) )
							&& DBTableModel.UPDATE_TYPE.equals( cdkh ) )
					||
					( ( "DELETE".equals( sqlType ) ) 
							&& DBTableModel.DELETE_TYPE.equals( cdkh ) ) ) {
					flist.add(  Integer.valueOf( rowNo[i] ) );
				}
			}

			int size = flist.size();
			rowNo = new int[size];
			for( int i=0; i<size; i++ ) {
				rowNo[i] = (flist.get(i)).intValue();
			}
		}

		return rowNo;
	}

	/**
	 * 【TAG】Query を発行する為のクラスIDを指定します(初期値:JDBCTableUpdate)。
	 *
	 * @og.tag
	 * 引数指定のINSERT/UPDATE文を実行する場合の、queryType 属性を使用します。
	 * このタグでは、execute( int[] ,DBTableModel )を実行します。
	 * 代表的なクラスとして、"JDBCTableUpdate" が標準で用意されています。
	 *
	 * タグにより使用できる／出来ないがありますが、これは、org.opengion.hayabusa.db
	 * 以下の Query_**** クラスの **** を与えます。
	 * これらは、Query インターフェースを継承したサブクラスです。
	 *
	 * @param	id Query を発行する為の実クラス ID
	 * @see		org.opengion.hayabusa.db.Query  Queryのサブクラス
	 * @see		org.opengion.hayabusa.db.Query#execute( int[] ,DBTableModel )
	 */
	public void setQueryType( final String id ) {
		super.setQueryType( nval( id,"JDBCTableUpdate" ) );
	}

	/**
	 * resourceType が設定されたときのみ使用される、キャッシュの初期化メソッドです。
	 *
	 * @param key String	初期化を行うキー
	 */
	private void clearResourceData( final String key ) {
		getResource().clear( key );
	}

	/**
	 * 【特殊】クリアするリソースの種類を指定します(GEA03,GEA04,GEA08 のどれか)
	 *
	 * @og.tag
	 * 注意：この属性は、リソース関連DBのメンテナンス時にのみ、内部リソースキャッシュを
	 * クリアする目的で使用します。一般の属性としては、使用することはないため、
	 * ご注意ください。
	 * リソース関連のテーブルを更新した場合、リソースキャッシュをクリアして
	 * 置かないと、データベースの値が反映されません。
	 * 昔は、リソースの更新ごとに、全件クリアしていましたが、部分クリアが
	 * できるようになったため、部分クリアを行います。
	 * こでは、(GEA03,GEA04,GEA08) のどれかを指定してください。
	 *
	 * @param	type String クリアするリソースタイプ
	 */
	public void setResourceType( final String type ) {
		resourceType = nval( getRequestParameter(type),resourceType );

		if( resourceType != null &&
//			"|GEA03|GEA04|GEA08|GEA09|".indexOf( "|" + resourceType + "|" ) < 0 ) {
//				String errMsg = "resourceTypeは GEA03,GEA04,GEA08,GEA09 のどれかです。"
//							+ "resourceType=" + type ;
			"|GEA03|GEA04|GEA08|".indexOf( "|" + resourceType + "|" ) < 0 ) {
				String errMsg = "resourceTypeは GEA03,GEA04,GEA08 のどれかです。"
							+ "resourceType=" + type ;
				throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【TAG】条件判定するカラムＩＤを指定します(初期値は、null)
	 *
	 * @og.tag
	 * 指定のカラムＩＤの値と、conditionList の値を比較して、
	 * 存在する場合は、Query 処理を実行します。
	 * 例えば、conditionKey="CDKH" として、conditionList="A" とすれば、
	 * 改廃コードが"A"のデータで、かつ選択されたデータのみを処理します。
	 * 設定しない場合は、通常の処理と同様に、選択行のみ処理されます。
	 *
	 * @param	key String
	 * @see		#setConditionList( String )
	 */
	public void setConditionKey( final String key ) {
		conditionKey = nval( getRequestParameter( key ),null ) ;
	}

	/**
	 * 【TAG】条件判定する値のリストを、"|"で区切って登録します(初期値は、無条件)
	 *
	 * @og.tag
	 * conditionKey とペアで指定します。ここには、カラムの設定値のリストを
	 * 指定することで、複数条件(ＯＲ結合)での比較を行い、リストにカラム値が
	 * 存在する場合のみ、Query 処理を実行します。
	 * 値が設定されている場合は、その値とマッチする必要があります。なにもセット
	 * されない場合、または、null の場合は、null データとマッチする場合のみ処理
	 * されますので、ご注意ください。
	 *
	 * @param	list String
	 * @see		#setConditionKey( String )
	 */
	public void setConditionList( final String list ) {
		conditionList = "|" + nval( getRequestParameter( list ),"" ) + "|" ;
	}

	/**
	 * 【TAG】データを全件選択済みとして処理するかどうか(true/false)を指定します(初期値:false)。
	 *
	 * @og.tag
	 * 全てのデータを選択済みデータとして扱って処理します。
	 * 全件処理する場合に、（true/false)を指定します。
	 * 初期値は false です。
	 *
	 * @param  all データを全件選択済み(true) / 通常（false)
	 */
	public void setSelectedAll( final String all ) {
		selectedAll = nval( getRequestParameter( all ),selectedAll );
	}

	/**
	 * 【特殊】SQL実行後に結果をDBTableModelに反映させるかどうかを指定します。(初期値:true)
	 *
	 * @og.tag
	 * 注意：この属性は、リソース関連DBのメンテナンス時に、複数DBへの登録を行うための、
	 * 暫定対応として定義しています。
	 * falseにした場合は、実データとDBTableModelの整合性が取れなくなるため、使用には十分注して下さい。
	 * 初期値は true です。
	 *
	 * @og.rev 4.0.2.0 (2007/12/25) 新規作成
	 *
	 * @param  commitTblMdl テーブルモデルへ反映する(true) / 反映しない（false)
	 */
	public void setCommitTableModel( final String commitTblMdl ) {
		commitTableModel = nval( getRequestParameter( commitTblMdl ),commitTableModel );
	}

	/**
	 * 引数の名称配列
	 *
	 * @return	String[]
	 */
	protected String[] getNames() {
		return table.getNames() ;
	}

	/**
	 * 【TAG】BODY部に書かれている Param の SQLタイプを指定します。
	 *
	 * @og.tag
	 * TableUpdateParamTag は、上位の TableUpdateTag の sqlType 属性 と同じ
	 * sqlType 属性の場合のみ、SQL文を合成・出力します。
	 * つまり、TableUpdateTag側のsqlType 属性をパラメータに、TableUpdateParamTag
	 * の sqlType 属性を固定値にすることで、どのパラメータを使用するかを
	 * 選択できる機能を実現する事が可能です。
	 *
	 * @og.rev 4.1.2.0 (2008/03/12) 新規追加
	 *
	 * @param	type BODY部に書かれている SQL タイプ
	 */
	public void setSqlType( final String type ) {
		sqlType = nval( getRequestParameter( type ),sqlType );
	}

	/**
	 * 【TAG】DBTableModelの改廃Cに従って処理を行うかを指定します。
	 *
	 * @og.tag
	 * この属性は、sqlTypeが指定されている場合のみ有効です。
	 * sqlTypeが指定されている場合、そのsqlTypeに対応した、改廃Cが設定されている
	 * 行のみを処理します。
	 * 対応関係は、以下の通りです。
	 *  sqlType = "INSERT" or "COPY" -> 改廃C='A'のみ処理
	 *  sqlType = "UPDATE" or "CHANGE" -> 改廃C='C'のみ処理
	 *  sqlType = "DELETE" -> 改廃C='D'のみ処理
	 *
	 * @og.rev 4.3.2.0 (2008/09/09) 新規追加
	 *
	 * @param	flg DBTableModelの改廃Cに従って処理を行うか
	 */
	public void setFollowCdkh( final String flg ) {
		followCdkh = nval( getRequestParameter( flg ),followCdkh );
	}

	/**
	 * SQLタイプを返します。
	 *
	 * @og.rev 4.1.2.0 (2008/03/12) 新規追加
	 *
	 * @return	String[]
	 */
	protected String getSqlType() {
		return sqlType ;
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @og.rev 4.0.2.0 (2007/12/25) resourceColumn、commitTableModel追加
	 *
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION		)
				.println( "resourceType"	,resourceType	)
				.println( "resTypeColNo"	,resTypeColNo	)
				.println( "conditionKey"	,conditionKey	)
				.println( "conditionList"	,conditionList	)
				.println( "followCdkh"		,followCdkh		)
				.println( "CMD_ENTRY"		,CMD_ENTRY		)
				.println( "commitTabelModel",commitTableModel )		// 4.0.2.0 (2007/12/25)
				.println( "sql"				,sql			)		// 4.1.2.0 (2008/03/12)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString()
			+ HybsSystem.CR
			+ super.toString() ;
	}
}
