/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.ErrorMessage;

import java.util.Locale ;

/**
 * 半角大文字の英数字の文字列を扱う為の、カラム属性を定義します。
 *
 * 入力文字列は、trim された後、大文字化(toUpperCase)された後、タイプチェックされます。
 * 半角カタカナは半角大文字に分類されない為、対象外(使用できない)です。
 *
 * タイプチェックとして、以下の条件を判定します。
 * ・文字列長は、Byte換算での文字数との比較
 * ・半角大文字チェック「c &lt; 0x20 || c &gt; 0x7e || ( 'a' &lt;= c && c &lt;= 'z' )以外」エラー
 * ・文字パラメータの 正規表現チェック
 * ・クロスサイトスクリプティングチェック
 *
 * @og.group データ属性
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DBType_XU extends AbstractDBType {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	/**
	 * 半角０文字の固定長でFILL埋めされた文字列を返します。
	 * なお、エラーチェックは行われません。
	 * 実行前に、必ず valueCheck( String value ,int len ) が行われる必要があります。
	 *
	 * @og.rev 3.5.4.5 (2004/01/23) エンコード指定に変更します。
	 *
	 * @param   value    ＦＩＬＬ埋めする文字列
	 * @param   sizeX    整数部分の文字列の長さ
	 * @param   sizeY    少数部分の文字列の長さ
	 * @param   encode   固定長で変換する文字エンコード
	 * @return  ＦＩＬＬ埋めした新しい文字列
	 */
	public String valueFill( final String value ,final int sizeX ,final int sizeY,final String encode ) {
		int len = (sizeY == 0) ? sizeX : sizeX + sizeY + 1;

		return StringUtil.stringXFill( value,len );
	}

	/**
	 * エディターで編集されたデータを登録する場合に、データそのものを
	 * 変換して、実登録データを作成します。
	 * 例えば,大文字のみのフィールドなら、大文字化します。
	 * 実登録データの作成は、DBType オブジェクトを利用しますので,
	 * これと Editor とがアンマッチの場合は、うまくデータ変換
	 * されない可能性がありますので、注意願います。
	 *
	 * @og.rev 3.3.3.0 (2003/07/09) 前後のスペースを取り除いておく。
	 * @og.rev 3.3.3.1 (2003/07/18) 後ろスペースを取り除く。(StringUtil#rTrim)
	 *
	 * @param   value String （一般に編集データとして登録されたデータ）
	 * @return  修正後の文字列（一般にデータベースに登録するデータ）
	 */
	public String valueSet( final String value ) {
		if( value != null ) { return (StringUtil.rTrim( value )).toUpperCase(Locale.JAPAN); }
		else                { return null; }
	}

	/**
	 * データが登録可能かどうかをチェックします。
	 * データがエラーの場合は、そのエラー内容を返します。
	 *
	 * @og.rev 3.6.0.0 (2004/09/22) dbType パラメータを引数に追加
	 *
	 * @param   key String
	 * @param   value String
	 * @param   sizeX    整数部分の文字列の長さ
	 * @param   sizeY    少数部分の文字列の長さ
	 * @param   param    dbType パラメータ
	 * @return  エラー内容
	 */
	public ErrorMessage valueCheck( final String key ,final String value ,
									final int sizeX ,final int sizeY ,final String param ) {
		ErrorMessage msg = new ErrorMessage();
		if( value == null || value.length() == 0 ) { return msg; }

		int len = (sizeY == 0) ? sizeX : sizeX + sizeY + 1;
		if( value.length() > len ) {
			// 文字列の長さが指定の長さよりも長いです。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0006", key,value, String.valueOf( value.length() ), String.valueOf( len ) );
		}

		StringBuilder val = new StringBuilder();
		boolean isError = false;
		for( int i=0; i<value.length(); i++ ) {
			char ch = value.charAt( i );
			if( ch < 0x20 || ch > 0x7e || ( 'a' <= ch && ch <= 'z' ) ) {
				val.append( "<span class=\"NG\">" ).append( ch ).append( "</span>" );
				isError = true;
			}
			else {
				val.append( ch );
			}
		}
		if( isError ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,val.toString() );
		}

		// 3.6.0.0 (2004/09/22) dbType パラメータを使用したマッチチェック
		String check = DBTypeCheckUtil.matcheCheck( value,param );
		if( check != null ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,check );
		}

		// クロスサイトスクリプティング対策：'<', '>' は登録させない。
		msg = xssCheck( key ,value, msg );
		return msg;
	}
}
