/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.resource.GUIInfo;
import org.opengion.fukurou.util.XHTMLTag ;

import static org.opengion.fukurou.util.StringUtil.nval ;

import javax.servlet.http.HttpServletRequest ;

/**
 * サブミットボタンの表示や、画面遷移先情報(forward.jsp)を制御するタグです。
 *
 * ユーザー情報と画面情報を元に、ロール/グループの権限と
 * RW(読取/書込)モードを比較して、条件が成立しない場合は,そのBODY部を表示しないように
 * できます。
 * このタグは、form 内に一つだけ記述してください。
 *
 * @og.formSample
 * ●形式：&lt;og:writeCheck /&gt;
 * ●body：あり
 *
 * ●使用例
 *      &lt;og:writeCheck&gt;
 *          &lt;og:submit  value=&quot;copy&quot;     lbl=&quot;COPY&quot;    accesskey=&quot;C&quot; /&gt;
 *          &lt;og:submit  value=&quot;modify&quot;   lbl=&quot;MODIFY&quot;  accesskey=&quot;M&quot; /&gt;
 *          &lt;og:submit  value=&quot;delete&quot;   lbl=&quot;DELETE&quot;  accesskey=&quot;D&quot; /&gt;
 *          &lt;jsp:directive.include file=&quot;{&#064;SYS.JSP}/common/Excel_direct.jsp&quot;  /&gt;
 *          &lt;br /&gt;
 *      &lt;/og:writeCheck&gt;
 *
 * @og.group 画面制御
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class WriteCheckTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/01/31)

	// 3.5.4.0 (2003/11/25) BACK_GAMENID を使用する/しないを指定します。
	private boolean		useGamenId	= true;
	// 3.5.4.0 (2003/11/25) 画面と別にロールズを与えられます。
	private String 		roles		= null;

	// 3.8.5.1 (2006/05/08) 書き込み許可属性 writable 追加
	private boolean		writable 	= true;

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 3.0.1.2 (2003/03/07) forward.jsp の代替用 CommonForwardTag を新規作成
	 * @og.rev 3.5.4.0 (2003/11/25) useGamenId で、BACK_GAMENID を使用する/しないを指定します。
	 * @og.rev 3.8.5.1 (2006/05/08) 書き込み許可属性 writable 追加
	 *
	 * @return  int 後続処理の指示
	 */
	@Override
	public int doStartTag() {
		if( writable && isWrite() ) {
			if( useGamenId ) { jspPrint( getHiddenTag() ); }
			return( EVAL_BODY_INCLUDE );
		}
		else {
			return( SKIP_BODY );
		}
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 3.5.4.0 (2003/11/25) 新規追加。
	 * @og.rev 3.8.5.1 (2006/05/08) 書き込み許可属性 writable 追加
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		useGamenId	= true;
		roles		= null;
		writable 	= true;			// 3.8.5.1 (2006/05/08)
	}

	/**
	 * guiInfo.isWrite() の結果をチェックして返します。
	 *
	 * @og.rev 3.5.4.0 (2003/11/25) 画面と別にロールズを与えられます。
	 * @og.rev 4.3.0.0 (2008/07/04) ロールモードマルチ対応
	 */
	private boolean isWrite() {
		final boolean rtnFlag;

		// 4.0.0 (2005/11/30) 指定のロールが、アクセス許可の場合は、出力します。
		if( roles != null ) {
//			rtnFlag = getUser().isAccess( roles ) ;
			rtnFlag = getUser().isAccess( roles ) ;
		}
		else {
			GUIInfo guiInfo = (GUIInfo)getSessionAttribute( HybsSystem.GUIINFO_KEY );
			rtnFlag = guiInfo.isWrite() ;		// 4.0.0 (2005/01/31)
		}

		return rtnFlag;
	}

	/**
	 * 隠し属性タグを取得します。
	 *
	 * ※内部コール用
	 *
	 * @og.rev 3.0.1.2 (2003/03/07) forward.jsp の代替用 CommonForwardTag を新規作成
	 * @og.rev 3.1.7.0 (2003/05/02) 画面ＩＤのセットで、自画面を、BACK_GAMENID 、飛び先を、GAMENID にする。
	 * @og.rev 3.1.8.0 (2003/05/16) BACK_GAMENID を、リクエスト情報ではなく、GUI.KEY を利用するように修正。
	 * @og.rev 3.5.5.5 (2004/04/23) hidden の出力に、XHTMLTag.hidden を使用します。
	 * @og.rev 3.8.8.2 (2007/01/26) hidden の出力に、GAMENID を使用します。
	 *
	 * @return  String 隠し属性タグ
	 */
	private final String getHiddenTag() {

		String requestURI = ((HttpServletRequest)getRequest()).getRequestURI();
		if( requestURI == null ) {
			return "";
		}

		int index2 = requestURI.lastIndexOf( '/' );
		String jspID   = requestURI.substring( index2 + 1 );
		String gamenId = getGUIInfoAttri( "KEY" );			// 4.0.0 (2005/01/31)

		StringBuilder strRet = new StringBuilder( HybsSystem.BUFFER_MIDDLE );
		strRet.append( XHTMLTag.hidden( "BACK_GAMENID",gamenId ) );
		strRet.append( XHTMLTag.hidden( "BACK_JSPID"  ,jspID   ) );
		strRet.append( XHTMLTag.hidden( "GAMENID"     ,gamenId ) );		// 3.8.8.2

		return strRet.toString();
	}

	/**
	 * 【TAG】GAMENID,BACK_GAMENID 属性を付加するかどうか(true/false)を設定します(初期値:true)。
	 *
	 * @og.tag
	 * "true"の場合は、BACK_GAMENID 、BACK_JSPID 属性を hidden で出力します。
	 * "false"の場合は、付加しません。
	 * これは、戻るリンクを作成する場合に利用されます。
	 * 初期値は、 "true"です。
	 * ボタンで遷移先がポップアップ画面の場合に、戻るリンクを出しても戻れない為、
	 * そのようなケースでは false にします。
	 *
	 * @og.rev 3.5.4.0 (2003/11/25) useGamenId のために、新規追加。
	 *
	 * @param	flag  BACK_GAMENID 属性を付加するかどうか
	 *          "true"(追加する)/その他(追加しない)
	 */
	public void setUseGamenId( final String flag ) {
		useGamenId = nval( getRequestParameter( flag ),useGamenId );
	}

	/**
	 * 【TAG】画面と別にロールズを与えます。
	 *
	 * @og.tag
	 * このロールズが与えられた場合は、画面のロールズではなく、
	 * 個別のロールズを使用して、表示の制御を行います。
	 *
	 * @og.rev 3.5.4.0 (2003/11/25) 新規追加。
	 *
	 * @param	rols  ロールズ
	 */
	public void setRoles( final String rols ) {
		roles = nval( getRequestParameter( rols ),roles );
	}

	/**
	 * 【TAG】書き込み可能かどうか(true/false)を設定します(初期値:true)。
	 *
	 * @og.tag
	 * <p>書込み許可する場合は, true、許可しない場合は, true 以外を設定します。<br />
	 * デフォルト値は、互換性のため、true です。<br />
	 * ここで、false を設定するとBODY部の内容は表示されません。true を設定した場合、
	 * 通常通り BODY 部は表示されます。
	 * これは、roles の簡易版として、外部から ON/OFF を制御するのに使用できます。</p>
	 *
	 * @og.rev 3.8.5.1 (2006/05/08) 書き込み許可属性 writable 追加
	 *
	 * @param	flag 書込み許可(true)／書込み禁止（true 以外)
	 */
	public void setWritable( final String flag ) {
		writable = nval( getRequestParameter( flag ),writable );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "useGamenId"	,useGamenId	)
				.println( "roles"		,roles		)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
