/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.process;

import org.opengion.fukurou.util.Argument;
import org.opengion.fukurou.util.SystemParameter;
import org.opengion.fukurou.util.LogWriter;

import org.opengion.fukurou.util.HybsEntry ;
import org.opengion.fukurou.util.Closer;
import org.opengion.fukurou.db.ConnectionFactory;

import java.util.Map ;
import java.util.LinkedHashMap ;
import java.util.Locale ;

import java.sql.Connection;
import java.sql.Statement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;

/**
 * Process_DBReaderは、データベースから読み取った内容を、LineModel に設定後、
 * 下流に渡す、FirstProcess インターフェースの実装クラスです。
 *
 * データベースから読み取った内容より、LineModelを作成し、下流（プロセス
 * チェインは、チェインしているため、データは上流から下流へと渡されます。）
 * に渡します。ここで指定できるのは、検索系SQL のみです。
 *
 * データベース接続先等は、ParamProcess のサブクラス(Process_DBParam)に
 * 設定された接続(Connection)を使用します。
 *
 * 引数文字列中にスペースを含む場合は、ダブルコーテーション("") で括って下さい。
 * 引数文字列の 『=』の前後には、スペースは挟めません。必ず、-key=value の様に
 * 繋げてください。
 *
 * SQL文には、{&#064;SYS.YMDH}等のシステム変数が使用できます。
 * 現時点では、{&#064;SYS.YMD}、{&#064;SYS.YMDH}、{&#064;SYS.HMS} が指定可能です。
 *
 * @og.formSample
 *  Process_DBReader -dbid=DBGE -sql="select * from GEA08"
 *
 *   [ -dbid=DB接続ID           ] ： -dbid=DBGE (例: Process_DBParam の -configFile で指定する DBConfig.xml ファイルで規定)
 *   [ -sql=検索SQL文           ] ： -sql="select * from GEA08"
 *   [ -sqlFile=検索SQLファイル ] ： -sqlFile=select.sql
 *                                      -sql= を指定しない場合は、ファイルで必ず指定してください。
 *   [ -sql_XXXX=固定値         ] ： -sql_SYSTEM_ID=GE
 *                                     SQL文中の{&#064;XXXX}文字列を指定の固定値で置き換えます。
 *                                     WHERE SYSTEM_ID='{&#064;SYSTEM_ID}' ⇒ WHERE SYSTEM_ID='GE'
 *   [ -fetchSize=100 ]           ：フェッチする行数(初期値:100)
 *   [ -display=false|true ]      ：結果を標準出力に表示する(true)かしない(false)か（初期値 false:表示しない)
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Process_DBReader extends AbstractProcess implements FirstProcess {
	private static final String SQL_KEY  = "sql_" ;

	private Connection	connection	= null;
	private Statement	stmt		= null ;
	private ResultSet	resultSet	= null;
	private LineModel	newData		= null;
	private int			count		= 0;
	private int			fetchSize	= 100;

	private String		dbid		= null;
	private boolean		display		= false;	// 表示しない

	private final static Map<String,String> mustProparty   ;		// ［プロパティ］必須チェック用 Map
	private final static Map<String,String> usableProparty ;		// ［プロパティ］整合性チェック Map

	static {
		mustProparty = new LinkedHashMap<String,String>();

		usableProparty = new LinkedHashMap<String,String>();
		usableProparty.put( "dbid",	"Process_DBParam の -configFile で指定する DBConfig.xml ファイルで規定" );
		usableProparty.put( "sql",		"検索SQL文（sql or sqlFile 必須）例: \"select * from GEA08\"" );
		usableProparty.put( "sqlFile",	"検索SQLファイル（sql or sqlFile 必須）例: select.sql" );
		usableProparty.put( "sql_",		"SQL文中の{&#064;XXXX}文字列を指定の固定値で置き換えます。" +
									CR + "WHERE SYSTEM_ID='{&#064;SYSTEM_ID}' ⇒ WHERE SYSTEM_ID='GE'" );
		usableProparty.put( "fetchSize","フェッチする行数 (初期値:100)" );
		usableProparty.put( "display",	"結果を標準出力に表示する(true)かしない(false)か" +
										CR + "(初期値 false:表示しない)" );
	}

	/**
	 * デフォルトコンストラクター。
	 * このクラスは、動的作成されます。デフォルトコンストラクターで、
	 * super クラスに対して、必要な初期化を行っておきます。
	 *
	 */
	public Process_DBReader() {
		super( "org.opengion.fukurou.process.Process_DBReader",mustProparty,usableProparty );
	}

	/**
	 * プロセスの初期化を行います。初めに一度だけ、呼び出されます。
	 * 初期処理（ファイルオープン、ＤＢオープン等）に使用します。
	 *
	 * @param   paramProcess ParamProcess
	 */
	public void init( final ParamProcess paramProcess ) {
		Argument arg = getArgument();

		String sql		=arg.getFileProparty("sql","sqlFile",true);
		String fSize	=arg.getProparty("fetchSize");
		display			=arg.getProparty("display",display);

		dbid			= arg.getProparty("dbid");
		connection		= paramProcess.getConnection( dbid );

		// 3.8.0.1 (2005/06/17) SQL文の {@XXXX} 文字列の固定値への置き換え
		HybsEntry[] entry	=arg.getEntrys(SQL_KEY);		//配列
		SystemParameter sysParam = new SystemParameter( sql );
		sql = sysParam.replace( entry );

		// SQL文の {@XXXX} 文字列の固定値への置き換え
		if( fSize != null ) { fetchSize = Integer.parseInt( fSize ); }

		try {
			stmt = connection.createStatement();
			if( fetchSize > 0 ) { stmt.setFetchSize( fetchSize ); }
			resultSet = stmt.executeQuery( sql );

			newData = createLineModel( resultSet );

			if( display ) { println( newData.nameLine() ); }
		}
		catch (SQLException ex) {
			String errMsg = "Query の実行に問題があります。[" + sql + "]" ;
			throw new RuntimeException( errMsg,ex );
		}
	}

	/**
	 * プロセスの終了を行います。最後に一度だけ、呼び出されます。
	 * 終了処理（ファイルクローズ、ＤＢクローズ等）に使用します。
	 *
	 * @og.rev 4.0.0.0 (2007/11/27) commit,rollback,remove 処理を追加
	 *
	 * @param   isOK トータルで、OKだったかどうか(true:成功/false:失敗）
	 */
	public void end( final boolean isOK ) {
		boolean flag1 = Closer.resultClose( resultSet );
		resultSet  = null;
		boolean flag2 = Closer.stmtClose( stmt );
		stmt       = null;

		ConnectionFactory.remove( connection,dbid );

		if( !flag1 || !flag2 ) {
			String errMsg = "ステートメントをクローズ出来ません。";
			throw new RuntimeException( errMsg );
		}
	}

	/**
	 * このデータの処理において、次の処理が出来るかどうかを問い合わせます。
	 * この呼び出し１回毎に、次のデータを取得する準備を行います。
	 *
	 * @return	boolean 処理できる:true / 処理できない:false
	 */
	public boolean next() {
		try {
			return resultSet.next() ;
		}
		catch (SQLException ex) {
			String errMsg = "ネクストすることが出来ません。";
			throw new RuntimeException( errMsg,ex );
		}
	}

	/**
	 * 最初に、 行データである LineModel を作成します
	 * FirstProcess は、次々と処理をチェインしていく最初の行データを
	 * 作成して、後続の ChainProcess クラスに処理データを渡します。
	 *
	 * @param   rowNo int 処理中の行番号
	 * @return  LineModel  処理変換後のLineModel
	 */
	public LineModel makeLineModel( final int rowNo ) {
		count++ ;
		try {
			for(int clm = 0; clm < newData.size(); clm++) {
				Object obj = resultSet.getObject(clm+1);
				if( obj == null ) {
		//			newData.setValue( clm, "" );
					newData.setValue( clm, null );
				}
				else {
					newData.setValue( clm, obj );
				}
			}
			newData.setRowNo( rowNo );
			if( display ) { println( newData.dataLine() ); }
		}
		catch (SQLException ex) {
			String errMsg = "データを処理できませんでした。[" + rowNo + "]件目 "
								+ newData.toString() ;
			throw new RuntimeException( errMsg,ex );
		}
		return newData;
	}

	/**
	 * 内部で使用する LineModel を作成します。
	 * このクラスは、プロセスチェインの基点となりますので、新規 LineModel を返します。
	 * Exception 以外では、必ず LineModel オブジェクトを返します。
	 *
	 * @param   rs ResultSet データベースカーソル（リザルトセット）
	 * @return  LineModel  データベースから取り出して変換した LineModel
	 * @throws RuntimeException カラム名を取得できなかった場合。
	 */
	private LineModel createLineModel( final ResultSet rs ) {
		LineModel model = new LineModel();

		try {
			ResultSetMetaData metaData	= rs.getMetaData();

			int size =  metaData.getColumnCount();
			model.init( size );

			for(int clm = 0; clm < size; clm++) {
				String name = (metaData.getColumnLabel(clm+1)).toUpperCase(Locale.JAPAN) ;
				model.setName( clm,name );
			}
		}
		catch (SQLException ex) {
			String errMsg = "ResultSetMetaData から、カラム名を取得できませんでした。";
			throw new RuntimeException( errMsg,ex );
		}
		return model;
	}

	/**
	 * プロセスの処理結果のレポート表現を返します。
	 * 処理プログラム名、入力件数、出力件数などの情報です。
	 * この文字列をそのまま、標準出力に出すことで、結果レポートと出来るような
	 * 形式で出してください。
	 *
	 * @return   処理結果のレポート
	 */
	public String report() {
		String report = "[" + getClass().getName() + "]" + CR
				+ TAB + "DBID        : " + dbid + CR
				+ TAB + "Input Count : " + count ;

		return report ;
	}

	/**
	 * このクラスの使用方法を返します。
	 *
	 * @return	String このクラスの使用方法
	 */
	public String usage() {
		StringBuilder buf = new StringBuilder();

		buf.append( "Process_DBReaderは、データベースから読み取った内容を、LineModel に設定後、"	).append( CR );
		buf.append( "下流に渡す、FirstProcess インターフェースの実装クラスです。"					).append( CR );
		buf.append( CR );
		buf.append( "データベースから読み取った内容より、LineModelを作成し、下流（プロセス"			).append( CR );
		buf.append( "チェインは、チェインしているため、データは上流から下流へと渡されます。）"		).append( CR );
		buf.append( "に渡します。ここで指定できるのは、検索系SQL のみです。"						).append( CR );
		buf.append( CR );
		buf.append( "データベース接続先等は、ParamProcess のサブクラス(Process_DBParam)に"			).append( CR );
		buf.append( "設定された接続(Connection)を使用します。"										).append( CR );
		buf.append( CR );
		buf.append( "引数文字列中に空白を含む場合は、ダブルコーテーション(\"\") で括って下さい。"	).append( CR );
		buf.append( "引数文字列の 『=』の前後には、空白は挟めません。必ず、-key=value の様に"		).append( CR );
		buf.append( "繋げてください。"																).append( CR );
		buf.append( CR );
		buf.append( "SQL文には、{@SYS.YMDH}等のシステム変数が使用できます。"						).append( CR );
		buf.append( "現時点では、{@SYS.YMD}、{@SYS.YMDH}、{@SYS.HMS} が指定可能です。"				).append( CR );
		buf.append( CR ).append( CR );

		buf.append( getArgument().usage() ).append( CR );

		return buf.toString();
	}

	/**
	 * このクラスは、main メソッドから実行できません。
	 *
	 * @param	args String[]
	 */
	public static void main( final String[] args ) {
		LogWriter.log( new Process_DBReader().usage() );
	}
}
