/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.fukurou.util.FileString;

import org.opengion.fukurou.util.StringUtil ;
import static org.opengion.fukurou.util.StringUtil.nval ;

import java.io.IOException;
import java.io.File;

/**
 * 各種アクションを指定して、ファイル関連の操作をおこなうタグです。
 *
 * 各種アクション に応じた振る舞いを行います。
 * 結果については、false の場合は,body 要素を表示して、終了します。
 * 判定結果を反転したい場合は、notEquals 属性を使用してください。また、
 * 結果に応じて、処理を止めたくない場合は、useStop 属性を false に指定することで、
 * 後続処理を実行できます。
 *
 * [各種アクション]
 * canRead          読み込めるかどうかを判定。
 * canWrite         変更できるかどうか判定。
 * createNewFile    空の新しいファイルを不可分 (atomic) に生成。(そのファイルがまだ存在しない場合だけ)
 * delete           ファイルまたはディレクトリを削除。
 * renameTo         ファイルまたはディレクトリ名を変更。
 * exists           ファイルが存在するかどうか判定。
 * isDirectory      ファイルがディレクトリであるかどうか判定。
 * isFile           ファイルが普通のファイルかどうか判定。
 * isHidden         ファイルが隠しファイルかどうか判定。
 * mkdir            ディレクトリを生成。
 * mkdirs           ディレクトリを複数生成。
 * read             ファイルを読み込んでjspWriterに出力
 * existsLength     ファイルサイズが０Byte以上のファイルが存在するかどうか判定。
 *
 * @og.formSample
 * ●形式：&lt;og:file action="…" fileURL="…" &gt;･･･&lt;/og:file&gt;
 * ●body：あり
 *
 * ●使用例
 *    ・ファイルの存在チェック→存在しなければエラーメッセージを表示。
 *        &lt;og:file action="exists" fileURL="N:/CIR/" file1="{&#064;USER.LKISB}/{&#064;USER.LDNO1KAI}.cir/001.sht"&gt;
 *            &lt;og:message msg="RKE_0157" comment="回路図が存在しません。" /&gt;
 *        &lt;/og:file&gt;
 *
 *    ・N:/Filetemp/にユーザーディレクトリが存在しなければ作成。→失敗した場合エラーメッセージを表示。
 *        &lt;og:file action="mkdir" fileURL="N:/Filetemp/{&#064;USER.ID}" &gt;
 *            &lt;og:message comment="エラーが発生しました。システム管理者に連絡してください。" /&gt;
 *        &lt;/og:file&gt;
 *
 *    ・N:/Filetemp/test.txt ファイルの削除。ファイルが存在しなくても処理を続ける。
 *        &lt;og:file action="delete" fileURL="N:/Filetemp/" file1="test.txt" useStop="false" &gt;
 *            &lt;og:message comment="ファイルは存在しませんでした。" /&gt;
 *        &lt;/og:file&gt;
 *
 * @og.group その他部品
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class FileTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/01/31)

	/** action 引数に渡す事の出来る アクションコマンド  読み込めるかどうか {@value} */
	public static final String ACT_CANREAD	= "canRead" ;
	/** action 引数に渡す事の出来る アクションコマンド  変更できるかどうか {@value} */
	public static final String ACT_CANWRITE    = "canWrite" ;
	/** action 引数に渡す事の出来る アクションコマンド  空の新しいファイルを不可分 (atomic) に生成します (そのファイルがまだ存在しない場合だけ {@value} */
	public static final String ACT_CREATENEWFILE	= "createNewFile" ;
	/** action 引数に渡す事の出来る アクションコマンド  ファイルまたはディレクトリを削除{@value}	*/
	public static final String ACT_DELETE	 = "delete" ;
	/** action 引数に渡す事の出来る アクションコマンド  ファイルが存在するかどうか {@value} */
	public static final String ACT_EXISTS	 = "exists" ;
	/** action 引数に渡す事の出来る アクションコマンド  ファイルがディレクトリであるかどうか{@value}	*/
	public static final String ACT_ISDIRECTORY	  = "isDirectory" ;
	/** action 引数に渡す事の出来る アクションコマンド  ファイルが普通のファイルかどうか{@value}	*/
	public static final String ACT_ISFILE	 = "isFile" ;
	/** action 引数に渡す事の出来る アクションコマンド  ファイルが隠しファイルかどうか {@value} */
	public static final String ACT_ISHIDDEN    = "isHidden" ;
	/** action 引数に渡す事の出来る アクションコマンド  ディレクトリを生成します。 {@value} */
	public static final String ACT_MKDIR	= "mkdir" ;
	/** action 引数に渡す事の出来る アクションコマンド  ディレクトリを生成します。 {@value} */
	public static final String ACT_MKDIRS	= "mkdirs" ;
	/** action 引数に渡す事の出来る アクションコマンド  ファイル名を変更します。 {@value} */
	public static final String ACT_RENAMETO	= "renameTo" ;		// 3.5.6.5 (2004/08/09)
	/** action 引数に渡す事の出来る アクションコマンド  ファイルを読み込んで表示します。 {@value} */
	public static final String ACT_READ	= "read" ;		// 3.6.0.0 (2004/09/25)
	/** action 引数に渡す事の出来る アクションコマンド  ファイルサイズが０Byte以上のファイルが存在するかどうか判定。 {@value} */
	public static final String ACT_EXISTSLENGTH	 = "existsLength" ;		// 3.8.5.2 (2006/05/31)

	/** action 引数に渡す事の出来る コマンド リスト  */
	private static final String[] ACTION_LIST = new String[] {
		ACT_CANREAD , ACT_CANWRITE , ACT_CREATENEWFILE , ACT_DELETE , ACT_EXISTS , ACT_ISDIRECTORY ,
		ACT_ISFILE , ACT_ISHIDDEN , ACT_MKDIR , ACT_MKDIRS , ACT_RENAMETO , ACT_READ , ACT_EXISTSLENGTH };

	private String	fileURL 	= HybsSystem.sys( "FILE_URL" );
	private String	file1		= "";
	private String	file2		= null;
	private String	action		= null;
	private boolean rtnCode		= false;

	private boolean notEquals	= false;	// 3.8.5.2 (2006/05/31) 判定結果を反転させて処理します。
	private boolean useStop		= true;		// 3.8.5.2 (2006/05/31) エラー時BODYを処理後に停止(true)するかどうか

	private String encode		= null;		// 5.1.9.0 (2010/08/01) READ時のエンコード指定

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 3.6.0.0 (2004/09/25) file オブジェクトの作成を actionExec 移動
	 * @og.rev 3.8.5.2 (2006/05/31) notEquals追加。 判定結果を反転させて処理します。
	 *
	 * @return  int 後続処理の指示
	 */
	@Override
	public int doStartTag() {

		try {
			rtnCode = notEquals ^ actionExec( action );		// 3.8.5.2 (2006/05/31) 排他的論理和（XOR)
		}
		catch( IOException ex ) {
			String errMsg = "指定のアクションは実行できません。アクションエラー"
							+ HybsSystem.CR
							+ "action=[" + action + "] , "
							+ "  fileURL=[" + fileURL + "]"
							+ "  file1=[" + file1 + "]"
							+ "  file2=[" + file2 + "]" ;
			throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}

		if( rtnCode ) { return(SKIP_BODY); }			// Body を評価しない
		else {			return( EVAL_BODY_INCLUDE ); }	// Body インクルード（ extends TagSupport 時）
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.8.5.2 (2006/05/31) useStop 追加。 エラー時BODYを処理後に停止(true)するかどうか
	 *
	 * @return  int 後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		int rtn = ( useStop && !rtnCode ) ? SKIP_PAGE : EVAL_PAGE ;

		return( rtn );
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.6.0.0 (2004/09/24) columns 、tableId 、file 削除
	 * @og.rev 3.8.5.2 (2006/05/31) notEquals 、useStop 追加
	 * @og.rev 5.1.9.0 (2010/08/01) READ時のエンコード指定
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		fileURL 	= HybsSystem.sys( "FILE_URL" );
		file1		= "";
		file2		= null;
		action		= null;
		rtnCode		= false;
		notEquals	= false;	// 3.8.5.2 (2006/05/31) 判定結果を反転させて処理します。
		useStop		= true;		// 3.8.5.2 (2006/05/31) エラー時BODYを処理後に停止(true)するかどうか
		encode		= null;		// 5.1.9.0 (2010/08/01) READ時のエンコード指定
	}

	/**
	 * アクションを実行します。
	 * アクションは,指定のアクションコマンドに対応する処理を入力データに
	 * 対して行います。
	 *
	 * @og.rev 3.0.0.0 (2002/12/25) ACTION_LIST のチェックを削除
	 * @og.rev 3.6.0.0 (2004/09/25) ACT_read を追加 , file オブジェクトを移動
	 * @og.rev 3.8.5.2 (2006/05/31) existsLength 追加
	 * @og.rev 4.0.0.0 (2007/11/28) メソッドの戻り値をチェックします。
	 * @og.rev 5.1.9.0 (2010/08/01) READ時のエンコード指定
	 *
	 * @param	action アクションコマンド（public static final 宣言されている文字列)
	 * @return	実行後のデータ
	 */
	private boolean actionExec( final String action ) throws IOException {
		String directory = HybsSystem.url2dir( fileURL );
		File file = new File( StringUtil.urlAppend( directory,file1 ) );

		boolean rtnVal = false;
		if( action != null ) {
			if( 	 ACT_CANREAD.equalsIgnoreCase(		 action ) ) { rtnVal = file.canRead(); }
			else if( ACT_CANWRITE.equalsIgnoreCase( 	 action ) ) { rtnVal = file.canWrite(); }
			else if( ACT_CREATENEWFILE.equalsIgnoreCase( action ) ) { rtnVal = file.createNewFile(); }
			else if( ACT_DELETE.equalsIgnoreCase(		 action ) ) { rtnVal = file.delete(); }
			else if( ACT_EXISTS.equalsIgnoreCase(		 action ) ) { rtnVal = file.exists(); }
			else if( ACT_ISDIRECTORY.equalsIgnoreCase(	 action ) ) { rtnVal = file.isDirectory(); }
			else if( ACT_ISFILE.equalsIgnoreCase(		 action ) ) { rtnVal = file.isFile(); }
			else if( ACT_ISHIDDEN.equalsIgnoreCase( 	 action ) ) { rtnVal = file.isHidden(); }
			else if( ACT_MKDIR.equalsIgnoreCase(		 action ) ) {
				if( file.isDirectory() ) { rtnVal = true; }
				else { rtnVal = file.mkdir(); }
			}
			else if( ACT_MKDIRS.equalsIgnoreCase(		 action ) ) { rtnVal = file.mkdirs(); }
			else if( ACT_RENAMETO.equalsIgnoreCase(		 action ) ) {
				if( file2 != null ) {
					File newFile = new File( StringUtil.urlAppend( directory,file2 ) );
//					if( newFile.exists() ) { newFile.delete(); }
					if( newFile.exists() && !newFile.delete() ) {
						String errMsg = "所定のファイルを削除できませんでした。[" + newFile + "]" ;
						throw new RuntimeException( errMsg );
					}
					rtnVal = file.renameTo( newFile );
				}
			}
			// 3.6.0.0 (2004/09/25) ACT_read を追加
			else if( ACT_READ.equalsIgnoreCase(			 action ) ) {
				if( file.isFile() ) {
					FileString fs = new FileString();
					fs.setFilename( StringUtil.urlAppend( directory,file1 ) );
					if( encode != null ) { fs.setEncode( encode ); } // 5.1.9.0 (2010/08/01) READ時のエンコード指定
					String val = fs.getValue();

					jspPrint( nval( getRequestParameter( val ),"" ) );
					rtnVal = true;
				}
				else {
					String errMsg = "ファイルが存在しないか、ファイルではありません。"
							+ HybsSystem.CR
							+ "action=[" + action + "] , "
							+ "  fileURL=[" + fileURL + "]"
							+ "  directory=[" + directory + "]"
							+ "  file1=[" + file1 + "]" ;
					throw new HybsSystemException( errMsg );
				}
			}
			// 3.8.5.2 (2006/05/31) ファイルサイズが０Byte以上のファイルが存在するかどうか判定。
			else if( ACT_EXISTSLENGTH.equalsIgnoreCase( action ) ) {
				rtnVal = ( file.exists() && file.length() > 0L );
			}
		}
		else {
			String errMsg = "アクションが指定されていません。アクション NULL エラー"
							+ HybsSystem.CR
							+ "  file=[" + file1 + "]" ;
			throw new HybsSystemException( errMsg );
		}

		return rtnVal;
	}

	/**
	 * 【TAG】アクション(canRead,canWrite,createNewFile,delete,exists,isDirectory,isFile,isHidden,mkdir,mkdirs)を指定します。
	 *
	 * @og.tag
	 * アクションは,HTMLから（get/post)指定されますので,ACT_xxx で設定される
	 * フィールド定数値のいづれかを、指定できます。
	 *
	 * canRead          読み込めるかどうかを判定。
	 * canWrite         変更できるかどうか判定。
	 * createNewFile    空の新しいファイルを不可分 (atomic) に生成。(そのファイルがまだ存在しない場合だけ)
	 * delete           ファイルまたはディレクトリを削除。
	 * renameTo         ファイルまたはディレクトリ名を変更。
	 * exists           ファイルが存在するかどうか判定。
	 * isDirectory      ファイルがディレクトリであるかどうか判定。
	 * isFile           ファイルが普通のファイルかどうか判定。
	 * isHidden         ファイルが隠しファイルかどうか判定。
	 * mkdir            ディレクトリを生成。
	 * mkdirs           ディレクトリを複数生成。
	 * read             ファイルを読み込んでjspWriterに出力
	 * existsLength     ファイルサイズが０Byte以上のファイルが存在するかどうか判定。
	 *
	 * @og.rev 3.0.0.0 (2002/12/25) ACTION_LIST のチェックを導入
	 * @og.rev 3.5.6.2 (2004/07/05) 文字列の連結にStringBuilderを使用します。
	 *
	 * @param	cmd アクション文字列
	 * @see		<a href="{@docRoot}/constant-values.html#org.opengion.hayabusa.taglib.FileTag.ACT_canRead">アクション定数</a>
	 */
	public void setAction( final String cmd ) {
		action = getRequestParameter( cmd );

		if( ! check( action, ACTION_LIST ) ) {

			StringBuilder errMsg = new StringBuilder( HybsSystem.BUFFER_MIDDLE );
			errMsg.append( "指定のアクションは実行できません。アクションエラー" );
			errMsg.append( HybsSystem.CR );
			errMsg.append( "action=[" ).append( action ).append( "] " );
			errMsg.append( HybsSystem.CR );

			for( int i=0; i<ACTION_LIST.length; i++ ) {
				errMsg.append( " | " );
				errMsg.append( ACTION_LIST[i] );
			}
			errMsg.append( " | " );
			throw new HybsSystemException( errMsg.toString() );
		}
	}

	/**
	 * 【TAG】操作するファイルのディレクトリを指定します(初期値:システムパラメータのFILE_URL)。
	 *
	 * @og.tag
	 * この属性で指定されるディレクトリのファイルを操作します。
	 * 指定方法は、通常の fileURL 属性と同様に、先頭が、'/' （UNIX) または、２文字目が、
	 * ":" （Windows）の場合は、指定のURLそのままのディレクトリに、そうでない場合は、
	 * システムパラメータ の FILE_URL 属性で指定のフォルダの下を、使用します。
	 *
	 * @og.rev 4.0.0 (2005/01/31) urlAppend メソッドの利用
	 * @og.rev 4.0.0.0 (2007/11/20) 指定されたディレクトリ名の最後が"\"or"/"で終わっていない場合に、"/"を付加する。
	 *
	 * @param	url ファイルURL
	 */
	public void setFileURL( final String url ) {
		String furl = nval( getRequestParameter( url ),null );
		if( furl != null ) {
			char ch = furl.charAt( furl.length()-1 );
			if( ch != '/' && ch != '\\' ) { furl = furl + "/"; }
			fileURL = StringUtil.urlAppend( fileURL,furl );
		}
	}

	/**
	 * 【TAG】基準となるファイル名を指定します(コマンドの左辺のファイル名です)。
	 *
	 * @og.tag
	 * コマンドの左辺のファイル名です。
	 *
	 * @param	fname ファイル名１
	 */
	public void setFile1( final String fname ) {
		file1 = nval( getRequestParameter( fname ),file1 );
	}

	/**
	 * 【TAG】処理結果となるファイル名を指定します(コマンドの右辺のファイル名です)。
	 *
	 * @og.tag
	 * コマンドの右辺のファイル名です。
	 *
	 * @param	fname ファイル名２
	 */
	public void setFile2( final String fname ) {
		file2 = nval( getRequestParameter( fname ),file2 );
	}

	/**
	 * 【TAG】判定結果を反転させるかどうか(true/false)指定します(初期値:false)。
	 *
	 * @og.tag
	 * <p>通常の判定結果において、不成立(false)の場合に、BODY を実行します。
	 * 通常の処理結果の正反対の処理を行います。
	 * デフォルトは、通常 (true 以外)です。</p>
	 *
	 * @og.rev 3.8.5.2 (2006/05/31) 新規追加
	 *
	 * @param   flag 反転する ("true")／通常(それ以外)
	 */
	public void setNotEquals( final String flag ) {
		notEquals = nval( getRequestParameter( flag ),notEquals );
	}

	/**
	 * 【TAG】エラー時BODYを処理後に停止するかどうかを指定します(初期値:true)。
	 *
	 * @og.tag
	 * 処理結果などに応じて、以下の処理を停止したい場合に、使用します。
	 * 通常は、条件を判定後、false の場合に、BODY部を出力(処理)した後に、
	 * 処理を停止します。(useStop="true")
	 * false を指定すると、判定結果に無関係に、以下の処理を実行します。
	 * 処理は継続したいが、警告表示する場合に、useStop="false" を指定します。
	 * 初期値は、停止する ("true")です。
	 *
	 * @og.rev 3.8.5.2 (2006/05/31) 新規追加
	 *
	 * @param   flag 判定する ("true")／しない(それ以外)
	 */
	public void setUseStop( final String flag ) {
		useStop = nval( getRequestParameter( flag ),useStop );
	}

	/**
	 * 【TAG】ファイルを読み込む(action="READ")際のエンコードを指定します。(初期値:OS依存文字コード)。
	 *
	 * @og.tag
	 * ファイルを読み込む(action="READ")際のエンコードを指定します。
	 * action="READ"以外場合には、この属性値は利用されません。
	 * 指定しない場合は、OS依存文字コードで読み込まれます。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) 新規作成
	 *
	 * @param   enc ファイル読み込みのエンコード
	 */
	public void setEncode( final String enc ) {
		encode = nval( getRequestParameter( enc ),encode );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "fileURL"		,fileURL	)
				.println( "file1"		,file1		)
				.println( "file2"		,file2		)
				.println( "action"		,action		)
				.println( "rtnCode"		,rtnCode	)
				.println( "notEquals"	,notEquals	)
				.println( "useStop"		,useStop	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
