/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.resource.GUIInfo;
import org.opengion.fukurou.db.DBUtil;
import org.opengion.fukurou.util.Attributes;
import org.opengion.fukurou.util.XHTMLTag;

import static org.opengion.fukurou.util.StringUtil.nval ;

import java.io.File;

/**
 * 画面ＩＤと同じヘルプファイルがあればリンクを作成するタグです(通常は query.jsp に組込み)。
 *
 * ヘルプファイルは、システムパラメータ の HELP_URL で定義されているhelpフォルダに配置します。
 * このフォルダに、画面IDと同じファイル(例えば、GE0001.html など)があれば、リンクを作成します。
 * ファイルがなければ、リンクは表示されません。
 * メッセージの表示の制御は、viewMsg 属性で指定します。(false でファイルが存在した場合のみ表示）
 * ファイルの拡張子も指定できますが、一般に、html でヘルプファイルを作成するほうが
 * すばやく表示できます。
 * また、og:topMenuタグ内にこのタグを記述することで、各画面分類に対するヘルプを表示することが
 * できるようになります。
 * (この場合も、画面分類のキーがヘルプファイルのキーになります)
 *
 * @og.formSample
 * ●形式：一般ユーザーが直接組み込むことはありません。
 * ●body：なし
 *
 * ●使用例
 *     &lt;og:help guiInfoKey="{&#064;GUI.KEY}" msg="HELP" /&gt;
 *
 *     &lt;og:help
 *        guiInfoKey    = "GUIInfo のキーを指定します。(必須)"
 *        extension     = "拡張子を指定します(初期値:html)。"
 *        msg           = "メッセージリソースのメッセージIDを指定します。"
 *        target        = "TARGET 属性を指定します(初期値:_blank)。"
 *        viewMsg       = "メッセージを常時表示させるかどうか(true/false)を指定します(初期値:false)。"
 *	      iconURL       = "ヘルプアイコンのURL(初期値:/image/help.png)"; // 5.3.8.0 (2011/08/01)
 *     /&gt;
 *
 * @og.group メニュー制御
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class HelpTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/01/31)

	private static final String	JSP = HybsSystem.sys( "JSP" );
	
	private String	guiInfoKey	= null;
	private String	extension	= "html";
	private String  baseURL		= HybsSystem.sys( "HELP_URL" );
	private String	target		= "_blank";		// 3.6.0.7 (2004/11/06)
	private boolean	viewMsg		= false;
//	private String	iconURL		= "/image/help.png";	// 5.3.8.0 (2011/08/01)
	private String  iconURL		= HybsSystem.sys( "DEFAULT_HELP_ICON" ); // 5.4.3.6 (2012/01/19)
	private String  faqIconURL	= HybsSystem.sys( "DEFAULT_FAQ_ICON" ); // 5.5.0.4 (2012/03/16)
	private boolean  useFaq		= HybsSystem.sysBool( "USE_GUI_FAQ " ); // 5.5.0.4 (2012/03/16)
	private String  faqGUI		= HybsSystem.sys( "DEFAULT_FAQ_GUI" ); // 5.5.0.4 (2012/03/16)
	private String  syscode		= "*"; // 5.5.0.4 (2012/03/16)


	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 5.3.9.0 (2011/09/01) メニューでのヘルプアイコン対応
	 * @og.rev 5.5.0.4 (2012/03/16) FAQ対応
	 *
	 * @return  int 後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)

		TopMenuTag topMenu = (TopMenuTag)findAncestorWithClass( this,TopMenuTag.class );
		if( topMenu == null ) {
			jspPrint( makeTag() );
			if(useFaq){
				jspPrint( makeTagFaq() );
			}
		}
		else {
			// 5.3.9.0 (2011/09/01) メニューでのヘルプアイコン対応
			String linkFormat = getLink( baseURL + "{FILENAME}" );
			String baseDir = HybsSystem.url2dir( baseURL );
			topMenu.add( "helpLinkFormat",linkFormat );
			topMenu.add( "helpBaseDir",baseDir );
			if(useFaq){ // 5.5.0.4 (2012/03/16) FAQ対応
				GUIInfo guiInfo = getGUIInfo( faqGUI );		
				if( guiInfo != null ) {
					String address = guiInfo.getRealAddress( get( "href" ) );
					String faqFormat = getFAQLink(getRequestParameter( address+"?command=NEW&GAMENID="+faqGUI+"&KNRNGUI={GUIKEY}" ));
					topMenu.add( "faqLinkFormat",faqFormat );
				}
			}
		}

		return(EVAL_PAGE);
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.0.0.3 (2003/02/21) ターゲット属性の新規追加他
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.6.0.7 (2004/11/06) target 属性の初期値を _new から _blank に変更
	 * @og.rev 5.3.8.0 (2011/08/01) iconURL追加
	 * @og.rev 5.5.0.4 (2012/03/16) faq
	 */
	@Override
	protected void release2() {
		super.release2();
		guiInfoKey	= null;
		extension	= "html";
		baseURL		= HybsSystem.sys( "HELP_URL" );
		target		= "_blank";		// 3.6.0.7 (2004/11/06)
		viewMsg		= false;
//		iconURL		= "/image/help.png";	// 5.3.8.0 (2011/08/01)
		iconURL		= HybsSystem.sys( "DEFAULT_HELP_ICON" ); // 5.4.3.6 (2012/01/19)
		faqIconURL	= HybsSystem.sys( "DEFAULT_FAQ_ICON" ); // 5.5.0.4 (2012/03/16)
		useFaq		= HybsSystem.sysBool( "USE_GUI_FAQ " ); // 5.5.0.4 (2012/03/16)
		faqGUI		= HybsSystem.sys( "DEFAULT_FAQ_GUI" ); // 5.5.0.4 (2012/03/16)
		syscode		= "*"; // 5.5.0.4 (2012/03/16)
	}

	/**
	 * HELPリンクを作成します。
	 *
	 * @og.rev 3.0.0.3 (2003/02/21) ターゲット属性の新規追加
	 * @og.rev 3.0.1.0 (2003/03/03) viewMsg フラグの制御のバグ修正
	 * @og.rev 5.3.8.0 (2011/08/01) iconURL対応
	 * @og.rev 5.3.9.0 (2011/09/01) メニューでのヘルプアイコン対応
	 * @og.rev 5.5.0.4 (2012/03/16) faq
	 *
	 * @return	リンクタグ文字列
	 */
	protected String makeTag() {
		String rtn = "";

		if( guiInfoKey == null ) {
			guiInfoKey = getGUIInfoAttri( "KEY" );
		}

		String url = baseURL + guiInfoKey + "." + extension;
		File  file = new File( HybsSystem.url2dir( url ) );

		// ファイルの存在チェック
		if( file.exists() ) {						// 3.5.6.0 (2004/06/18)
			// 5.3.9.0 (2011/09/01) メニューでのヘルプアイコン対応
//			Attributes attri = new Attributes();
//			String path = getContextPath();
//			attri.set( "href",path + "/" + url );
//			attri.set( "body",getMsglbl() );
//			attri.set( "body",getLinkBody() );
//			attri.set( "target",target );
//
//			rtn = XHTMLTag.link( attri ) ;
			rtn = getLink( url );
		}
		else if( viewMsg ) {
//			rtn = getMsglbl() ;
//			rtn = getLinkBody();
			rtn = getLinkBody(null,getMsglbl()); // 5.5.0.4
		}

		return rtn;
	}
	
	/**
	 * FAQリンクを作成します。
	 *
	 * @og.rev 5.3.9.0 (2011/09/01) メニューでのヘルプアイコン対応
	 *
	 * @return	リンクタグ文字列
	 */
	protected String makeTagFaq() {
		String rtn = "";

		if( guiInfoKey == null ) {
			guiInfoKey = getGUIInfoAttri( "KEY" );
		}

		GUIInfo guiInfo = getGUIInfo( faqGUI );		
		if( guiInfo == null ) { return rtn; }	// 見つからない場合は、アクセス不可

		String address = guiInfo.getRealAddress( get( "href" ) );
		String url = getRequestParameter( address+"?command=NEW&GAMENID="+faqGUI+"&KNRNGUI="+guiInfoKey );
		

		rtn = getFAQLink( url );

		return rtn;
	}

	/**
	 * リンク文字列を作成します。
	 * 
	 * @og.rev 5.3.9.0 (2011/09/01) 新規作成
	 * @og.rev 5.5.0.4 (2012/03/16) faq
	 *
	 * @return	リンク文字列
	 */
	private String getLink( final String url ) {
		Attributes attri = new Attributes();
		String path = getContextPath();
		attri.set( "href",path + "/" + url );
//		attri.set( "body",getMsglbl() );
//		attri.set( "body",getLinkBody() );
		attri.set( "body",getLinkBody(iconURL,getMsglbl()) ); //5.5.0.4 (2012/03/16)
		attri.set( "target",target );
		attri.set( "class", "helplink" );

		return XHTMLTag.link( attri );
	}
	
	/**
	 * FAQリンク文字列を作成します。
	 * 
	 * @og.rev 5.5.0.4 (2012/03/16) 新規作成
	 *
	 * @return	リンク文字列
	 */
	private String getFAQLink( final String url ) {
		Attributes attri = new Attributes();
		attri.set( "href", url );
		attri.set( "body",getLinkBody(faqIconURL,"FAQ") );
		attri.set( "target",target );
		attri.set( "class", "faqlink" );

		return XHTMLTag.link( attri );
	}

	/**
	 * リンクのボディー部分を作成します。
	 * 
	 * @og.rev 5.3.8.0 (2011/08/01) 新規作成
	 * @og.rev 5.3.9.0 (2011/09/01) 画像表示時にtitle属性を付加
	 * @og.rev 5.5.0.4 (2012/03/16) 引数対応
	 *
	 * @return	リンクボディー文字列
	 */
//	private String getLinkBody() {
	private String getLinkBody(final String icon, final String title) {
		String rtn = null;
//		if( iconURL == null || iconURL.length() == 0 ) {
		if( icon == null || icon.length() == 0 ) {
			rtn = getMsglbl();
		}
		else {
			rtn = "<img src=\"" + JSP + icon + "\" title=\"" + title + "\"/>";
		}
		return rtn;
	}

	/**
	 * 【TAG】GUIInfo のキーを指定します。
	 *
	 * @og.tag GUIInfo のキーを指定します。
	 *
	 * @param	key String
	 */
	public void setGuiInfoKey( final String key ) {
		guiInfoKey = getRequestParameter( key );
	}

	/**
	 * 【TAG】拡張子を指定します(初期値:html)。
	 *
	 * @og.tag
	 * なにも設定されていない場合は、"html" が初期値となります。
	 * ここでは、ピリオドは、含める必要はありません。
	 *
	 * @param	ext 拡張子
	 */
	public void setExtension( final String ext ) {
		extension = nval( getRequestParameter( ext ),extension );
	}

	/**
	 * 【TAG】TARGET 属性を指定します(初期値:_blank)。
	 *
	 * @og.tag
	 * 初期値は、 "_blank" として、新規に画面を立ち上げます。
	 * CONTENTS 等を指定すれば、コンテンツフレーム（メニューの右側全面）に、
	 * RESULT を指定すれば、リザルトフレーム（メニュー右下側）に表示します。
	 *
	 * @og.rev 3.0.0.3 (2003/02/21) ターゲット属性の新規追加
	 *
	 * @param	val TARGET 属性を指定します（初期値 "_blank"）
	 */
	public void setTarget( final String val ) {
		target = nval( getRequestParameter( val ),target );
	}

	/**
	 * 【TAG】メッセージを常時表示させるかどうか(true/false)を指定します(初期値:false)。
	 *
	 * @og.tag
	 * "true"の場合は、常時表示させます。
	 * ファイルが、存在した場合は、リンクが張られ、存在しない場合は、リンクが
	 * 張られません。
	 * "false" の場合は、ファイルが、存在した場合は、リンクが張られ、存在しない場合は、
	 * なにも表示されません。
	 * 初期値は、 "false"(メッセージを常時表示しない)です。
	 *
	 * @og.rev 3.0.0.3 (2003/02/21) メッセージ表示属性の新規追加
	 *
	 * @param	flag メッセージを常時表示させるかどうかを指定( true:常時表示/false:非表示 ）
	 */
	public void setViewMsg( final String flag ) {
		viewMsg = nval( getRequestParameter( flag ),viewMsg );
	}

	/**
	 * 【TAG】ヘルプリンクをアイコンで指定する場合のアイコンURLを指定します。(初期値:DEFAULT_HELP_ICON)。
	 *
	 * @og.tag
	 * ヘルプリンクをアイコンで指定する場合、そのアイコン画像のURLを指定します。
	 * URLは、/[CONTEXT_PATH]/jspを基準として指定します。
	 * 例) /ge/jsp/image/help.pngに存在する画像を指定する場合、iconURL=/image/help.pngを指定します。
	 * このURLが指定されない場合、ヘルプリンクは、msgLbl属性で指定されたテキストで表示されます。
	 *
	 * @og.rev 5.3.8.0 (2011/08/01) 新規追加
	 *
	 * @param url アイコンURL
	 */
	public void setIconURL( final String url ) {
		iconURL = nval( getRequestParameter( url ),iconURL );
	}
	
	/**
	 * 【TAG】FAQリンクをアイコンで指定する場合のアイコンURLを指定します。(初期値:DEFAULT_FAQ_ICON)。
	 *
	 * @og.tag
	 * FAQリンクをアイコンで指定する場合、そのアイコン画像のURLを指定します。
	 * URLは、/[CONTEXT_PATH]/jspを基準として指定します。
	 * 例) /ge/jsp/image/help.pngに存在する画像を指定する場合、iconURL=/image/help.pngを指定します。
	 *
	 * @og.rev 5.3.8.0 (2011/08/01) 新規追加
	 *
	 * @param url アイコンURL
	 */
	public void setFaqIconURL( final String url ) {
		faqIconURL = nval( getRequestParameter( url ),faqIconURL );
	}
	
	/**
	 * 【TAG】FAQ表示の機能を利用するか指定します。初期値:USE_GUI_FAQ 
	 *
	 * @og.tag
	 * trueを指定すると、FAQ画面へのリンクが表示されます。（GE80にデータが存在するかは無関係）
	 * リンク先はfaqGUIでセットした画面に対して画面IDを引数としてわたします。
	 *
	 * @og.rev 5.5.0.4 (2012/03/167) 新規追加
	 *
	 * @param	flag 
	 */
	public void setUseFaq( final String flag ) {
		useFaq = nval( getRequestParameter( flag ),useFaq );
	}
	
	
	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "guiInfoKey"	,guiInfoKey	)
				.println( "extension"	,extension	)
				.println( "baseURL"		,baseURL	)
				.println( "target"		,target		)
				.println( "viewMsg"		,viewMsg	)
				.println( "iconURL"		,iconURL	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
