/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.util;

import java.io.File;

/**
 * HybsLoaderを生成するための、設定情報を管理するためのクラスです。
 *
 * @og.rev 5.1.1.0 (2009/12/01) 新規作成
 * @og.group 業務ロジック
 * 
 * @version 5.0
 * @author Hiroki Nakamura
 * @since JDK1.6,
 */
public class HybsLoaderConfig {
	private final String srcDir;
	private final String classDir;
//	private final boolean isAutoCompile;
//	private final boolean isHotDeploy;
	private final boolean useAutoCompile;		// 5.1.8.0 (2010/07/01) メソッド名と同じフィールド名なので、変更
	private final boolean useHotDeploy;			// 5.1.8.0 (2010/07/01) メソッド名と同じフィールド名なので、変更
	private final String classPath;

	/**
	 * ソースディレクトリとクラスディレクトリのみを指定し、HybsLoaderの設定情報を構築します。
	 * この場合、AutoCompile機能、HotDeploy機能は無効になります。
	 * 
	 * @param sDir ソースディレクトリ
	 * @param cDir クラスディレクトリ
	 */
//	public HybsLoaderConfig( final String sDir, final String cDir, final boolean isDebug ) {
	public HybsLoaderConfig( final String sDir, final String cDir ) {
		this( sDir, cDir, false, false, null );
	}

	/**
	 * HybsLoaderの設定情報を構築します。
	 * 
	 * @param sDir ソースディレクトリ
	 * @param cDir クラスディレクトリ
	 * @param acom AutoCompile機能を有効にするか
	 * @param hdep HotDeploy機能を有効にするか
	 * @param clsPath コンパイル時のクラスパス
	 */
	public HybsLoaderConfig( final String sDir, final String cDir
			,final boolean acom, final boolean hdep, final String clsPath ) {
		srcDir = sDir.charAt( sDir.length() -1 ) == File.separatorChar ? sDir : sDir + File.separator;
		classDir = cDir.charAt( cDir.length() -1 ) == File.separatorChar ? cDir : cDir + File.separator;

		useAutoCompile = acom;
		useHotDeploy = hdep;
		classPath  = clsPath;
	}

	/**
	 * ソースディレクトリを取得します。
	 * 
	 * @return ソースディレクトリ
	 */
	public String getSrcDir() {
		return srcDir;
	}

	/**
	 * クラスディレクトリを取得します。
	 * 
	 * @return クラスディレクトリ
	 */
	public String getClassDir() {
		return classDir;
	}

	/**
	 * AutoCompileが有効化どうかを取得します。
	 * 
	 * @return AutoCompileが有効か
	 */
	public boolean isAutoCompile() {
		return useAutoCompile;
	}

	/**
	 * HotDeployが有効化どうかを取得します。
	 * 
	 * @return HotDeployが有効か
	 */
	public boolean isHotDeploy() {
		return useHotDeploy;
	}

	/**
	 * コンパイルのためのクラスパスを返します。
	 * コンストラクタで複数のクラスパスが指定された場合、
	 * ここで返されるクラスパスは、';'区切りのクラスパスになります。
	 * 
	 * @return クラスパス
	 */
	public String getClassPath() {
		return classPath;
	}
}
