/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.xml;

import java.io.File ;
import java.io.Writer ;
import java.io.BufferedWriter ;
import java.io.OutputStreamWriter ;
import java.io.FileOutputStream ;
import java.io.IOException ;

import javax.xml.transform.dom.DOMSource ;
import javax.xml.transform.stream.StreamResult ;
import javax.xml.transform.Transformer ;
import javax.xml.transform.TransformerFactory ;
import javax.xml.transform.TransformerException ;
import javax.xml.transform.TransformerConfigurationException ;
import javax.xml.transform.OutputKeys ;

import javax.xml.parsers.DocumentBuilder ;
import javax.xml.parsers.DocumentBuilderFactory ;
import javax.xml.parsers.ParserConfigurationException ;
import org.xml.sax.SAXException ;
import org.opengion.fukurou.util.Closer;
import org.w3c.dom.Document ;

/**
 * XMLファイルを読み取って、Document オブジェクトを取得する、ユーティリティークラスです。
 *
 * javax.xml.parsers および、org.w3c.dom の簡易処理を行います。
 * read で、Document を読み込み、write で、ファイルに書き出します。
 * なお、書き出しに関しては、UTF-8 固定で、かつ、Transformer で行いますので、
 * 属性の並び順は、保障されません。（つまり、簡易的な書き出し機能です。）
 *
 * @og.rev 5.1.7.0 (2010/06/01) 新規作成
 *
 * @version  5.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK6.0,
 */

public final class DomParser {

	/**
	 * プライベート コンストラクター
	 *
	 * オブジェクトの作成を拒否します。
	 */
	private DomParser() {}

	/**
	 * XMLファイルを読み込み、org.w3c.dom.Documentを返す
	 * @param aFile File
	 * @return Document 構築した Document( nullは読み込み失敗 )
	 */
	public static Document read( final File aFile ) {

		Document document = null;
		try {
			//----------------------------------------------------
			// step1. DocumentBuilderを構築する
			//----------------------------------------------------
			DocumentBuilderFactory bfactory = DocumentBuilderFactory.newInstance();
			DocumentBuilder builder = bfactory.newDocumentBuilder();

			//----------------------------------------------------
			// step2. XMLファイルを読み込んで、Documentを構築する
			//----------------------------------------------------
			document = builder.parse( aFile );

		} catch (ParserConfigurationException ex) {
			System.err.println( ex.getMessage() );
			ex.printStackTrace();
		} catch (SAXException ex) {
			// 文法エラーが発生した場合
			System.err.println( ex.getMessage() );
			ex.printStackTrace();
		} catch (IOException ex) {
			// ファイルが読み込めなかった場合
			System.err.println( ex.getMessage() );
			ex.printStackTrace();
		}

		// 完了
		return document;
	}

	/**
	 * Documentを指定ファイルに保存する
	 * @param aFile 保存先ファイル
	 * @param aDocument Documentインスタンス
	 * 
	 * @og.rev 5.1.9.0 (2010/08/01) Closeされないバグを修正
	 */
	public static void write( final File aFile, final Document aDocument ){
		Writer out = null;
		try {
			//---------------------------------
			// step1. Transformerの準備
			//---------------------------------
			TransformerFactory tfactory = TransformerFactory.newInstance();
			Transformer transformer = tfactory.newTransformer();

			//---------------------------------
			// step2. Transformerの動作設定
			//---------------------------------
			transformer.setOutputProperty(OutputKeys.INDENT,   "yes");
			transformer.setOutputProperty(OutputKeys.ENCODING, "UTF-8");

			//---------------------------------
			// step3. Writerの準備
			//---------------------------------
//			Writer out = new BufferedWriter( new OutputStreamWriter( new FileOutputStream(aFile),"UTF-8" )) ;
			out = new BufferedWriter( new OutputStreamWriter( new FileOutputStream(aFile),"UTF-8" )) ;

			//---------------------------------
			// step4. 書き出し
			//---------------------------------
			transformer.transform( new DOMSource( aDocument ), new StreamResult( out ) );

		} catch (TransformerConfigurationException ex) {
			System.err.println( ex.getMessage() );
			ex.printStackTrace();
		} catch (TransformerException ex) {
			// 書き出しエラー発生
			System.err.println( ex.getMessage() );
			ex.printStackTrace();
		} catch (IOException ex) {
			// ファイルが読み込めなかった場合
			System.err.println( ex.getMessage() );
			ex.printStackTrace();
		// 5.1.9.0 (2010/08/01) Closeされないバグを修正
		} finally {
			Closer.ioClose( out );
		}
	}
}
