/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;

/**
 * ユーザー単位のエディット設定情報を管理するためのクラスです。
 * 
 * 画面ID+エディット名をキーとしてエディット設定オブジェクトの
 * 追加、削除、参照を行います。
 *
 * @og.rev 5.3.6.0 (2011/06/01) 新規追加
 *
 * @version  5.0
 * @author   Hiroki Nakamura
 * @since    JDK6.0,
 */
public class DBEditConfigManager  {

	// エディット設定情報の管理オブジェクト(画面ID+エディット名単位で管理)
	private Map<String,Map<String,DBEditConfig>> editConfigMap = new HashMap<String,Map<String,DBEditConfig>>();

	/**
	 * デフォルトコンストラクターです。
	 */
	public DBEditConfigManager() {
	}

	/**
	 * エディット設定オブジェクトを追加します。
	 * 
	 * ここでの追加はあくまでメモリ上での登録になります。
	 * 登録した内容を永続的に登録する場合は、別途DBへの登録が必要になります。
	 * 
	 * @param guikey 画面ID
	 * @param editName エディット名
	 * @param config エディット設定オブジェクト
	 */
	public void addEditConfig( final String guikey, final String editName, final DBEditConfig config ) {
		if( guikey == null || guikey.length() == 0 ) { return; }
		if( editName == null || editName.length() == 0 ) { return; }
		if( config == null ) { return; }

		synchronized( editConfigMap ) {
			Map<String,DBEditConfig> map = editConfigMap.get( guikey );
			if( map == null ) {
				map = new HashMap<String, DBEditConfig>();
			}
			map.put( editName, config );

			editConfigMap.put( guikey, map );
		}
	}

	/**
	 * エディット設定オブジェクトを削除します。
	 * 
	 * ここでの追加はあくまでメモリ上での削除になります。
	 * 登録した内容を永続的に削除する場合は、別途DBへの登録が必要になります。
	 * 
	 * @param guikey 画面ID
	 * @param editName エディット名
	 * @return エディット設定オブジェクト
	 */
	public DBEditConfig deleteEditConfig( final String guikey, final String editName ) {
		if( guikey == null || guikey.length() == 0 ) { return null; }
		if( editName == null || editName.length() == 0 ) { return null; }

		DBEditConfig config = null;
		synchronized( editConfigMap ) {
			Map<String,DBEditConfig> map = editConfigMap.get( guikey );
			if( map == null ) { return null; }
			config = map.remove( editName );
			editConfigMap.put( guikey, map );
		}
		return config;
	}

	/**
	 * エディット設定オブジェクトを取得します。
	 * 
	 * @param guikey 画面ID
	 * @param editName エディット名
	 * @return エディット設定オブジェクト
	 */
	public DBEditConfig getEditConfig( final String guikey, final String editName ) {
		if( guikey == null || guikey.length() == 0 ) { return null; }
		if( editName == null || editName.length() == 0 ) { return null; }

		synchronized( editConfigMap ) {
			Map<String,DBEditConfig> map = editConfigMap.get( guikey );
			if( map == null ) { return null; }
			return map.get( editName );
		}
	}

	/**
	 * 画面IDをキーにエディット設定の一覧(配列)を返します。
	 * 返される配列は、エディット名順にソートされた状態で返されます。
	 * 
	 * @param guikey 画面ID
	 * @return エディット設定一覧(配列)
	 */
	public DBEditConfig[] getEditConfigs( final String guikey ) {
		if( guikey == null || guikey.length() == 0 ) { return null; }

		Map<String,DBEditConfig> map = editConfigMap.get( guikey );
		if( map == null ) { return null; }

		DBEditConfig[] configs = map.values().toArray( new DBEditConfig[0] );
		Arrays.sort( configs
				, new Comparator<DBEditConfig> (){
					public int compare( final DBEditConfig c1, final DBEditConfig c2 ) {
						return ( c1 == null ? -1 : c1.getEditName().compareTo( c2.getEditName() ) );
					}
				}
		);

		return configs;
	}
}
