/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.io;


import java.awt.Shape;
import java.awt.Polygon;
import java.awt.geom.Ellipse2D;
import java.awt.geom.Rectangle2D;

/**
 * <p>ShapeList は、LineRenderer で使用される Shape(line間のドット)の形状を
 * 管理しているクラスです。
 * 内部的には、●、▲、■、◆、縦●、▼、縦■、横●、右三角、横■、左三角
 * という繰返しになります。
 *
 * @og.rev 4.1.1.0 (2008/02/18) 新規作成
 *
 * @version  0.9.0	2008/02/04
 * @author	 Kazuhiko Hasegawa
 * @since	 JDK1.6,
 */
public class ShapeList {

	/** The shape sequence. */
	private final static Shape[] SHAPES = createStandardSeriesShapes();
	private final static int	 SIZE = SHAPES.length;

	/**
	 * Returns the next shape in the sequence.
	 *
	 * @return The shape.
	 */
	public int size() {
		return SIZE;
	}

	/**
	 * Returns the next shape in the sequence.
	 *
	 * @return The shape.
	 */
	public Shape getShape( final int index ) {
		return SHAPES[index % SIZE];
	}

	/**
	 * Creates an array of standard shapes to display for the items in series 
	 * on charts.
	 *
	 * @return The array of shapes.
	 */
	private static Shape[] createStandardSeriesShapes() {

		Shape[] result = new Shape[11];

		double size = 6.0;
		double delta = size / 2.0;
		int[] xpoints = null;
		int[] ypoints = null;

		// circle、●
		result[0] = new Ellipse2D.Double(-delta, -delta, size, size);

		// up-pointing triangle、▲
		xpoints = intArray(0.0, delta, -delta);
		ypoints = intArray(-delta, delta, delta);
		result[1] = new Polygon(xpoints, ypoints, 3);

		// square、■
		result[2] = new Rectangle2D.Double(-delta, -delta, size, size);

		// diamond、◆
		xpoints = intArray(0.0, delta, 0.0, -delta);
		ypoints = intArray(-delta, 0.0, delta, 0.0);
		result[3] = new Polygon(xpoints, ypoints, 4);

		// vertical ellipse、縦●
		result[4] = new Ellipse2D.Double(-delta / 2, -delta, size / 2, size);

		// down-pointing triangle、▼
		xpoints = intArray(-delta, +delta, 0.0);
		ypoints = intArray(-delta, -delta, delta);
		result[5] = new Polygon(xpoints, ypoints, 3);

		// vertical rectangle、縦■
		result[6] = new Rectangle2D.Double(-delta / 2, -delta, size / 2, size);

		// horizontal ellipse、横●
		result[7] = new Ellipse2D.Double(-delta, -delta / 2, size, size / 2);

		// right-pointing triangle、右三角
		xpoints = intArray(-delta, delta, -delta);
		ypoints = intArray(-delta, 0.0, delta);
		result[8] = new Polygon(xpoints, ypoints, 3);

		// horizontal rectangle、横■
		result[9] = new Rectangle2D.Double(-delta, -delta / 2, size, size / 2);

		// left-pointing triangle、左三角
		xpoints = intArray(-delta, delta, delta);
		ypoints = intArray(0.0, -delta, +delta);
		result[10] = new Polygon(xpoints, ypoints, 3);

		return result;
	}

	/**
	 * Helper method to avoid lots of explicit casts in getShape().  Returns
	 * an array containing the provided doubles cast to ints.
	 *
	 * @param val  double...
	 * @return int[] with converted params.
	 */
	private static int[] intArray( double ... val ) {
		int[] rtn = new int[val.length] ;
		for( int i=0;i<val.length; i++ ) {
			rtn[i] = (int)val[i];
		}

		return rtn;
	}
}
