/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.fukurou.util.XHTMLTag;

/**
 * ボタンを作成するHTML拡張タグです。
 *
 * ボタンごとに必要な画面を選び出して制御を転送することができます。
 * BODY 部分にラベルや強調文字などのタグを記述すれば、そのままボタンとして表示されます。
 * [ヒント]
 * buttonタグとinputタグが生成するボタンは、機能的には同等ですが、buttonタグの方が
 * レタリング能力があり、内容を持つことができます。
 * 例えば、画像を内容に持つbutton機能は、type属性がimageのinputタグと
 * そっくり同じですが、buttonタグは内容を持つことができます。
 *
 * @og.formSample
 * ●形式：&lt;og:button type="…" name="…" value="…" lbl="…" /&gt;
 * ●body：あり
 *
 * ●使用例
 * &lt;og:button type="submit" name="test" value="1" lbl="test" /&gt;     LabelResource プロパティのラベル使用したいとききはlbl属性に、LabelResourceを使います。
 * &lt;og:button type="submit" name="test" value="1" msg="MSG0103" /&gt;  MessageResource プロパティのラベル使用したいときはmsg属性を使います。
 *
 * @og.group 画面部品
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class ButtonTag extends HTMLTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/01/31)

	private static final String CHECK_TYPE = "|submit|reset|button|" ;

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。ボディが存在する場合のみボディが呼ばれる対応。
	 * @og.rev 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
	 *
	 * @return  int 後続処理の指示( EVAL_BODY_BUFFERED )
	 */
	@Override
	public int doStartTag() {
		// 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
		if( useTag() ) {
			set( "body",getLongLabel() );
			return( EVAL_BODY_BUFFERED );	// Body を評価する。（ extends BodyTagSupport 時）
		}
		return ( SKIP_BODY );				// Body を評価しない
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) ボディの内容を取得する処理を、CommonTagSupport で行う。
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。ボディが存在する場合のみボディが呼ばれる対応。
	 *
	 * @return  int 後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		String label = getBodyString();

		if( label != null && label.length() > 0 ) {
			set( "body",label );
		}

		return(SKIP_BODY);
	}

	/**
	 * ボタンを作成します。
	 *
	 * &lt;button type="submit" name="名前" value="値"&gt;
	 *
	 * @return  ボタンタグ文字列
	 */
	protected String makeTag() {
		return XHTMLTag.button( getAttributes() );
	}

	/**
	 * 【HTML】ボタンのタイプ(submit/reset/button)を指定します。
	 *
	 * @og.tag
	 *
	 * submit:提出ボタンを作成します。
	 * reset:リセットボタンを作成します。
	 * button:押しボタンを作成します。
	 *
	 * @param   type String
	 */
	public void setType( final String type ) {
		String tp = getRequestParameter( type );
		if( check( "|" + tp + "|" , CHECK_TYPE ) ) {
			set( "type",tp );
		}
		else {
			String errMsg = "type は、下記の範囲より指定してください。type=["
						 + tp + "]  in [" + CHECK_TYPE + "]";
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【HTML】ボタンの名称を指定します。
	 *
	 * @og.tag
	 * @param   name String ボタンの名称
	 */
	public void setName( final String name ) {
		set( "name",getRequestParameter( name ) );
	}

	/**
	 * 【HTML】ボタンの値を指定します。
	 *
	 * @og.tag
	 * @param   value ボタンの値
	 */
	public void setValue( final String value ) {
		set( "value",getRequestParameter( value ) );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;

	}
}
