/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import java.util.Calendar;
import java.util.Locale;
import java.text.DateFormat;
import java.text.SimpleDateFormat;

/**
 * 文字列の日付属性(年/月/日 時:分:秒)の半角の日付を扱う為の、カラム属性を定義します。
 *
 * YYYYMMDDHHMMSS に対応している必要があります。
 * ただし、日付の整合性チェックは行いませんが、valueAdd( String value )での
 * 日付の加算時には、正式な日付データにて加算します。
 *
 * タイプチェックとして、以下の条件を判定します。
 * ・文字列長は、直接計算で文字数との比較
 * ・日付使用文字チェック「('0' &gt; c || '9' &lt; c)以外」エラー
 * ・文字パラメータの 正規表現チェック
 *
 * @og.group データ属性
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DBType_YMDH extends DBType_YMD {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	/**
	 * String引数の文字列を＋１した文字列を返します。
	 * これは、英字の場合（A,B,C など）は、B,C,D のように,最終桁の文字コードを
	 * ＋１ します。
	 * 文字列が数字タイプの場合は, 数字に変換して、+1 します。（桁上がりもあり）
	 * 混在タイプの場合は,最後の桁だけを確認して ＋１します。
	 * 引数が null の場合と、ゼロ文字列（""）の場合は,物理的初期設定値(String getDefault())
	 * の値を返します。
	 *
	 * @og.rev 3.6.0.0 (2004/09/17) 日の加算ロジックの誤り修正。(roll ⇒ add)
	 *
	 * @param   value  String
	 * @return  String引数の文字列を＋１した文字列
	 */
	public String valueAdd( final String value ) {
		if( value == null || value.length() == 0 ) { return getDefault(); }
		Calendar cal = Calendar.getInstance();

		int year   = Integer.parseInt( value.substring( 0,4 ) );
		int month  = Integer.parseInt( value.substring( 4,6 ) ) - 1;
		int date   = Integer.parseInt( value.substring( 6,8 ) );
		int hour   = Integer.parseInt( value.substring( 8,10 ) );
		int minute = Integer.parseInt( value.substring( 10,12 ) );
		int second = Integer.parseInt( value.substring( 12,14 ) );

		cal.set( year,month,date,hour,minute,second );
		cal.add( Calendar.DATE, 1 );		// 日にちを１日加算します。

		DateFormat formatter = new SimpleDateFormat( "yyyyMMddHHmmss",Locale.JAPAN );
		return formatter.format( cal.getTime() );
	}

//	/**
//	 * エディターで編集されたデータを登録する場合に、データそのものを
//	 * 変換して、実登録データを作成します。
//	 * 例えば,大文字のみのフィールドなら、大文字化します。
//	 * 実登録データの作成は、DBType オブジェクトを利用しますので,
//	 * これと Editor とがアンマッチの場合は、うまくデータ変換
//	 * されない可能性がありますので、注意願います。
//	 *
//	 * @param   value String （一般に編集データとして登録されたデータ）
//	 * @return  修正後の文字列（一般にデータベースに登録するデータ）
//	 */
//	public String valueSet( final String value ) {
//		String rtn = ( value == null ) ? "" : value.trim();
//
//		int len = rtn.length() ;
//		if(      len == 6 ) { rtn = "20" + rtn + "000000" ; }
//		else if( len == 8 ) { rtn = rtn + "000000"; }
//		else if( rtn.startsWith( "0000" ) ) { rtn = "00000101000000"; }
//		else if( rtn.startsWith( "9999" ) ) { rtn = "99991231000000"; }
//
//		return rtn;
//	}
}
