/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;

import org.opengion.hayabusa.db.AbstractRenderer;
import org.opengion.hayabusa.db.CellRenderer;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.Selection;
import org.opengion.hayabusa.db.SelectionFactory;
import org.opengion.fukurou.util.StringFormat;

/**
 * DBMENU レンデラーは、表示パラメータで指定された SQL文を実行し、
 * プルダウンメニューで表示する場合に使用するクラスです。
 *
 *  カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.rev 3.2.3.0 (2003/06/06) 新規作成
 * @og.rev 3.4.0.1 (2003/09/03) ＤＢ検索をリアルタイムに変更。
 * @og.rev 3.5.4.2 (2003/12/15) 継承元クラスを、Editor_DBMENU から AbstractRenderer に変更。
 * @og.group データ表示
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Renderer_DBMENU extends AbstractRenderer {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private final String query ;
	private final String dbid ;
	private final String lang ;				// 4.0.0 (2006/11/15)
	private final String name  ;
	private final String useSlabel ;	// 5.5.1.0 (2012/04/03)

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 * @og.rev 3.4.0.2 (2003/09/05) クラス内部見直し。query の final 化
	 * @og.rev 3.5.4.2 (2003/12/15) name , query 変数を、初期設定しておきます。
	 *
	 */
	public Renderer_DBMENU() {
		query   = null;			// 3.5.4.2 (2003/12/15)
		dbid	= null;
		lang	= null;			// 4.0.0 (2006/11/15)
		name	= null;			// 3.5.4.2 (2003/12/15)
		useSlabel = "auto";		// 5.5.1.0 (2012/04/03)
	}

	/**
	 * デフォルトコンストラクター。
	 *
	 * @og.rev 3.3.1.1 (2003/07/03) CodeSelection の設定において、バグ修正。
	 * @og.rev 3.4.0.1 (2003/09/03) 継承の親元の変更に伴う実装の移動。
	 * @og.rev 3.5.4.2 (2003/12/15) 継承元クラスを、Editor_DBMENU から AbstractRenderer に変更。
	 * @og.rev 3.5.5.9 (2004/06/07) editorParam 属性が null の場合は、エラーとします。
	 *
	 * @param   clm DBColumn オブジェクト
	 */
	private Renderer_DBMENU( final DBColumn clm ) {
		name	= clm.getName();
		query	= clm.getRendererParam();
		dbid	= clm.getDbid();
		lang	= clm.getLang();			// 4.0.0 (2006/11/15)
		useSlabel = clm.getUseSlabel() ;	// 5.5.1.0 (2012/04/03)

		// 3.5.5.9 (2004/06/07)
		if( query == null || query.length() == 0 ) {
			String errMsg = "DBMENU Renderer では、表示パラメータは必須です。"
					+ " name=[" + name + "]" + HybsSystem.CR ;
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @param   clm DBColumn オブジェクト
	 * @return  CellRenderer オブジェクト
	 */
	public CellRenderer newInstance( final DBColumn clm ) {
		return new Renderer_DBMENU( clm );
	}

	/**
	 * データの表示用文字列を返します。
	 *
	 * ここでは、AAA:BBB:CCC:DDD という値を、$1,$2,$3,$4 に割り当てなおして、
	 * QUERYを実行します。また、$1 は、本来の値として、メニューの初期値設定等に
	 * 使用します。上記の例では、AAA が値で、それ以降は、引数になります。
	 * さらに、元の文字列"AAA:BBB:CCC:DDD"は、$0 に割り当てられます。割り当てがない
	 * 変数は、""(ゼロ文字列)として、扱われます。<br />
	 *
	 * @og.rev 3.4.0.1 (2003/09/03) ＤＢ検索をリアルタイムに変更。
	 * @og.rev 3.4.0.2 (2003/09/05) AAA:BBB:CCC:DDD という値を、$1,$2,$3,$4 に割り当てます。
	 * @og.rev 3.5.4.2 (2003/12/15) makeCodeSelection メソッドを CodeSelectionクラスに変更。
	 * @og.rev 3.5.5.7 (2004/05/10) SelectionFactory を使用して、オブジェクト作成
	 * @og.rev 4.0.0 (2006/11/15) SelectionFactory に lang 属性を追加します。
	 * @og.rev 4.3.4.0 (2008/12/01) $Cのカラム名置換えを追加
	 * @og.rev 5.5.1.0 (2012/04/03) Slabel対応
	 *
	 * @param   value String
	 * @return  データの表示用文字列
	 */
	public String getValue( final String value ) {
		final boolean uslbl = !"true".equals(useSlabel) ? false : true; // 5.5.1.0 (2012/04/03)

		// StringFormat format = new StringFormat( query,value);
		StringFormat format = new StringFormat( query, value, name ); // 4.3.4.0 (2008/12/01)
		String newQuery = format.format();
		String newValue = format.getValue();

		Selection selection = SelectionFactory.newDBSelection( newQuery,dbid,lang );
//		return selection.getValueLabel( newValue );
		return selection.getValueLabel( newValue,uslbl );
	}

	/**
	 * name属性を変えた、データ表示/編集用のHTML文字列を作成します。
	 * テーブル上の name に 行番号を付加して、名前_行番号 で登録するキーを作成し,
	 * リクエスト情報を１つ毎のフィールドで処理できます。
	 *
	 * @og.rev 4.0.0 (2005/11/30) 一覧表示では、短縮ラベルを使用します。
	 * @og.rev 4.0.0 (2006/11/15) SelectionFactory に lang 属性を追加します。
	 * @og.rev 4.3.4.0 (2008/12/01) $Cのカラム名置換えを追加
	 * @og.rev 5.5.1.0 (2012/04/03) Slabel対応
	 *
	 * @param   row   int 行番号
	 * @param   value String
	 * @return  データ表示/編集用の文字列
	 */
	public String getValue( final int row,final String value ) {
		final boolean uslbl = ("auto".equals(useSlabel) || "true".equals(useSlabel) ) ? true : false; // 5.5.1.0 (2012/04/03)

		// StringFormat format = new StringFormat( query,value);
		StringFormat format = new StringFormat( query, value, name ); // 4.3.4.0 (2008/12/01)
		String newQuery = format.format();
		String newValue = format.getValue();

		Selection selection = SelectionFactory.newDBSelection( newQuery,dbid,lang );
//		return selection.getValueLabel( newValue,true );
		return selection.getValueLabel( newValue,uslbl );
	}
}
