/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.transfer;

import java.io.File;
import java.io.PrintWriter;

import org.opengion.fukurou.db.Transaction;
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.StringUtil;

/**
 * 伝送要求に対してのデータをファイルに書込みします。
 * 但し、書き込まれるデータについては、旧伝送システムの形式と互換性を持たせるため、
 * データの前30Byteに空白で埋め、さらに全体で500Byteになるように行末にも空白埋めをします。
 *
 * 書込みするファイル名は、実行対象で指定します。ファイル名は絶対パスで指定して下さい。
 * また、書込するテキストファイルのエンコードは書込パラメーターが指定することができます。
 * 指定しない場合、システムリソースの"DB_ENCODE"で指定された値が適用されます。
 *
 * @og.group 伝送システム
 *
 * @version  5.0
 * @author   Hiroki.Nakamura
 * @since    JDK1.6
 */
public class TransferExec_SAMCB implements TransferExec {

	// 書込ファイルオブジェクト
//	private File fileWrite = null;			// 5.5.2.4 (2012/05/16) ローカル変数化
	
	// 書込ファイルのエンコード
//	private String fileEncode = null;		// 5.5.2.4 (2012/05/16) ローカル変数化

	/**
	 * ファイルに書込みします。
	 * 
	 * @param vals 伝送データ(配列)
	 * @param config 伝送設定オブジェクト
	 * @param tran トランザクションオブジェクト
	 */
	@Override
	public void execute( final String[] vals, final TransferConfig config, final Transaction tran ) {
		File fileWrite = new File( config.getExecObj() );

		String fileEncode = config.getExecPrm();
		if( fileEncode == null || fileEncode.length() == 0 ) {
			fileEncode = "UTF-8";
		}
		
		PrintWriter writer = FileUtil.getPrintWriter( fileWrite,fileEncode );
		String line = null;
		for( String s : vals ) {
			// 前30Byteを空白埋め
			String preSpace = StringUtil.stringFill( "", 30, fileEncode );
			// 全体で500Byteになるように後ろに空白埋め
			line = StringUtil.stringFill( preSpace + s, 500, fileEncode );
			writer.println( line );
		}
	}
}
