/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.fukurou.util.StringUtil;

/**
 * ＤＢカラムの属性チェックに使用されるメソッドを集約した、クラスです。
 *
 * 全変数は、public static final 宣言されており、全メソッドは、public static synchronized 宣言されています。
 *
 * @og.group データ属性
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class DBTypeCheckUtil {
	/** String を Byte[]に変換するコード
	 * 例：） "MS932" , "JISAutoDetect" ,"JIS", "EUC_JP", "MS932", "SJIS" , "Windows-31J" , "Shift_JIS"
	 */
	private static final String CODE = HybsSystem.sys( "DB_ENCODE" );

	// 5.3.9.0 (2011/09/01) 文字数チェック方式の指定
	private static final boolean USE_TEXT_LEN ;
	static {
		String useTextLen = HybsSystem.sys( "DB_USE_TEXT_LENGTH" );
		if( useTextLen != null ) {
			USE_TEXT_LEN = Boolean.valueOf( useTextLen ).booleanValue();
		}
		else {
			USE_TEXT_LEN = false;
		}
	}

	/**
	 * オブジェクトを作らせない為の、private コンストラクタ
	 */
	private DBTypeCheckUtil() {}

	/**
	 * 文字列の長さ(整数部)をチェックします。
	 *
	 * @param	value 元の文字列
	 * @param	sizeX	 整数部分の文字列の長さ
	 * @param	sizeY	 少数部分の文字列の長さ
	 * @return	エラー文字列長さ（正常時は、null）
	 */
	public static String sizeXCheck( final String value ,final int sizeX ,final int sizeY ) {
		int valuesizeX;
		int pos = value.indexOf( '.' );
		if( pos >= 0 ) {
			valuesizeX = pos;
		}
		else {
			valuesizeX = value.length();
		}
		if( value.charAt(0) == '-' ) { valuesizeX--; }

		if( valuesizeX > sizeX ) {
			// 整数部の長さが指定の長さよりも長いです。
			return String.valueOf(valuesizeX);
		} else {
			return null;
		}

	}

	/**
	 * 文字列の長さ(小数部)をチェックします。
	 *
	 * @param	value 元の文字列
	 * @param	sizeX	 整数部分の文字列の長さ
	 * @param	sizeY	 少数部分の文字列の長さ
	 * @return	エラー文字列長さ（正常時は、null）
	 */
	public static String sizeYCheck( final String value ,final int sizeX ,final int sizeY ) {
		if( sizeY == 0 ) {
			return null;
		}
		int valuesizeY;
		int pos = value.indexOf( '.' );
		if( pos >= 0 ) {
			valuesizeY = value.length() - pos - 1;
		}
		else {
			valuesizeY = 0;
		}

		if( valuesizeY > sizeY ) {
			// 少数部の長さが指定の長さよりも長いです。
			return String.valueOf(valuesizeY);
		} else {
			return null;
		}
	}

	/**
	 * 文字列の小数点の位置をチェックします。
	 * 小数点(.)が、２箇所以上存在する（存在する位置が異なる）場合エラー
	 *
	 * @param	value 元の文字列
	 * @return	エラー文字列（正常時は、null）
	 */
	public static String decimalPointCheck( final String value ) {
		String rtn = null;
		if( value.indexOf( '.' ) != value.lastIndexOf( '.' ) ) {
			rtn = changeErrorPath( value, '.' );
		}
		return rtn ;
	}

	/**
	 * 文字列の符号の位置をチェックします。
	 * マイナス(-)が、存在しないか、先頭以外の場合は、エラー
	 *
	 * @param	value 元の文字列
	 * @return	エラー文字列（正常時は、null）
	 */
	public static String decimalCodeCheck( final String value ) {
		String rtn = null;
		if( value.lastIndexOf( '-' ) > 0 ) {
			rtn = changeErrorPath( value, '-' );
		}
		return rtn ;
	}

	/**
	 * 文字列の整合性(整数)をチェックします。
	 * ０～９およびマイナス(-)を許可します。
	 *
	 * @param	value 元の文字列
	 * @return	エラー文字列（正常時は、null）
	 */
	public static String numberFormatCheck( final String value ) {
		boolean isError = false;
		int i = 0;
		char ch;
		while( i<value.length() ) {
			ch = value.charAt( i );
			if( ( '0'>ch || '9'<ch ) && ( '-'!=ch ) ) {
				isError = true;
				break;
			}
			i++;
		}
		if( isError ) {
			StringBuilder val = new StringBuilder();
			for( i=0; i<value.length(); i++ ) {
				ch = value.charAt( i );
				if( ( '0'>ch || '9'<ch ) && ( '-'!=ch ) ) {
					val.append( "<span class=\"NG\">" ).append( ch ).append( "</span>" );
				}
				else {
					val.append( ch );
				}
			}
			return val.toString();
		} else {
			return null;
		}
	}

	/**
	 * 文字列の整合性(小数)をチェックします。
	 * ０～９、マイナス(-)および小数点(.)を許可します。
	 * 
	 * og.rev 4.2.4.0 (2008/06/26) '.' or '-' のみはエラー
	 *
	 * @param	value 元の文字列
	 * @return	エラー文字列（正常時は、null）
	 */
	public static String decimalFormatCheck( final String value ) {
		boolean isError = false;
		int i = 0;
		char ch;
		while( i<value.length() ) {
			ch = value.charAt( i );
			if( ( '0'>ch || '9'<ch ) && ( '.'!=ch ) && ( '-'!=ch ) ) {
				isError = true;
				break;
			}
			i++;
		}

		// 4.2.4.0 (2008/06/26) '.' or '-' のみはエラー
		if( ( value.length() ==1 ) && ( value.charAt( 0 ) == '.' || value.charAt( 0 ) == '-' ) ) {
			isError = true;
		}

		if( isError ) {
			StringBuilder val = new StringBuilder();
			for( i=0; i<value.length(); i++ ) {
				ch = value.charAt( i );
				if( ( '0'>ch || '9'<ch ) && ( '.'!=ch ) && ( '-'!=ch ) ) {
					val.append( "<span class=\"NG\">" ).append( ch ).append( "</span>" );
				}
				else {
					val.append( ch );
				}
			}
			return val.toString();
		} else {
			return null;
		}
	}

	/**
	 * 文字列のエラー文字列を返します。
	 *
	 * @param	val String 元の文字列
	 * @param	inChar char エラー対象文字
	 * @return	String エラー文字列
	 */
	private static String changeErrorPath( final String val, final char inChar ) {
		StringBuilder buf = new StringBuilder();
		for( int i=0; i<val.length(); i++ ) {
			char ch = val.charAt( i );
			if( inChar==ch ) {
				buf.append( "<span class=\"NG\">" ).append( ch ).append( "</span>" );
			} else {
				buf.append( ch );
			}
		}
		return buf.toString();
	}

	/**
	 * 文字列の長さをチェックします。
	 * バイト数に換算して比較チェックします。
	 *
	 * @og.rev 3.0.1.3 (2003/03/11) メソッド新規追加
	 * @og.rev 3.5.5.3 (2004/04/09) StringUtil の CODE を使用したメソッドを削除する。
	 * @og.rev 5.3.9.0 (2011/09/01) DB_USE_TEXT_LENGTH を考慮した、｢文字数｣、｢バイト数｣チェック
	 *
	 * @param	value 元の文字列
	 * @param	len	 文字列の長さ
	 * @return	エラー文字列（正常時は、null）
	 */
	public static String byteLengthCheck( final String value ,final int len ) {
		String rtn = null;

		// 5.3.9.0 (2011/09/01) ｢文字数｣、｢バイト数｣チェック
		final int valLen ;
		if( USE_TEXT_LEN ) {	// true:｢文字数｣チェック方式
			valLen = value.length();
		}
		else {					// false:｢バイト数｣チェック方式
			byte[] byteValue = StringUtil.makeByte( value,CODE );	// 3.5.5.3 (2004/04/09)
			valLen = byteValue.length;
		}

//		byte[] byteValue = StringUtil.makeByte( value,CODE );	// 3.5.5.3 (2004/04/09)

//		if( byteValue.length > len ) {
//			rtn = String.valueOf( byteValue.length );
//		}

		if( valLen > len ) {
			rtn = String.valueOf( valLen );
		}

		return rtn ;
	}

	/**
	 * 文字列の整合性を、dbType パラメータを利用してチェックします。
	 * regex が、null または、長さゼロの文字列の場合は、なにもしません。
	 *
	 * @og.rev 3.6.0.0 (2004/09/22) 新規作成
	 *
	 * @param	value 元の文字列
	 * @param	regex チェックする正規表現文字列
	 * @return	エラー文字列（正常時は、null）
	 */
	public static String matcheCheck( final String value,final String regex ) {
		if( regex == null || regex.length() == 0 ) { return null; }

		if( value.matches( regex ) ) {
			return null;
		}
		else {
			return "<span class=\"NG\">" + value + "</span> regex=" + regex ;
		}
	}
}
