/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.db.DBTableModelUtil;
import org.opengion.hayabusa.db.DBColumn;

import static org.opengion.fukurou.util.StringUtil.nval ;

import java.util.prefs.Preferences;
import java.util.prefs.BackingStoreException;
import java.util.Locale ;

/**
 * 指定の Windowsレジストリにアクセスするためのタグです(特殊な環境設定が必要です)。
 *
 * 検索した結果は、DBTableModel にセットされるため、
 * JDBCQuery と同様に、viewタグで表示させることが可能です。
 * 警告
 * このタグを使用するには、java.util.prefs.WindowsPreferenceクラスを変更し、
 * 正規のクラスローダーより先に読み込ませる必要があります。それは、endorsedを利用します。
 * 但し、現時点では、全クラスが対象になってしまうため、標準的に利用することはできません。
 * （セキュリティ及び、不正な書込みを防ぐため）
 *
 * @og.formSample
 * ●形式：&lt;og:regQuery baseKey="･･･" ･･･ /&gt;
 * ●body：なし
 *
 * ●使用例
 *    &lt;og:regQuery  debug=&quot;false&quot;
 *        command     = &quot;NEW&quot;                       NEW,RENEW が使用できます。
 *        baseKey     = &quot;/microsoft/windows nt&quot;     検索時のベースとなるレジストリキー名
 *        key         = &quot;driver&quot;                    検索したいレジストリキー名（初期値は全件）
 *        value       = &quot;Ne&quot;                        検索したいレジストリ値（初期値は全件）
 *        maxRowCount = &quot;0&quot;                         最大検索件数（0で無制限）
 *        maxLevel    = &quot;0&quot;                         最大検索階層レベル（0で無制限）
 *        like        = &quot;true&quot;                      true で曖昧検索/false は一致検索
 *    /&gt;
 *
 * like は、key / value を設定したときのみ有効です。また、key / value を両方同時に設定した場合は、
 * like 属性は両方に同時に適用されます。
 *
 * @og.rev 3.1.0.0 (2003/03/20) Windowsレジストリにアクセスできる、RegistryQueryTag.java を新規に作成。
 * @og.group その他入力
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class RegistryQueryTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/01/31)

	/** command 引数に渡す事の出来る コマンド  新規 {@value} */
	public static final String CMD_NEW	 = "NEW" ;
	/** command 引数に渡す事の出来る コマンド  再検索 {@value} */
	public static final String CMD_RENEW = "RENEW" ;
	/** command 引数に渡す事の出来る コマンド リスト  */
	private static final String[] COMMAND_LIST = new String[] { CMD_NEW , CMD_RENEW };

	private static final String PREF_ROOT = "HKEY_CURRENT_USER/Software" ;

	private static final String[] COLUMN_KEY =
			new String[] { "LEBEL","KEY","VALUE","CHILD","PATH" };

	private String	tableId 		= HybsSystem.TBL_MDL_KEY;			// sessionｷｰ
	private String	command			= CMD_NEW;							// ｺﾏﾝﾄﾞ
	private String	baseKey 		= null;								// 検索ベースｷｰ
	private String	key				= null;								// 検索キー
	private String	value			= null;								// 検索バリュー
	private int 	maxRowCount		= 0;								// 最大検索数(0は無制限)
	private String	orderBy			= null;								// ソート項目
//	private String	displayMsg		= "MSG0033";						// 　件検索しました。
	private String	displayMsg		= HybsSystem.sys( "VIEW_DISPLAY_MSG" );
	private String	overflowMsg		= "MSG0007";						// 検索結果が、制限行数を超えましたので、残りはカットされました。
	private String	notfoundMsg		= "MSG0077";						// 対象データはありませんでした。
	private int		maxLevel		= 1;								// 下位層展開最大レベル(0は無制限)
	private boolean	like			= false;							// あいまい検索ﾌﾗｸﾞ

	private int		executeCount	= 0;								// 検索/実行件数
	private boolean	isMainTrans		= true;								// 5.1.6.0 (2010/05/01) DBLastSqlの処理の見直し

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.6.0.8 (2004/11/19) DBTableModel をセーブする時に、トランザクションチェックを行います。
	 * @og.rev 4.0.0.0 (2007/10/18) メッセージリソース統合( getResource().getMessage > getResource().getLabel )
	 * @og.rev 5.1.6.0 (2010/05/01) DBLastSqlの処理は、DBTableModelが新規作成された処理でのみ行う。
	 *
	 * @return  int 後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)

		if( check( command, COMMAND_LIST ) ) {
			useMainTrans( isMainTrans );			// 5.1.6.0 (2010/05/01) DBLastSqlの処理の見直し
			startQueryTransaction( tableId );		// 3.6.0.8 (2004/11/19)

			DBTableModel table = makeDBTable();
			// 3.6.0.8 (2004/11/19) トランザクションチェックを行います。
			if( ! commitTableObject( tableId, table ) ) {
				jspPrint( "RegistryQueryTag Query処理が割り込まれました。DBTableModel は登録しません。" );
				return (SKIP_PAGE);
			}

//			String label  = "";				// 4.0.0 (2005/11/30) 検索しなかった場合。
			StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_SMALL );

			// 実行件数の表示 command="NEW" のときのみ、displayMsg を表示させます。
			// 4.0.0 (2005/11/30) 出力順の変更。一番最初に出力します。
//			boolean	useStatusBar = HybsSystem.sysBool( "VIEW_USE_DISPLAY_MSG" );
			if( CMD_NEW.equals( command ) ) {
//				if( useStatusBar && executeCount > 0 && displayMsg != null && displayMsg.length() > 0 ) {
				if( executeCount > 0 && displayMsg != null && displayMsg.length() > 0 ) {
					buf.append( executeCount );
//					buf.append( getResource().getMessage( displayMsg ) );
					buf.append( getResource().getLabel( displayMsg ) );
					buf.append( HybsSystem.BR );
				}
				else if( executeCount == 0 && notfoundMsg != null && notfoundMsg.length() > 0 ) {
//					buf.append( getResource().getMessage( notfoundMsg ) );
					buf.append( getResource().getLabel( notfoundMsg ) );
					buf.append( HybsSystem.BR );
				}
			}

			if( maxRowCount > 0 && maxRowCount <= executeCount ) {
//				buf.append( getResource().getMessage( overflowMsg ) );
				buf.append( getResource().getLabel( overflowMsg ) );
				buf.append( HybsSystem.BR );
			}
//			label = buf.toString();

			jspPrint( buf.toString() );
		}

		return(EVAL_PAGE);
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 5.1.6.0 (2010/05/01) DBLastSqlの処理は、DBTableModelが新規作成された処理でのみ行う。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();

		tableId 		= HybsSystem.TBL_MDL_KEY;	// sessionｷｰ
		orderBy			= null;								// ソート項目
		command			= CMD_NEW;							// ｺﾏﾝﾄﾞ
		baseKey 		= null;								// 検索ベースｷｰ
		key				= null;								// 検索キー
		executeCount	= 0;								// 検索/実行件数
		maxRowCount		= 0;								// 最大検索数(0は無制限)
//		displayMsg 		= "MSG0033";						// 件検索しました。
		displayMsg		= HybsSystem.sys( "VIEW_DISPLAY_MSG" );
		overflowMsg		= "MSG0007";						// 検索結果が、制限行数を超えましたので、残りはカットされました。
		notfoundMsg		= "MSG0077";						// 対象データはありませんでした。
		maxLevel		= 1;								// 下位層展開最大レベル(0は無制限)
		like			= false;							// あいまい検索ﾌﾗｸﾞ
		isMainTrans		= true;								// 5.1.6.0 (2010/05/01) DBLastSqlの処理の見直し
	}

	/**
	 * レジストリ から、値を取り出し、DBTableModel を作成します。
	 *
	 * @return	DBTableModel テーブルモデル
	 */
	private DBTableModel makeDBTable() {
		DBTableModel table = DBTableModelUtil.newDBTable();
		table.init( COLUMN_KEY.length );
		for( int i=0; i<COLUMN_KEY.length; i++ ) {
			DBColumn dbColumn = getResource().makeDBColumn( COLUMN_KEY[i] );
			table.setDBColumn( i,dbColumn );
		}
		Preferences hayabusaRoot = Preferences.userRoot() ;

		try {
			if( ! hayabusaRoot.nodeExists( baseKey ) ) {
				String errMsg = "Preferences BaseKey が Windows Registry に存在しませんでした。"
								+ "[" + PREF_ROOT + baseKey + "]" ;
				throw new HybsSystemException( errMsg );
			}

			showChild( table,hayabusaRoot.node(baseKey),1 );
		}
		catch(BackingStoreException e) {
			String errMsg = "レジストリ から、値を取り出す事が出来ませんでした。" 
						+ "[" + PREF_ROOT + baseKey + "]" ;		// 5.1.8.0 (2010/07/01) errMsg 修正
			throw new HybsSystemException( errMsg,e );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}

		return table ;
	}

	/**
	 * Preferences の子要素を検索します。
	 *
	 * この処理は、再帰定義により、階層を順次下がっていきます。
	 * レベル制限、検索最大数制限のチェックを行っています。
	 * addTableValue メソッドを呼び出して、検索結果を、DBTableModel に順次追加していきます。
	 *
	 * @param	table DBTableModel
	 * @param	prefs Preferences
	 * @param	lvl int
	 */
	private void showChild( final DBTableModel table,
							final Preferences prefs,
							final int lvl ) throws BackingStoreException {

		if( maxRowCount > 0 && maxRowCount <= executeCount ) { return ; }
		if( maxLevel > 0 && lvl > maxLevel ) { return; }

		String [] child = prefs.childrenNames();

		if( child != null && child.length > 0 ) {
	 		for( int i = 0; i < child.length; i++) {
				String name = child[i];
				Preferences childPrefs = prefs.node( name.toLowerCase(Locale.JAPAN) );

				addTableValue( name,table,childPrefs,lvl,true );
				showChild( table,childPrefs,lvl+1 );
			}
		}
		else {
			showPrefs( table,prefs,lvl );
		}
 	}

	/**
	 * Preferences の属性を検索します。
	 *
	 * addTableValue メソッドを呼び出して、検索結果を、DBTableModel に順次追加していきます。
	 *
	 * @param	table DBTableModel
	 * @param	prefs Preferences
	 * @param	lvl int
	 */
	private void showPrefs( final DBTableModel table,
							final Preferences prefs,
							final int lvl ) throws BackingStoreException {

		String [] keys = prefs.keys();

		if( keys != null ) {
	 		for( int i = 0; i < keys.length; i++) {
				String name = keys[i];
				addTableValue( name,table,prefs,lvl,false );
			}
		}
 	}

	/**
	 * 検索された Preferencesを、DBTableModel に順次追加していきます。
	 *
	 * @param	name String
	 * @param	table DBTableModel
	 * @param	prefs Preferences
	 * @param	lvl int
	 * @param	flag boolean
	 */
	private void addTableValue( final String name,
								final DBTableModel table,
								final Preferences prefs,
								final int lvl,
								final boolean flag ) {

		if( maxRowCount > 0 && maxRowCount <= executeCount ) { return ; }

		String lowerName = name.toLowerCase(Locale.JAPAN);
		String val       = ( flag ) ? "" : prefs.get(lowerName, "");

		if( key != null ) {
			if( like ) {
				if( lowerName.indexOf( key ) < 0 ) { return; }
			}
			else {
				if( ! lowerName.equalsIgnoreCase( key ) ) { return; }
			}
		}
		if( value != null ) {
			if( like ) {
				if( (val.toLowerCase(Locale.JAPAN)).indexOf( value ) < 0 ) { return; }
			}
			else {
				if( ! val.equalsIgnoreCase( value ) ) { return; }
			}
		}

		String[] value = new String[COLUMN_KEY.length];
		value[0] = String.valueOf( lvl );				// LEVEL
		value[1] = name;								// KEY
		value[2] = val;									// VALUE
		value[3] = String.valueOf( flag );				// CHILD
		value[4] = prefs.absolutePath() ;				// PATH

		table.addColumnValues( value );
		executeCount++ ;
	}

	/**
	 * 【TAG】(通常は使いません)結果をDBTableModelに書き込んで、sessionに登録するときのキーを指定します。
	 *
	 * @og.tag
	 * 初期値は、HybsSystem.TBL_MDL_KEY です。
	 *
	 * @param	id sessionに登録する時の ID
	 */
	public void setTableId( final String id ) {
		tableId = nval( getRequestParameter( id ), tableId );
	}

	/**
	 * 【TAG】検索した結果を表示する表示順をファイル属性名で指定します。
	 *
	 * @og.tag
	 * 現仕様では、複数のキーを指定することは出来ません。
	 *
	 * @param	ordr	ソートキーを指定。
	 */
	public void setOrderBy( final String ordr ) {
		orderBy = nval( getRequestParameter( ordr ),orderBy );
	}

	/**
	 * 【TAG】コマンド(NEW,RENEW)をセットします(初期値:NEW)。
	 *
	 * @og.tag
	 * コマンドは,HTMLから（get/post)指定されますので,CMD_xxx で設定される
	 * フィールド定数値のいづれかを、指定できます。
	 * 何も設定されない、または、null の場合は、"NEW" が初期値にセットされます。
	 *
	 * @param	cmd コマンド（public static final 宣言されている文字列)
	 * @see		<a href="{@docRoot}/constant-values.html#org.opengion.hayabusa.taglib.RegistryQueryTag.CMD_NEW">コマンド定数</a>
	 */
	public void setCommand( final String cmd ) {
		String cmd2 = getRequestParameter( cmd );
		if( cmd2 != null && cmd2.length() > 0 ) { command = cmd2.toUpperCase(Locale.JAPAN); }
	}

	/**
	 * 【TAG】検索ベースキーを設定します(HKEY_CURRENT_USER\Software 以下をベースとします)。
	 *
	 * @og.tag
	 * キーは,HKEY_CURRENT_USER\Software 以下をベースとします。
	 *
	 * @param	bkey 検索ベースキー
	 */
	public void setBaseKey( final String bkey ) {
		baseKey = nval( getRequestParameter( bkey ),baseKey );
		if( baseKey != null ) { baseKey = baseKey.toLowerCase(Locale.JAPAN); }
	}

	/**
	 * 【TAG】検索キーを設定します。
	 *
	 * @og.tag 検索キーを設定します。
	 *
	 * @param	ky 検索キー
	 */
	public void setKey( final String ky ) {
		key = nval( getRequestParameter( ky ),key );
		if( key != null ) { key = key.toLowerCase(Locale.JAPAN); }
	}

	/**
	 * 【TAG】検索バリューを設定します。
	 *
	 * @og.tag 検索バリューを設定します。
	 *
	 * @param	val 検索バリュー
	 */
	public void setValue( final String val ) {
		value = nval( getRequestParameter( val ),value );
		if( value != null ) { value = value.toLowerCase(Locale.JAPAN); }
	}

	/**
	 * 【TAG】レジストリの最大検索件数をセットします(初期値:0[無制限])。
	 *
	 * @og.tag
	 * DBTableModelのデータとして登録する最大件数をこの値に設定します。
	 * サーバーのメモリ資源と応答時間の確保の為です。
	 * 0 は、無制限です。（初期値は、無制限です。）
	 *
	 * @param	count レジストリの最大検索件数
	 */
	public void setMaxRowCount( final String count ) {
		maxRowCount = nval( getRequestParameter( count ),maxRowCount );
	}

	/**
	 * 【TAG】検索結果を画面上に表示するメッセージIDを指定します(初期値:MSG0033[ 件検索しました])。
	 *
	 * @og.tag
	 * ここでは、検索結果の件数や登録された件数をまず出力し、
	 * その次に、ここで指定したメッセージをリソースから取得して
	 * 表示します。
	 * 表示させたくない場合は, displayMsg = "" をセットしてください。
	 * 初期値は、検索件数を表示します。
	 *
	 * @param	id ディスプレイに表示させるメッセージ ID
	 */
	public void setDisplayMsg( final String id ) {
		displayMsg = getRequestParameter( id );
	}

	/**
	 * 【TAG】検索結果がゼロ件の場合に表示するメッセージリソースIDを指定します(初期値:MSG0077[対象データはありませんでした])。
	 *
	 * @og.tag
	 * ここでは、検索結果がゼロ件の場合のみ、特別なメッセージを表示させます。
	 * 従来は、displayMsg と兼用で、『0　件検索しました』という表示でしたが、
	 * displayMsg の初期表示は、OFF になりましたので、ゼロ件の場合のみ別に表示させます。
	 * 表示させたくない場合は, notfoundMsg = "" をセットしてください。
	 * 初期値は、MSG0077[対象データはありませんでした]です。
	 *
	 * @param	id ディスプレイに表示させるメッセージ ID
	 */
	public void setNotfoundMsg( final String id ) {
		String ids = getRequestParameter( id );
		if( ids != null ) { notfoundMsg = ids; }
	}

	/**
	 * 【TAG】オーバーフロー時に画面上に表示するメッセージIDを指定します(初期値:MSG0007)。
	 *
	 * @og.tag
	 * 表示させたくない場合は, overflowMsg = "" をセットしてください。
	 *
	 * @param	id ディスプレイに表示させるメッセージ ID
	 */
	public void setOverflowMsg( final String id ) {
		overflowMsg = getRequestParameter( id );
	}

	/**
	 * 【TAG】検索時の最大展開レベル(0は無制限)を指定します(初期値:1)。
	 *
	 * @og.tag
	 * ０を指定すると、無制限に階層を展開します。
	 * 初期値は、１レベルです。
	 *
	 * @param	lvl 検索時の最大展開レベル
	 */
	public void setMaxLevel( final String lvl ) {
		maxLevel = nval( getRequestParameter( lvl ),maxLevel );
	}

	/**
	 * 【TAG】キーおよびバリューについて,like 検索を行うかどうか(true/false)を指定します(初期値:false)。
	 *
	 * @og.tag
	 * like検索とは、キーの一部の文字を含む場合にマッチしたとして、値を取り出します。
	 * ここでの設定は、キーもバリューも同時に適用されます。また、大文字小文字の区別も行いません。
	 *
	 * @param	lik like 検索を行うかどうか
	 */
	public void setLike( final String lik ) {
		like = nval( getRequestParameter( lik ),like );
	}

	/**
	 * タグの名称を、返します。
	 * 自分自身のクラス名より、自動的に取り出せないため、このメソッドをオーバーライドします。
	 *
	 * @og.rev 4.0.0 (2005/01/31) 新規追加
	 *
	 * @return  タグの名称
	 */
	protected String getTagName() {
		return "regQuery" ;
	}

	/**
	 * 【TAG】(通常使いません)タグで処理される処理がメインとなるトランザクション処理かどうかを指定します。(初期値:false)
	 *
	 * @og.tag
	 * (通常使いません)タグで処理される処理が、メインとなるトランザクション処理かどうかを指定します。(初期値:false)
	 * この値は、ファイルダウンロード処理に影響します。この値がtrueに指定された時にcommitされたDBTableModelが
	 * ファイルダウンロードの対象の表になります。
	 * 
	 * このパラメーターは、通常、各タグにより実装され、ユーザーが指定する必要はありません。
	 * 但し、1つのJSP内でDBTableModelが複数生成される場合に、前に処理したDBTableModelについてファイルダウンロードをさせたい
	 * 場合は、後ろでDBTableModelを生成するタグで、明示的にこの値をfalseに指定することで、ファイルダウンロード処理の対象から
	 * 除外することができます。
	 * 
	 * @og.rev 5.1.6.0 (2010/05/01) 新規作成
	 *
	 * @param  flag メイントランザクションかどうか
	 */
	public void setMainTrans( final String flag ) {
		isMainTrans = nval( getRequestParameter( flag ),isMainTrans );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION		)
				.println( "tableId"		,tableId		)
				.println( "command"		,command		)
				.println( "baseKey"		,baseKey		)
				.println( "key"			,key			)
				.println( "value"		,value			)
				.println( "maxRowCount"	,maxRowCount	)
				.println( "orderBy"		,orderBy		)
				.println( "displayMsg"  ,displayMsg		)
				.println( "overflowMsg" ,overflowMsg	)
				.println( "maxLevel"	,maxLevel		)
				.println( "like"		,like			)
				.println( "executeCount",executeCount	)
				.println( "PREF_ROOT"	,PREF_ROOT		)
				.println( "COLUMN_KEY"	,COLUMN_KEY		)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
