/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import static org.opengion.fukurou.util.StringUtil.nval;

// import javax.servlet.jsp.tagext.BodyContent;

/**
 * 左右分割スクロール処理を行う、View を設定します。
 *
 * BODY部に記述された Viewタグに対して、左右分割情報を追記設定します。
 * Viewタグの左側には、このタグで指定された fixDisplay のカラムを columnDisplay に
 * 設定するとともに、このタグで指定された numberType をセットします。
 * (初期値は、numberType="sequence" です。）
 * Viewタグの右側には、fixDisplay のカラムを noDisplay のカラムに設定するとともに、
 * command="VIEW" , noMessage="true" , useSelectedRow="false" , numberType="delete"
 * をセットします。(既存の設定値があれば、それに追記されます。)
 *
 * @og.formSample
 * ●使用例
 *    <og:splitView fixDisplay="CLM,NAME_JA">
 *      <og:view
 *          viewFormType = "HTMLTable"
 *          command      = "{@command}"
 *          checked      = "{@checked}"
 *          startNo      = "{@startNo}"
 *          pageSize     = "{@pageSize}"
 *          noWritable      = "{@noWritable}"
 *          columnWritable  = "{@columnWritable}"
 *      />
 *    </og:splitView>
 *
 * @og.rev 5.3.0.0 (2010/12/01) 新規作成
 * @og.group 画面部品
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class SplitViewTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.3.0.0 (2010/12/01)" ;

	private static final long serialVersionUID = 5300 ;	// 5.3.0.0 (2010/12/01)

	private static final String SPLIT_A = 
				  "<style type=\"text/css\">#GantBody div tr { height:22px; }</style>"				+ HybsSystem.CR
				+ "<table id=\"GantBody\" border=\"0px\" cellpadding=\"0px\" cellspacing=\"0px\"" 	+ HybsSystem.CR
				+ "       frame=\"box\" rules=\"all\" style=\"margin:0px;padding:0px;\">"			+ HybsSystem.CR
				+ "  <tr style=\"margin:0px;padding:0px;\">"										+ HybsSystem.CR
				+ "    <td valign=\"top\" style=\"margin:0px; padding:0px;\" >"						+ HybsSystem.CR
				+ "      <div id=\"X1\" style=\"overflow-x:hidden; overflow-y:hidden;\" >"			+ HybsSystem.CR ;

	private static final String SPLIT_B = 
				  "      </div>"													+ HybsSystem.CR
				+ "    </td>"														+ HybsSystem.CR
				+ "    <td  valign=\"top\" style=\"margin:0px; padding:0px;\">"		+ HybsSystem.CR
				+ "      <div id=\"X2\" style=\"position:absolute; overflow-x:hidden; overflow-y:hidden;\" >" + HybsSystem.CR ;

	private static final String SPLIT_C = 
				  "       </div>"		+ HybsSystem.CR
				+ "    </td>"			+ HybsSystem.CR
				+ "  </tr>"				+ HybsSystem.CR
				+ "</table>"			+ HybsSystem.CR ;

	private String  fixDisplay    = null;
	private boolean firstStepFlag = true;		// BODY部の view 処理の制御

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @return  int 後続処理の指示( EVAL_BODY_BUFFERED )
	 */
	@Override
	public int doStartTag() {
		firstStepFlag = true;
		// EVAL_BODY_BUFFERED ではなく、Viewなので、INCLUDE 処理します。
		jspPrint( SPLIT_A );
		return( EVAL_BODY_INCLUDE );	// Body インクルード（ extends TagSupport 時）
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @return  int 後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		// EVAL_BODY_INCLUDE なので、コンテンツの取得ではなく、処理のみ実行されます。

		if( firstStepFlag ) {
			firstStepFlag = false;

			jspPrint( SPLIT_B );
			return( EVAL_BODY_BUFFERED );	// ボディーを再評価（ extends BodyTagSupport 時）
		}
		else {
			jspPrint( SPLIT_C );
			return(SKIP_BODY);				// Body を評価しない
		}
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		fixDisplay = null;
	}

	/**
	 * 【TAG】固定するカラム名を、CSV形式(カンマ区切り文字列)で設定します。
	 *
	 * @og.tag
	 * Viewタグの左側(固定部)には、このタグで指定された fixDisplay のカラムを 
	 * columnDisplay に設定します。
	 * Viewタグの右側には、fixDisplay のカラムを noDisplay のカラムに設定します。
	 * 既存の設定値(noDisplay)があれば、それに追記されます。
	 *
	 * @param   clms 固定するカラム名(カンマ区切り文字列)
	 */
	public void setFixDisplay( final String clms ) {
		fixDisplay = nval( getRequestParameter( clms ),fixDisplay );
	}

	/**
	 * 固定するカラム名を、CSV形式(カンマ区切り文字列)で取得します。
	 *
	 * これは、BODY部に記述された、viewタグからアクセスされるメソッドです。
	 * 設定されていない場合は、null です。
	 *
	 * @return   fixDisplay 固定するカラム名(カンマ区切り文字列)
	 */
	protected String getFixDisplay() {
		return fixDisplay ;
	}

	/**
	 * BODY部の view 処理の制御を行うためのフラグを返します。
	 *
	 * 左右分割を行うには、Viewタグを２回出力する必要があります。
	 * ここでは isFirstStep="true" が１回目(左側：固定部)で、false が
	 * 右側(可変部)になるように、View側で制御します。
	 *
	 * @return boolean BODY部の view 処理の制御(true:１回目 / false:２回目)
	 */
	protected boolean isFirstStep() {
		return firstStepFlag ;
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION	)
				.println( "fixDisplay"		,fixDisplay	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
