/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.db.AbstractRenderer;
import org.opengion.hayabusa.db.CellRenderer;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.fukurou.util.StringFormat;

/**
 * FORM レンデラーは、表示パラメータで指定された FORM を表示するクラスで、
 * 元のValue を、$1 として、使用可能です。
 * (コロンで区切られた値の表示のコントロールを行うレンデラーです）
 *
 * カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.rev 2.1.1.1 (2002/11/15) FORM レンデラーを新規追加しました。
 * @og.rev 5.4.2.6 (2012/01/19) コメント修正
 * @og.group データ表示
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Renderer_FORM extends AbstractRenderer {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private final String form ;
	private final String name  ;

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 *
	 */
	public Renderer_FORM() {
		form   = null;
		name   = null; // 4.3.4.0 (2008/12/01)
	}

	/**
	 * デフォルトコンストラクター。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 3.4.0.0 (2003/09/01) 表示パラメータ、編集パラメータ、文字パラメータの追加。
	 * @og.rev 3.8.8.2 (2007/01/26) StringFormat の $0 対応。
	 *
	 * @param   clm DBColumn オブジェクト
	 */
	private Renderer_FORM( final DBColumn clm ) {
		String tmp = clm.getRendererParam();
		if( tmp == null || tmp.length() == 0 ) { tmp = "$0"; }	// 3.8.8.2 (2007/01/26)
		form = tmp ;
		name = clm.getName(); // 4.3.4.0 (2008/12/01)
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 3.1.2.1 (2003/04/10) synchronized を、削除します。
	 *
	 * @param   clm DBColumn オブジェクト
	 * @return  CellRenderer オブジェクト
	 */
	public CellRenderer newInstance( final DBColumn clm ) {
		return new Renderer_FORM( clm );
	}

	/**
	 * データの表示用文字列を返します。
	 *
	 * ここでは、AAA:BBB:CCC:DDD という値を、$1,$2,$3,$4 に割り当てなおして、
	 * QUERYを実行します。また、$1 は、本来の値として、メニューの初期値設定等に
	 * 使用します。上記の例では、AAA が値で、それ以降は、引数になります。
	 * さらに、元の文字列"AAA:BBB:CCC:DDD"は、$0 に割り当てられます。割り当てがない
	 * 変数は、""(ゼロ文字列)として、扱われます。<br />
	 *
	 * @og.rev 3.4.0.2 (2003/09/05) AAA:BBB:CCC:DDD という値を、$1,$2,$3,$4 に割り当てます。
	 * @og.rev 4.3.4.0 (2008/12/01) $Cのカラム名置換えを追加
	 *
	 * @param   value String
	 * @return  データの表示用文字列
	 */
	public String getValue( final String value ) {
		// StringFormat format = new StringFormat( form,value );
		StringFormat format = new StringFormat( form, value, name ); // 4.3.4.0 (2008/12/01)
		return format.format();
	}
}
