/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.db.TableFilter;
import org.opengion.fukurou.db.Transaction;
import org.opengion.fukurou.db.TransactionReal;
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.StringUtil;
import static org.opengion.fukurou.util.StringUtil.nval ;

import java.io.ObjectOutputStream;
import java.io.ObjectInputStream;
import java.io.IOException;

/**
 * TableFilter のサブクラスをCALLしてDBTableModelにアクセスするタグです。
 *
 * DBTableModel を TableFilter のサブクラス(classIdで指定)に渡して処理を実行します。
 * クラスを作成する場合は、org.opengion.hayabusa.db.TableFilter インターフェースを継承した
 * クラスにする必要があります。また、classId 属性には、システムリソース で
 * 設定した TableFilter.XXXX の XXXX を指定します。
 *
 * ※ このタグは、Transaction タグの対象です。
 *
 * @og.formSample
 * ●形式：&lt;og:tableFilter classId="…" /&gt;
 * ●body：あり
 *
 * ●Tag定義：
 *   &lt;og:tableFilter
 *       classId          ○【TAG】データベース処理を実行するクラスパスを指定します。(必須)
 *       tableId            【TAG】(通常は使いません)DBTableModel sessionに登録されているキーを指定します
 *       modifyType         【TAG】データ処理の方法(A:追加 C:更新 D:削除)を指定します
 *       keys               【TAG】リンク先に渡すキーを指定します
 *       vals               【TAG】names属性に対応する値をCSV形式で複数指定します
 *       selectedAll        【TAG】データを全件選択済みとして処理するかどうか[true/false]を指定します(初期値:false)
 *       scope              【TAG】キャッシュする場合のスコープ[request/page/session/applicaton]を指定します(初期値:session)
 *       dbid               【TAG】(通常は使いません)Queryオブジェクトを作成する時のDB接続IDを指定します
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:tableFilter&gt;
 *
 * ●使用例
 *    ・引数/プロシジャーを直接書く場合
 *    【entry.jsp】
 *        &lt;og:tableFilter
 *            classId     = "WL_LOGICSET"         :TableFilter のサブクラス(実行クラス)
 *            tableId     = "WL0000"              :登録元のDBTableModelのsession/request変数内の取得キー
 *            keys        = "AA,BB,CC"            :実行クラスへの引数のキー
 *            vals        = "{&64;AA},{&64;BB},{&64;CC}"   :実行クラスへの引数の値
 *            selectedAll = "false/true"          :処理対象の行を全行選択するかどうか(初期値:false)
 *            modifyType  = "A/C/D"               :処理の方法(A:追加 C:更新 D:削除)を指定します。初期値は自動です。
 *        /&gt;
 *
 * @og.group その他
 * @og.rev 3.8.5.0 (2006/03/20) 新規作成
 *
 * @version  0.9.0  2000/10/17
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.1,
 */
public class TableFilterTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/01/31)

	private static final String errMsgId	= HybsSystem.ERR_MSG_KEY;
	private transient DBTableModel	table	= null;

	private String		tableId		= HybsSystem.TBL_MDL_KEY;
	private String		classId		= null;
	private String		modifyType	= null;
	private String[]	keys		= null;
	private String[]	vals		= null;

	private   String	dbid		= null ; // 4.2.4.0 (2008/06/23)
	private   String	body		= null ; // 4.2.4.0 (2008/06/23)

	private boolean		selectedAll	= false;

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @return	後続処理の指示( EVAL_BODY_BUFFERED )
	 */
	@Override
	public int doStartTag() {
		table = (DBTableModel)getObject( tableId );

		if( keys != null && vals != null && ( keys.length != vals.length ) ) {
			String errMsg = "keys と vals の設定値の数が異なります。: " + HybsSystem.CR
						+ "keys.length=[" + keys.length + "] , "
						+ "keys.length=[" + StringUtil.array2line( keys,"," ) + "]"
						+ HybsSystem.CR
						+ "vals.length=[" + vals.length + "] , "
						+ "vals.length=[" + StringUtil.array2line( vals,"," ) + "]";
			throw new HybsSystemException( errMsg );
		}

		startQueryTransaction( tableId );
//		return(SKIP_BODY);				// Body を評価しない
		return ( EVAL_BODY_BUFFERED );	// Body を評価する
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		body = nval( getBodyString(),body );

		return(SKIP_BODY);
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 * @og.rev 4.2.3.0 (2008/06/23) DBIDとボディー部分の記述を下位クラスに渡す用に修正
	 * @og.rev 4.3.7.4 (2009/07/01) Resouceオブジェクトを下位クラスに渡す用に修正
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応
	 * @og.rev 5.2.1.0 (2010/10/01) debugPrint() メソッドの処理条件見直し
	 * @og.rev 5.3.7.0 (2011/07/01) TransactionReal の引数変更 、Transaction対応で、close処理を入れる。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		// デバッグ時には、オブジェクト内部情報を表示する。
//		if( isDebug() ) {
//			debugPrint();	// 5.2.1.0 (2010/10/01) debugPrint() メソッド自体に、isDebug() が組み込まれている。
//		}
		debugPrint();	// 5.2.1.0 (2010/10/01) debugPrint() メソッド自体に、isDebug() が組み込まれている。
		int rtnCode = EVAL_PAGE;	// try ～ finally の関係で、変数化しておく

		int[] rowNo = getParameterRows();

		// 5.1.9.0 (2010/08/01) Transaction 対応
//		final Transaction tran ;
		Transaction tran = null;
		final TableFilter filter ;
		// 5.3.7.0 (2011/07/01) Transaction対応で、close処理を入れる。
		try {
			TransactionTag tranTag = (TransactionTag)findAncestorWithClass( this,TransactionTag.class );
			if( tranTag == null ) {
	//			tran = new TransactionReal( dbid,getApplicationInfo() );
				tran = new TransactionReal( getApplicationInfo() );		// 5.3.7.0 (2011/07/01) 引数変更
			}
			else {
				tran = tranTag.getTransaction();
			}

			String cls = HybsSystem.sys( "TableFilter_" + classId );
//			TableFilter filter = (TableFilter)HybsSystem.newInstance( cls );
			filter = (TableFilter)HybsSystem.newInstance( cls );

			filter.setDBTableModel( table );
			filter.setParameterRows( rowNo );
			filter.setModifyType( modifyType );
			filter.setKeysVals( keys,vals );
	//		filter.setApplicationInfo( getApplicationInfo() );	// 3.8.7.0 (2006/12/15)
			filter.setTransaction( tran );						// 5.1.9.0 (2010/08/01) Transaction 対応
			filter.setDebug( isDebug() );
			filter.setDbid( dbid );					// 4.2.4.0 (2008/06/23)
			filter.setSql( body );					// 4.2.4.0 (2008/06/23)
			filter.setResource( getResource() );	// 4.3.7.4 (2009/07/01)

			table = filter.execute();

			int errCode = filter.getErrorCode();
			ErrorMessage errMessage = filter.getErrorMessage();

			if( errCode >= ErrorMessage.NG )  { 	// 異常
				rtnCode = SKIP_PAGE;
			}

			String err = TaglibUtil.makeHTMLErrorTable( errMessage,getResource() );
			if( err != null && err.length() > 0 ) {
				jspPrint( err );
				setSessionAttribute( errMsgId,errMessage );
			}
			else {
				removeSessionAttribute( errMsgId );
			}
		}
		finally {
			if( tran != null ) { tran.close(); }
		}

		if( table != null && ! commitTableObject( tableId, table ) ) {
			return (SKIP_PAGE);
		}

//		if( errCode >= ErrorMessage.NG )  { 	// 異常
//			return(SKIP_PAGE);
//		}
//		else {
//			return(EVAL_PAGE);
//		}

		return( rtnCode );
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		table		= null;
		tableId		= HybsSystem.TBL_MDL_KEY;
		classId		= null;
		modifyType	= null;
		keys		= null;
		vals		= null;
		selectedAll	= false;
		dbid		= null; // 4.2.4.0 (2008/06/23)
		body		= null; // 4.2.4.0 (2008/06/23)
	}

	/**
	 * 表示データの HybsSystem.ROW_SEL_KEY を元に、選ばれた 行を処理の対象とします。
	 *
	 * @return	選択行の配列
	 */
	@Override
	protected int[] getParameterRows() {
		final int[] rowNo ;
		if( selectedAll ) {
			int rowCnt = table.getRowCount();
			rowNo = new int[ rowCnt ];
			for( int i=0; i<rowCnt; i++ ) {
				rowNo[i] = i;
			}
		} else {
			rowNo = super.getParameterRows();		// 4.0.0 (2005/01/31)
		}
		return rowNo;
	}

	/**
	 * 【TAG】データベース処理を実行するクラスパスを指定します。
	 *
	 * @og.tag
	 * ここで指定するクラスIDは、システムリソース にて TableFilter の
	 * サブクラス(インターフェース継承)として指定する必要があります。
	 *
	 * クラス自身は、org.opengion.hayabusa.db.TableFilter インターフェースを継承している必要があります。
	 *
	 * @param	id TableFilter インターフェースを継承している実クラスの ID
	 * @see		org.opengion.hayabusa.db.TableFilter  TableFilter インターフェース
	 */
	public void setClassId( final String id ) {
		classId = nval( getRequestParameter( id ),classId );
	}

	/**
	 * 【TAG】(通常は使いません)DBTableModel sessionに登録されているキーを指定します。
	 *
	 * @og.tag
	 * 初期値は、HybsSystem.TBL_MDL_KEY です。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規追加
	 *
	 * @param	id sessionに登録する時の ID
	 */
	public void setTableId( final String id ) {
		tableId = nval( getRequestParameter( id ),tableId );
	}

	/**
	 * 【TAG】データを全件選択済みとして処理するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * 全てのデータを選択済みデータとして扱って処理します。
	 * 全件処理する場合に、指定します。(true/false)
	 * デフォルト false です。
	 *
	 * @param  all データを全件選択済み [true:全件選択済み/false:通常]
	 */
	public void setSelectedAll( final String all ) {
		selectedAll = nval( getRequestParameter( all ),selectedAll );
	}

	/**
	 * 【TAG】データ処理の方法(A:追加 C:更新 D:削除)を指定します。
	 *
	 * @og.tag
	 * 通常は、DBTableModel に自動設定されている modifyType を元に、データ処理方法を
	 * 選別します。(A:追加 C:更新 D:削除)
	 * この場合、行単位で modifyType の値を取得して判別する必要がありますが、一般には
	 * 処理対象は、全件おなじ modifyType である可能性が高いです。
	 * また、selectedAll などで強制的に全件処理対象とする場合は、modifyType に値が
	 * 設定さていません。その様な場合に外部より modifyType を指定します。
	 * 初期値は、自動判定 です。
	 *
	 * @param  type データ処理の方法(A:追加 C:更新 D:削除)
	 */
	public void setModifyType( final String type ) {
		modifyType = nval( getRequestParameter( type ),modifyType );

		if( modifyType != null && !"A".equals( modifyType ) && !"C".equals( modifyType ) && !"D".equals( modifyType ) ) {
			String errMsg = "modifyType は A:追加 C:更新 D:削除 のどれかを指定してください。: " + HybsSystem.CR
						+ "modifyType=[" + modifyType + "]";
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【TAG】リンク先に渡すキーを指定します。
	 *
	 * @og.tag
	 * 戻る時に、検索時のキャッシュに指定した引数以外に指定したり、別の値に置き換えたり
	 * する場合のキーを設定できます。カンマ区切りで複数指定できます。
	 * vals 属性には、キーに対応する値を、設定してください。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @param	key リンク先に渡すキー
	 */
	public void setKeys( final String key ) {
		keys = getCSVParameter( key );
	}

	/**
	 * 【TAG】names属性に対応する値をCSV形式で複数指定します。
	 *
	 * @og.tag
	 * キーに設定した値を、カンマ区切り文字で複数して出来ます。
	 * 指定順序は、キーと同じにしておいて下さい。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @param	val names属性に対応する値
	 */
	public void setVals( final String val ) {
		vals = getCSVParameter( val );
	}

	/**
	 * 【TAG】(通常は使いません)Queryオブジェクトを作成する時のDB接続IDを指定します。
	 *
	 * @og.tag
	 * Queryオブジェクトを作成する時のDB接続IDを指定します。
	 * これは、システムリソースで、DEFAULT_DB_URL 等で指定している データベース接続先
	 * 情報に、XX_DB_URL を定義することで、 dbid="XX" とすると、この 接続先を使用して
	 * データベースにアクセスできます。
	 *
	 * @param	id データベース接続ID
	 */
	public void setDbid( final String id ) {
		dbid = nval( getRequestParameter( id ),dbid );
	}

	/**
	 * シリアライズ用のカスタムシリアライズ書き込みメソッド
	 *
	 * @og.rev 4.0.0 (2006/09/31) 新規追加
	 * @serialData
	 *
	 * @param	strm	ObjectOutputStreamオブジェクト
	 */
	private void writeObject( final ObjectOutputStream strm ) throws IOException {
		strm.defaultWriteObject();
	}

	/**
	 * シリアライズ用のカスタムシリアライズ読み込みメソッド
	 *
	 * ここでは、transient 宣言された内部変数の内、初期化が必要なフィールドのみ設定します。
	 *
	 * @og.rev 4.0.0 (2006/09/31) 新規追加
	 * @serialData
	 *
	 * @param	strm	ObjectInputStreamオブジェクト
	 * @see #release2()
	 */
	private void readObject( final ObjectInputStream strm ) throws IOException , ClassNotFoundException {
		strm.defaultReadObject();
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	@Override
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION		)
				.println( "tableId"			,tableId		)
				.println( "classId"			,classId		)
				.println( "modifyType"		,modifyType		)
				.println( "selectedAll"		,selectedAll	)
				.println( "keys"			,keys			)
				.println( "vals"			,vals			)
				.println( "dbid"			,dbid			) // 4.2.4.0 (2008/06/23)
				.println( "body"			,body			) // 4.2.4.0 (2008/06/23)
				.fixForm().toString() ;
	}
}
