/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.html.ViewCalendarParam;

import static org.opengion.fukurou.util.StringUtil.nval ;

import java.io.ObjectOutputStream;
import java.io.ObjectInputStream;
import java.io.IOException;

/**
 * viewタグの viewFormType が HTMLCalendar の場合にパラメータを設定します。
 *
 * カレンダーを表示する、ViewForm_HTMLCalendar クラスに対して、各種パラメータを
 * 設定します。
 * パラメータが設定されていない場合は、ViewCalendarParam の初期値が使用されます。
 * (パラメータを使用するには、viewタグのuseParam 属性をtrueに設定する必要があります。)
 *
 * @og.formSample
 * ●形式：&lt;og:calendarParam ymKey="YYYYMM" dayKey="DY" firstWeek="0" /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:calendarParam
 *       viewKeys           【TAG】行のキーとなるカラム名を複数指定します(事業所やWCなど)
 *       ymKey              【TAG】行のキーとなる年月を表すカラム名を指定します
 *       dayKey             【TAG】日付け(休日フラグ)のキーとなるカラム名の英字部分を指定します(DY1 ～ DY31)
 *       valueKey           【TAG】データの値となるカラム名の英字部分を指定します(初期値は使用しません)
 *       valueBRFlag        【TAG】データの値を使用する場合に、日付けとの関係で、BR を入れるかどうか指定します
 *       firstWeek          【TAG】カレンダ表示の週の初めを指定します( 0は、日曜日から、1は月曜日から)
 *       headerLocale       【TAG】週表示ヘッダーを表示する時の、Locale(language のみ) を指定します
 *       columnSize         【TAG】カレンダを横に並べる場合の数量を指定します(初期値: 3)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *     ViewFormTag の viewFormType が、ViewCalendarParam の場合に使用します。
 *     useParam 属性を設定しておかないと、使用されません。
 *     &lt;og:view
 *         <b>viewFormType = &quot;HTMLCalendar&quot;</b>
 *         command      = &quot;{&#064;command}&quot;
 *         startNo      = &quot;0&quot;
 *         pageSize     = &quot;20&quot;
 *         <b>useParam     = &quot;true&quot;</b>
 *     &gt;
 *         &lt;og:calendarParam
 *              viewKeys     = "CDJGS"
 *              ymKey        = "YYYYMM"
 *              dayKey       = "DY"
 *              valueKey     = ""
 *              valueBRFlag  = "true"
 *              firstWeek    = "0"
 *              headerLocale = "en"
 *              columnSize   = "3"
 *         /&gt;
 *     &lt;/og:view &gt;
 *
 * @og.rev 3.5.6.0 (2004/06/18) 新規作成
 * @og.group 画面表示
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class ViewCalendarParamTag extends ViewParamTag {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/01/31)

	/**
	 * 【TAG】行のキーとなるカラム名を複数指定します(事業所やWCなど)。
	 *
	 * @og.tag
	 * エンジン標準カレンダ(GE13)では、年月(YYYYMM)と事業所コード(CDJGS)
	 * でユニークキーになります。また、能力カレンダなどでは、事業所、ワークセンタ
	 * で、ユニークになります。
	 * それらのキー項目を編集できるように、カラムをカンマ区切りで指定します。
	 *
	 * @og.rev 3.5.6.0 (2004/06/18) 新規作成
	 *
	 * @param	viewKeys 行のキーとなるカラム名を複数指定
	 */
	public void setViewKeys( final String viewKeys ) {
		putParam( ViewCalendarParam.VIEW_KEYS ,
				   nval( getRequestParameter( viewKeys ),null ) );
	}

	/**
	 * 【TAG】行のキーとなる年月を表すカラム名を指定します。
	 *
	 * @og.tag
	 * 現カレンダーは、年月毎に、３１個の日付けフィールドを持ったデータを
	 * 前提としています。工場や事業所などの区分は、取得時には確定させておく必要があります。
	 * 初期値は、ViewCalendarParam に依存します。(例：YYYYMM)
	 *
	 * @param	ymKey 行のキーとなる年月を表すカラム名
	 */
	public void setYmKey( final String ymKey ) {
		putParam( ViewCalendarParam.YM_KEY ,
				  nval( getRequestParameter( ymKey ),null ) );
	}

	/**
	 * 【TAG】日付け(休日フラグ)のキーとなるカラム名の英字部分を指定します(DY1 ～ DY31)。
	 *
	 * @og.tag
	 * カレンダーを表示する場合の、各日付け欄のカラム名の英字部分を指定します。
	 * 現カレンダーは、年月毎に、３１個の日付けフィールドを持ったデータを
	 * 前提としています。よって、このカラム名(英字部分)＋日付け(１～３１)を
	 * 動的に作成して、日付けカラムとして取得します。
	 * 例えば、"DY" という文字を指定した場合、DY1 ～ DY31 のカラム名を
	 * 使用して、検索結果の DBTableModel より値を取得します。
	 * 初期値は、ViewCalendarParam に依存します。(例：DY)
	 *
	 * @param	dayKey 日付け(休日フラグ)のキーとなるカラム名の英字部分
	 */
	public void setDayKey( final String dayKey ) {
		putParam( ViewCalendarParam.DAY_KEY ,
				  nval( getRequestParameter( dayKey ),null ) );
	}

	/**
	 * 【TAG】データの値となるカラム名の英字部分を指定します(初期値は使用しません)。
	 *
	 * @og.tag
	 * カレンダーを表示する場合の、各日付け欄の値のカラム名の英字部分を指定します。
	 * 現カレンダーは、年月毎に、３１個の値フィールドを持ったデータを
	 * 前提としています。よって、このカラム名(英字部分)＋日付け(１～３１)を
	 * 動的に作成して、値カラムとして取得します。
	 * 例えば、"VAL" という文字を指定した場合、VAL1 ～ VAL31 のカラム名を
	 * 使用して、検索結果の DBTableModel より値を取得します。
	 * 初期値は、ViewCalendarParam に依存します。(例：使用していません)
	 *
	 * @param	valueKey データの値となるカラム名の英字部分を指定します(初期値は使用しません)
	 */
	public void setValueKey( final String valueKey ) {
		putParam( ViewCalendarParam.VALUE_KEY ,
				  nval( getRequestParameter( valueKey ),null ) );
	}

	/**
	 * 【TAG】データの値を使用する場合に、日付けとの関係で、BR を入れるかどうか指定します。
	 *
	 * @og.tag
	 * データを表示する場合、日付けの横につけるか、日付けの下に付けるかを指定します。
	 * ただし、valueKey で、値カラムを指定した場合のみ、有効になります。
	 * 初期値は、ViewCalendarParam に依存します。(例：true)
	 *
	 * @param	valueBRFlag 日付け(休日フラグ)のキーとなるカラム名の英字部分
	 */
	public void setValueBRFlag( final String valueBRFlag ) {
		putParam( ViewCalendarParam.VALUE_BR_FLAG_KEY ,
				  nval( getRequestParameter( valueBRFlag ),null ) );
	}

	/**
	 * 【TAG】カレンダ表示の週の初めを指定します( 0は、日曜日から、1は月曜日から)。
	 *
	 * @og.tag
	 * カレンダの週表示で、どの曜日から表示を始めるかを指定します。
	 * 0は、日曜日から、1は月曜日から表示を開始します。
	 * 初期値は、ViewCalendarParam に依存します。(例：0 日曜日から)
	 *
	 * @param	firstWeek カレンダ表示の週の初め
	 */
	public void setFirstWeek( final String firstWeek ) {
		putParam( ViewCalendarParam.FIRSTWEEK_KEY ,
				   nval( getRequestParameter( firstWeek ),null ) );
	}

	/**
	 * 【TAG】週表示ヘッダーを表示する時の、Locale(language のみ) を指定します。
	 *
	 * @og.tag
	 * 現実装では、"ja" と "en" のみ指定可能です。
	 * 将来的には、SimpleDateFormat を使用し、多言語対応しますが、現段階では、
	 * (日、月、火、水、木、金、土)と、(SUN,MON,TUE,WED,THU,FRI,SAT)の
	 * ２種類を切り替えているだけです。
	 * どの曜日から表示を始めるかは、firstWeek 属性で設定します。
	 * 初期値は、ViewCalendarParam に依存します。(例："en")
	 *
	 * @param	headerLocale ヘッダーの Locale(language のみ)
	 */
	public void setHeaderLocale( final String headerLocale ) {
		putParam( ViewCalendarParam.HEADER_LOCALE_KEY ,
				   nval( getRequestParameter( headerLocale ),null ) );
	}

	/**
	 * 【TAG】カレンダを横に並べる場合の数量を指定します(初期値: 3)。
	 *
	 * @og.tag
	 * カレンダを複数並べる場合の横方向の件数を指定します。
	 * １年を表す場合、３＊４ や ２＊６ のカレンダが一般的です。
	 * ここでは、横方向(３や２)の件数を指定することで、レイアウトを
	 * 指定します。
	 *
	 * @og.rev 3.5.6.0 (2004/06/18) 新規作成
	 *
	 * @param	columnSize カレンダを横に並べる場合の数量(初期値: 3)
	 */
	public void setColumnSize( final String columnSize ) {
		putParam( ViewCalendarParam.COLUMN_SIZE_KEY ,
				   nval( getRequestParameter( columnSize ),null ) );
	}

	/**
	 * タグの名称を、返します。
	 * 自分自身のクラス名より、自動的に取り出せないため、このメソッドをオーバーライドします。
	 *
	 * @og.rev 4.0.0 (2005/01/31) 新規追加
	 *
	 * @return  タグの名称
	 */
	@Override
	protected String getTagName() {
		return "calendarParam" ;
	}

	/**
	 * シリアライズ用のカスタムシリアライズ書き込みメソッド
	 *
	 * @og.rev 4.0.0 (2006/09/31) 新規追加
	 * @serialData
	 *
	 * @param	strm	ObjectOutputStreamオブジェクト
	 */
	private void writeObject( final ObjectOutputStream strm ) throws IOException {
		strm.defaultWriteObject();
	}

	/**
	 * シリアライズ用のカスタムシリアライズ読み込みメソッド
	 *
	 * ここでは、transient 宣言された内部変数の内、初期化が必要なフィールドのみ設定します。
	 *
	 * @og.rev 4.0.0 (2006/09/31) 新規追加
	 * @serialData
	 *
	 * @param	strm	ObjectInputStreamオブジェクト
	 * @see #release2()
	 */
	private void readObject( final ObjectInputStream strm ) throws IOException , ClassNotFoundException {
		strm.defaultReadObject();
	}
}
