/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

// import java.util.Calendar;
// import java.util.Locale;
// import java.text.DateFormat;
// import java.text.SimpleDateFormat;

import org.opengion.fukurou.util.HybsDateUtil;

/**
 * 文字列の日付属性(年/月)の半角の日付を扱う為の、カラム属性を定義します。
 *
 * YYYYMM に対応している必要があります。
 * ただし、日付の整合性チェックは行いませんが、valueAdd( String value )による
 * 日付(月)の加算時には、0000XX と 9999XX は、加算されず、それ以外は、
 * 正式な日付で加算されます。
 *
 * タイプチェックとして、以下の条件を判定します。
 * ・文字列長は、直接計算で文字数との比較
 * ・日付使用文字チェック「('0' &gt; c || '9' &lt; c)以外」エラー
 * ・文字パラメータの 正規表現チェック
 *
 * @og.rev 3.6.0.0 (2004/09/17) 新規作成
 * @og.group データ属性
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DBType_YM extends DBType_YMD {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	/**
	 * String引数の文字列を＋１した文字列を返します。
	 * ここでは、YYYYMM 形式のデータに、＋１ するので、月を＋１します。
	 * 
	 * 0000XX と、9999XX は、引数の値そのままを返します。
	 * 引数が null の場合と、ゼロ文字列("")の場合は,物理的初期設定値(String getDefault())
	 * の値を返します。
	 * それ以外の場合は、日付計算を行い、月を加算します。よって、年月フォーマットに
	 * 合わないデータを指定した場合は、エラーになりますので、ご注意ください。
	 *
	 * @og.rev 5.5.7.2 (2012/10/09) HybsDateUtil を利用するように修正します。
	 *
	 * @param	value	String引数の文字列
	 *
	 * @return  String引数の文字列を＋１した文字列
	 */
	@Override
	public String valueAdd( final String value ) {
		if( value == null || value.length() == 0 ) { return getDefault(); }
		if( value.startsWith( "0000" ) || value.startsWith( "9999" ) ) { return value; }	// 特別な値

		return HybsDateUtil.getMonthPlus( value , 1 );		// 5.5.7.2 (2012/10/09) HybsDateUtil を利用

// 		Calendar cal = Calendar.getInstance();
// 
// 		int year  = Integer.parseInt( value.substring( 0,4 ) );
// 		int month = Integer.parseInt( value.substring( 4,6 ) ) - 1;
// 
// 		cal.set( year,month,1 );
// 		cal.add( Calendar.MONTH , 1 );		// 月を１月加算します。
// 
// 		DateFormat formatter = new SimpleDateFormat( "yyyyMM",Locale.JAPAN );
// 		return formatter.format( cal.getTime() );
	}

	/**
	 * エディターで編集されたデータを登録する場合に、データそのものを
	 * 変換して、実登録データを作成します。
	 * 例えば,大文字のみのフィールドなら、大文字化します。
	 * 実登録データの作成は、DBType オブジェクトを利用しますので,
	 * これと Editor とがアンマッチの場合は、うまくデータ変換
	 * されない可能性がありますので、注意願います。
	 *
	 * @og.rev 5.5.7.2 (2012/10/09) HybsDateUtil を利用するように修正します。
	 *
	 * @param	value	(一般に編集データとして登録されたデータ)
	 * @return  修正後の文字列(一般にデータベースに登録するデータ)
	 */
	@Override
	public String valueSet( final String value ) {
		if( value == null || value.length() == 0 ) { return getDefault(); }

		return HybsDateUtil.parseDate( value , 6 );		// 不要な記号等を削除し、桁数をそろえる。

// 		String rtn = ( value == null ) ? "" : value.trim();
// 
// 		if( rtn.length() == 4 ) { rtn = "20" + rtn ; }
// 		else if( rtn.length() > 6 ) { rtn = rtn.substring(0,6) ; }
// 		else if( rtn.startsWith( "0000" ) ) { rtn = "000001"; }
// 		else if( rtn.startsWith( "9999" ) ) { rtn = "999912"; }
// 
// 		return rtn;
	}
}
