/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.view;

import java.util.List;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.html.TableFormatter;

/**
 * JavaScript のツリー階層を持ったテーブル表示を行う、ツリーテーブル表示クラスです。
 *
 * AbstractViewForm により、setter/getterメソッドのデフォルト実装を提供しています。
 * 各HTMLのタグに必要な setter/getterメソッドのみ，追加定義しています。
 *
 * AbstractViewForm を継承している為,ロケールに応じたラベルを出力させる事が出来ます。
 *
 * @og.group 画面表示
 *
 * @version  4.0
 * @author   Hiroki Nakamura
 * @since    JDK5.0,
 */
public class ViewForm_HTMLCustomTreeBOM extends ViewForm_HTMLTable  {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

//	public static final String COLUMN_LEVEL_KEY = "COLUMN_LEVEL";	// 5.1.9.0 (2010/08/01) 廃止

	private TableFormatter 		headerFormat	= null;
	private TableFormatter[]	bodyFormats		= null;
	private int					bodyFormatsCount = 0;

	private static final int BODYFORMAT_MAX_COUNT = 10;

	// 4.3.4.4 (2009/01/01)
//	/**
//	 * デフォルトコンストラクター
//	 *
//	 */
//	public ViewForm_HTMLCustomTreeBOM() {
//		super();
//	}

	/**
	 * DBTableModel から HTML文字列を作成して返します。
	 * startNo(表示開始位置)から、pageSize(表示件数)までのView文字列を作成します。
	 * 表示残りデータが pageSize 以下の場合は,残りのデータをすべて出力します。
	 *
	 * @og.rev 4.3.1.0 (2008/09/08) フォーマットが設定されていない場合のエラー追加
	 *
	 * @param  stNo     表示開始位置
	 * @param  pgSize   表示件数
	 *
	 * @return  DBTableModelから作成された HTML文字列
	 */
	@Override
	public String create( final int stNo, final int pgSize )  {
		// このクラスでは、テーブル全データを使用します。
		if( getRowCount() == 0 ) { return ""; }	// 暫定処置

		// 4.3.1.0 (2008/09/08)
		if( headerFormat == null ) {
			String errMsg = "ViewTagで canUseFormat() = true の場合、Formatter は必須です。";
			throw new HybsSystemException( errMsg );
		}

		int startNo = 0;
		int pageSize = getRowCount();

		int lastNo = getLastNo( startNo, pageSize );

		StringBuilder out = new StringBuilder( HybsSystem.BUFFER_LARGE );

		headerFormat.makeFormat( getDBTableModel() );

		if( bodyFormatsCount == 0 ) {
			bodyFormats[0] = headerFormat ;
			bodyFormatsCount ++ ;
		}
		else {
			for( int i=0; i<bodyFormatsCount; i++ ) {
				bodyFormats[i].makeFormat( getDBTableModel() );
			}
		}

		out.append( getHeader() );

		int level;
		boolean isFld;
		for( int row=startNo; row<lastNo; row++ ) {
			// カラム==０は、レベルを指定する。
			level = Integer.parseInt( getValueLabel(row,0) );
			isFld = false;
			if( row+1<lastNo ) {
				int nextLevel = Integer.parseInt( getValueLabel(row+1,0) );
				isFld = ( level < nextLevel ) ? true : false ;
			}
			out.append( getLevelScript( level,isFld ) );

			// 開始
			for( int i=0; i<bodyFormatsCount; i++ ) {
				TableFormatter bodyFormat = bodyFormats[i];

				int cl = 0;
				for( ; cl < bodyFormat.getLocationSize(); cl++ ) {
					String fmt = bodyFormat.getFormat(cl);
					int loc = bodyFormat.getLocation(cl);
					if( ! bodyFormat.isNoClass() && loc >= 0 ) {
						StringBuilder newtg = new StringBuilder( HybsSystem.BUFFER_LARGE );
						newtg.append("<td class=\"");
						newtg.append( getColumnDbType(loc) );
						newtg.append("\" ");
						String tdclass = newtg.toString();
						fmt = StringUtil.replace( bodyFormat.getFormat(cl) ,"<td", tdclass );
					}
					out.append( fmt );
					if( loc >= 0 ) {
						switch( bodyFormat.getType(cl) ) {
						case '#' : out.append( getColumnLabel(loc) );		break;
						case '$' : out.append( getRendererValue(row,loc) );	break;
						case '!' : out.append( getValue(row,loc) );			break;
						default  : out.append( getValueLabel(row,loc) );	break;
						}
					}
				}
				out.append( StringUtil.replace( bodyFormat.getFormat(cl), "</tr>", "" ) );
			}
			// 終了

			out.append( "', '', 'gold')" );
			if( level != 0 ) {
				out.append( ")" );
			}
			out.append( HybsSystem.CR );
		}
		out.append( getFutter() );

		return out.toString();
	}

	/**
	 * DBTableModel から テーブルのヘッダータグ文字列を作成して返します。
	 * JavaScript の TreeBody では、JavaScriptに関連する定義もこのヘッダーに
	 * 含めます。
	 *
	 * @return  テーブルのヘッダータグ文字列
	 */
	@Override
	protected String getHeader() {
		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_MIDDLE );

		buf.append("<table border=\"0\" cellspacing=\"2\" cellpadding=\"0\"  summary=\"bomTable\" id=\"viewTable\">"); // 3.9.0.1 (2007/12/18)
		buf.append( HybsSystem.CR );
		buf.append("<script type=\"text/javascript\">");
		buf.append( HybsSystem.CR );
		buf.append("<!--");
		buf.append( HybsSystem.CR );
		buf.append("aux0 = gFld('");
		// 開始
		int cl = 0;
		for( ; cl < headerFormat.getLocationSize(); cl++ ) {
			buf.append( StringUtil.replace( headerFormat.getFormat(cl) ,"td","th" ));
			int loc = headerFormat.getLocation(cl);
			if( loc >= 0 ) { buf.append( getColumnLabel(loc) ); }
			// ヘッダーフォーマット部では、何もしません。
		}
		buf.append( StringUtil.replace( StringUtil.replace( headerFormat.getFormat(cl) ,"td","th" ), "</tr>", "" ) );
		// 終了

		buf.append("', '', 'gold')");
		buf.append( HybsSystem.CR );

		return buf.toString();
	}

	/**
	 * DBTableModel から テーブルのフッタータグ文字列を作成して返します。
	 * JavaScript の TreeBody では、JavaScriptに関連する定義もこのフッターに
	 * 含めます。
	 *
	 * @return  テーブルのフッタータグ文字列
	 */
	protected String getFutter() {
		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_MIDDLE );

		buf.append("initializeDocument()").append( HybsSystem.CR );
		buf.append("//-->").append( HybsSystem.CR );
		buf.append("</script>").append( HybsSystem.CR );
		buf.append("</table>").append( HybsSystem.CR );

		return buf.toString();
	}

	/**
	 * 行のレベルに応じた JavaScript関数のヘッダー部分を返します。
	 *
	 * @og.rev 3.5.2.1 (2003/10/27) JavaScript 内のダブルコーテーションをシングルコーテーションに変更する。
	 *
	 * @param	lvl		ツリーのレベル
	 * @param	isFld	フォルダかどうか[true:フォルダ/false:最下層]
	 *
	 * @return  JavaScript関数のヘッダー部分
	 */
	private String getLevelScript( final int lvl,final boolean isFld ) {

		String auxX = "\taux" + ( lvl );
		String auxY = "aux" + ( lvl-1 );

		final String rtn ;
		if( isFld ) {
			rtn = auxX + " = insFld(" + auxY + ", gFld('";
		}
		else {
			rtn = "\tinsFld(" + auxY + ", gLnk('CONTENTS','";
		}

		return rtn;
	}

	/**
	 * フォーマットを設定します。
	 *
	 * @param	list	TableFormatterのリスト
	 */
	@Override
	public void setFormatterList( final List<TableFormatter> list ) {		// 4.3.3.6 (2008/11/15) Generics警告対応
		bodyFormats = new TableFormatter[BODYFORMAT_MAX_COUNT];

		bodyFormatsCount = 0;
		for( int i=0; i<list.size(); i++ ) {
			TableFormatter format = list.get( i );		// 4.3.3.6 (2008/11/15) Generics警告対応
			switch( format.getFormatType() ) {
			case TYPE_HEAD : headerFormat = format; break;
			case TYPE_BODY : bodyFormats[bodyFormatsCount++] = format; break;
			default : String errMsg = "FormatterType の定義外の値が指定されました。";
			// 4.3.4.4 (2009/01/01)
					  throw new HybsSystemException( errMsg );
			}
		}

		if( headerFormat == null ) {
			String errMsg = "og:thead タグの、フォーマットの指定は必須です。";
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * フォーマットメソッドを使用できるかどうかを問い合わせます。
	 *
	 * @return  使用可能(true)/ 使用不可能 (false)
	 */
	@Override
	public boolean canUseFormat() {
		return true;
	}

	/**
	 * ビューで表示したカラムの一覧をカンマ区切りで返します。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規追加
	 *
	 * @return	ビューで表示したカラムの一覧
	 */
	@Override
	public String getViewClms() {
		DBTableModel table = getDBTableModel();
		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_MIDDLE );
		for( int i=0; i<headerFormat.getLocationSize(); i++ ) {
			if( buf.length() > 0 ) { buf.append( ',' ); }
			buf.append( table.getColumnName( headerFormat.getLocation( i ) ) );
		}
		return buf.toString();
	}
}
