/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.io;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModel;

import java.util.List;
import java.util.ArrayList;
import java.util.Arrays;
import java.awt.Color;
import java.sql.Connection;
import java.sql.SQLException;
import java.text.NumberFormat;
import java.text.DecimalFormat;				// 5.5.2.1 (2012/05/07)
import java.text.DecimalFormatSymbols;		// 5.5.2.1 (2012/05/07)
import java.text.FieldPosition;				// 5.5.2.1 (2012/05/07)

import org.jfree.data.general.Dataset;
import org.jfree.data.general.DefaultValueDataset;
import org.jfree.chart.plot.ValueMarker;
import org.jfree.chart.axis.NumberAxis;
import org.jfree.chart.axis.NumberTickUnit;
import org.jfree.chart.renderer.category.CategoryItemRenderer;
import org.jfree.chart.renderer.category.BarRenderer;
import org.jfree.chart.renderer.category.LineAndShapeRenderer;
import org.jfree.chart.labels.CategoryItemLabelGenerator;
import org.jfree.chart.labels.StandardCategoryItemLabelGenerator;
import org.jfree.chart.urls.CategoryURLGenerator;
import org.jfree.ui.RectangleInsets;
import org.jfree.ui.TextAnchor;

/**
 * ChartDataset は、Dataset および、JFreeChart の個別属性を管理しているデータ管理クラスです。
 * JFreeChart のグラフ合成機能のための、個別のグラフ描画属性を管理しています。
 * 内部的に、引数の文字列を JFreeChart の各オブジェクトへの変換なども行っています。
 *
 * @version  0.9.0	2007/06/21
 * @author	 Kazuhiko Hasegawa
 * @since	 JDK1.1,
 */
public class ChartDataset {

	private Dataset		dataset			= null;		// Dataset オブジェクト
	private String		query			= null;		// 実行クエリー
	private String		chartType		= null;		// チャートの種類
	private boolean		baseLegend		= true;		// シリーズ単位の凡例 の表示可否
	private String		valueLabel		= null;
	private double[]	markValues		= null;		// doubleの配列
	private Color[]		markColors		= new Color[] { Color.RED };	// Colorの配列
	private Color[]		markOverColors	= null;		// 4.0.3.0 (2008/01/07)	マーカーラインの超過色
	private boolean		useMarkAnchor	= true;		// 4.1.1.0 (2008/02/04) 設定値をラベル表示するかどうか
	private int			dynamicOCNo		= -1;		// 4.1.1.0 (2008/02/04) 動的なマーカーラインの基準シリーズ番号
	private int			dynamicOCadrs	= -1;		// 4.1.1.0 (2008/02/21) 動的なマーカーラインの基準番号
	private String		lowerBound		= null;		// double
	private String		upperBound		= null;		// double
	private String		tickSize		= null;		// double	4.0.3.0 (2008/01/07)
	private boolean		useGradient  	= false;
	private boolean		shapesVisible	= true;
	private Color[]		shapeColors		= null;		// 4.0.3.0 (2008/01/07) データ毎にShapeを切り替える時の色指定
	private String		shapeScale		= null;		// 4.1.1.0 (2008/02/04) 基本形 Shape の表示倍率(double)
	private boolean		useDottedLine	= false;
	private Color[]		seriesColors	= null;		// シリーズに対応した色指定
	private boolean		isLabelsVisible	= true;		// 4.0.2.0 (2007/12/20) (縦軸)のラベルを表示するかどうか
	private boolean		isMarksVisible	= true;		// 4.0.2.0 (2007/12/20) (縦軸)のマーカーを表示するかどうか
	private boolean		isItemValVisible= false;	// 4.1.1.0 (2008/02/04) データの値(itemText)を表示するかどうか
	private boolean		isItemLastVisible= false;	// 4.1.2.0 (2008/03/12) データの値(itemText)で最後のみ表示するかどうか
	private boolean		useItemLabelSep	= true;		// 4.1.2.0 (2008/03/12) データの値(itemText)の表示に桁区切り文字を使用するかどうか
	private String		barWidth		= null;		// 4.0.3.0 (2008/01/07) バー幅(double)
	private String		maxBarWidth		= null;		// 4.0.2.0 (2007/12/20) バー幅の最大値(0.0～1.0)
//	private double		chartOffset		= 0.0;		// 4.1.1.0 (2008/02/04) グラフの書き出し位置(double)
	private String		itemMargin		= null;		// 4.0.2.0 (2007/12/20) バーアイテムのマージン(0.0～1.0)
	private String		visibleLimit	= null;		// 4.0.3.0 (2008/01/07) 表示下限値(これ以下のデータは未表示)
	private String		valueInset		= null;		// 4.0.3.0 (2008/01/07) 縦軸表示領域INSET値
	private double		domainMargin	= 0.0;		// 4.1.1.0 (2008/02/14) グラフの書き出し位置の調整比率

	private String		timeFormatType	= null;		// 5.5.2.1 (2012/05/07) 時刻の表示方法(E1/E2/E3/H1/H2/H3)
	private boolean		useVerticalLabels = false;	// 5.5.2.1 (2012/05/07) ラベルの表示向き(横書き:false/縦書き:true)

	/**
	 * Connection と query 文字列から、Dataset オブジェクトを作成します。
	 *
	 * setChartType( String ) で指定されたチャートタイプに応じて、Datasetオブジェクト
	 * を構築します。
	 * Dataset クラスは、"org.jfree.data.jdbc.JDBCXXXXDataset" の XXXX の箇所を
	 * ChartFactory クラスのチャートタイプ変換表に基づいて求めます。
	 * 現状は、Category , Pie , XY の３種類と、Value が指定可能です。
	 * ※ Value は、conn や sql を使用していません。値は、markValues を使用しています。
	 *    これは、現状の暫定方法であり、今後は変わる可能性があります。
	 *
	 * @og.rev 4.0.0.0 (2007/11/28) new Double(double) ⇒ Double.valueOf(double) 変更
	 *
	 * @param	conn	Datasetの取得先のコネクション
	 * @param	sql 	取得するクエリー文字列
	 * @throws SQLException データベース実行エラーが発生した場合
	 */
	public void makeDataset( final Connection conn,final String sql ) throws SQLException {
		dataset = ChartFactory.newDataset( conn,sql,chartType );
		query   = sql ;

		// 暫定的対応。本当は、SQL を実行してその値をセットすべき
		if( dataset instanceof DefaultValueDataset ) {
			if( markValues != null && markValues.length > 0 ) {
				((DefaultValueDataset)dataset).setValue( Double.valueOf( markValues[0] ) );
			}
			else {
				String errMsg = "markValues が指定されていません。";
				throw new HybsSystemException( errMsg );
			}
		}
	}

	/**
	 * DBTableModelオブジェクトから、Dataset オブジェクトを作成します。
	 *
	 * setChartType( String ) で指定されたチャートタイプに応じて、Datasetオブジェクト
	 * を構築します。
	 *
	 * 現状は、Category , Pie , XY の３種類と、Value が指定可能です。
	 * ※ Value は、DBTableModel を使用していません。値は、markValues を使用しています。
	 *    これは、現状の暫定方法であり、今後は変わる可能性があります。
	 *
	 * @og.rev 5.3.0.0 (2010/12/01) 新規追加
	 *
	 * @param	table	Datasetの取得先のテーブルモデル
	 */
	public void makeDataset( final DBTableModel table ) {
		dataset = ChartFactory.newDataset( table,chartType );

		// 暫定的対応。本当は、SQL を実行してその値をセットすべき
		if( dataset instanceof DefaultValueDataset ) {
			if( markValues != null && markValues.length > 0 ) {
				((DefaultValueDataset)dataset).setValue( Double.valueOf( markValues[0] ) );
			}
			else {
				String errMsg = "markValues が指定されていません。";
				throw new HybsSystemException( errMsg );
			}
		}
	}

	/**
	 * NumberAxis オブジェクトを作成します。
	 *
	 * NumberAxis オブジェクトは、Value(縦軸)の属性を設定する為に使用される
	 * オブジェクトです。
	 * これに、各種属性を設定して、値を返します。
	 * 属性としては、isLineVisible、isLabelsVisible、isMarksVisible、
	 * range(lowerBound、upperBound) です。
	 *
	 * @og.rev 4.0.2.0 (2007/12/20) 新規追加
	 * @og.rev 4.0.3.0 (2008/01/07) 縦軸レンジを個別設定、チック幅指定
	 * @og.rev 4.1.1.0 (2008/02/04) HybsNumberAxis 採用
	 * @og.rev 4.1.1.0 (2008/02/04) setAxisLineVisible を、isMarksVisible 値で設定する。
	 * @og.rev 5.5.2.1 (2012/05/07) timeFormatType 、useVerticalLabels 属性を追加する。
	 *
	 * @return	縦軸の属性(NumberAxisオブジェクト)
	 */
	protected NumberAxis makeNumberAxis() {
		NumberAxis axis = new HybsNumberAxis( valueLabel );

		// 縦軸レンジ幅の軸設定
		if( lowerBound != null ) {
			axis.setLowerBound( Double.parseDouble( lowerBound ) );
		}

		if( upperBound != null ) {
			axis.setUpperBound( Double.parseDouble( upperBound ) );
		}

		if( tickSize != null ) {
			axis.setTickUnit( new NumberTickUnit( Double.parseDouble( tickSize ) ) );
		}

		axis.setTickLabelsVisible( isLabelsVisible );
		axis.setTickMarksVisible ( isMarksVisible );
		axis.setAxisLineVisible ( isMarksVisible );		// 4.1.1.0 (2008/02/04)

		if( valueInset != null ) {
			double rightInset = Double.parseDouble( valueInset );
			RectangleInsets base = axis.getTickLabelInsets();
			RectangleInsets inset = new RectangleInsets(
					base.getUnitType(),		// UnitType unitType
					base.getTop(),			// double top
					base.getLeft(),			// double left
					base.getBottom(),		// double bottom
					base.getRight() + rightInset	// double right
			);
			axis.setTickLabelInsets( inset ) ;
		}

		// 5.5.2.1 (2012/05/07) 時刻の表示方法(E1/E2/E3/H1/H2/H3)
		if( timeFormatType != null ) {
			axis.setNumberFormatOverride( new HybsTimeFormat( timeFormatType ) );
		}

		// 5.5.2.1 (2012/05/07) ラベルの表示向き(横書き:false/縦書き:true)
		if( useVerticalLabels ) {
			axis.setVerticalTickLabels( true );		// 処理的には冗長。
		}

		return axis;
	}

	/**
	 * 検索を行う クエリー文字列を返します。
	 *
	 * これは、makeDataset( Connection ,String ) メソッドで指定した、
	 * sql 属性の設定値そのものです。
	 *
	 * @return	取得するクエリー文字列
	 */
	public String getQuery() { return query; }

	/**
	 * Dataset オブジェクトを取得します。
	 *
	 * makeDataset( Connection,String ) で、予めオブジェクトを構築しておく
	 * 必要があります。
	 * Dataset クラスは、"org.jfree.data.jdbc.JDBCXXXXDataset" の XXXX の箇所を
	 * ChartFactory クラスのチャートタイプ変換表に基づいて求めます。
	 * 現状は、Category , Pie , XY の３種類と、Value が指定可能です。
	 *
	 * @return	Datasetオブジェクト
	 * @see		#makeDataset( Connection ,String )
	 */
	protected Dataset getDataset() { return dataset; }

	/**
	 * Dataset オブジェクトを作成する時のチャートタイプを指定します。
	 *
	 * チャートタイプ は、外部からチャートを指定するのに便利なように、キー化
	 * されています。このキーに基づいて、ChartFactory クラスの
	 * チャートタイプ変換表に基づいて、レンデラーや、データセットを作成します。
	 *
	 * @param	type チャートタイプ
	 */
	public void setChartType( final String type ) {
		if( type != null && type.length() > 0 ) {
			chartType = type;
		}
	}

	/**
	 * Dataset オブジェクトを作成する時のチャートタイプを取得します。
	 *
	 * チャートタイプ は、外部からチャートを指定するのに便利なように、キー化
	 * されています。このキーに基づいて、ChartFactory クラスの
	 * チャートタイプ変換表に基づいて、レンデラーや、データセットを作成します。
	 *
	 * @return	チャートタイプ
	 */
	protected String getChartType() { return chartType; }

	/**
	 * シリーズ単位の凡例 の表示可否を設定します。
	 *
	 * @param	baseLeg	シリーズ単位の凡例 の表示可否
	 */
	public void setBaseLegend( final boolean baseLeg ) {
		baseLegend = baseLeg;
	}

	/**
	 * Renderer 関連の共通インターフェースを返します。
	 *
	 * setChartType( String )で設定された、チャートタイプに応じた、レンデラーを
	 * 返します。
	 * 対応表は、ChartFactory クラスに持っています。
	 * 基本的には、チャートタイプ は、この、レンデラー名称から決定しています。
	 *
	 * @og.rev 4.1.1.0 (2008/02/04) useDynamicOC , isItemValVisible , shapeScale 属性追加
	 * @og.rev 4.1.1.0 (2008/02/04) クリッカブル・マップの設定を移動、serNo 引数追加
	 * @og.rev 4.1.1.0 (2008/02/04) 返す型を CategoryItemRenderer に変更
	 * @og.rev 4.1.1.0 (2008/02/16) HybsStackedBarRenderer に domainMargin を設定
	 * @og.rev 4.1.1.0 (2008/02/19) Shapes の設定をこちらに移動。
	 * @og.rev 4.1.2.0 (2008/03/12) isItemLastVisible の設定
	 *
	 * @param	serNo	シリアルNo
	 * @param	urlGen CategoryURLGeneratorオブジェクト
	 *
	 * @return	レンデラーの共通インターフェース
	 * @see		org.jfree.chart.renderer.category.CategoryItemRenderer
	 * @see		org.jfree.chart.renderer.xy.XYItemRenderer
	 */
	protected CategoryItemRenderer getRenderer( final int serNo , final CategoryURLGenerator urlGen ) {
		TypeRenderer typeRen = ChartFactory.getTypeRenderer( chartType );
		CategoryItemRenderer rend = (CategoryItemRenderer)typeRen.getRenderer();

		// 4.1.1.0 (2008/02/19) LineAndShapeRenderer オブジェクトの場合の設定
		if( rend instanceof LineAndShapeRenderer ) {
			LineAndShapeRenderer lsRend = (LineAndShapeRenderer)rend;

			// シェープ(四角)表示。serNo 内の表示は、個別に行います。
			lsRend.setBaseShapesVisible( shapesVisible );
			if( serNo >= 0 ) {
				lsRend.setSeriesShapesVisible( serNo,true );
			}

			// 4.1.1.0 (2008/02/04) 繰返しshapeの開始番号(初期値:０)
			ShapeList shapeList = new ShapeList();
			for( int i=0; i<shapeList.size(); i++ ) {
				lsRend.setSeriesShape( i,shapeList.getShape(i) );
			}
		}

		// 4.0.3.0 (2008/01/07) HybsLineRenderer 特別処理
		if( rend instanceof HybsLineRenderer ) {
			HybsLineRenderer hyRend = (HybsLineRenderer)rend ;
			if( shapeColors != null ) {
				hyRend.setShapeColors( shapeColors );
			}

			// 4.1.1.0 (2008/02/04) shapeの大きさの倍率(初期値:null)
			hyRend.setShapeScale( shapeScale );

			// 縦軸マーカーの超過時Shape色指定(未使用時は null)
			if( markOverColors != null ) {
				hyRend.setValueMarkOverColors( getValueMarkOverColors(),dynamicOCNo );
			}

			// 表示下限値(これ以下のデータは未表示)の値(double)を指定
			if( visibleLimit != null ) {
				hyRend.setVisibleLimit( Double.valueOf( visibleLimit ) );
			}
		}

		// 4.0.2.0 (2007/12/20) BarRenderer オブジェクトの場合にマージン等を設定します。
		if( rend instanceof BarRenderer ) {
			BarRenderer barRend = (BarRenderer)rend ;

			// 棒の間を表示領域を１として小数点以下の数値で幅を設定する(0.0～1.0)
			if( itemMargin != null ) {
				double margin = Double.parseDouble( itemMargin );
				barRend.setItemMargin( margin ) ;
			}

			// 棒の幅を設定す　表示領域を１として小数点以下の数値で幅を設定する(0.0～1.0)
			if( maxBarWidth != null ) {
				barRend.setMaximumBarWidth( Double.parseDouble( maxBarWidth ) ) ;
			}
		}

		// 4.1.1.0 (2008/02/16) HybsStackedBarRenderer オブジェクトの場合に domainMargin を設定します。
		if( rend instanceof HybsStackedBarRenderer ) {
			HybsStackedBarRenderer stBarRend = (HybsStackedBarRenderer)rend ;

			stBarRend.setDomainMargin( domainMargin ) ;
		}

		// シリーズ単位の凡例 の表示可否
		rend.setBaseSeriesVisibleInLegend( baseLegend );

		// 4.1.1.0 (2008/02/04) データの値(itemText)を表示するかどうか
		if( isItemValVisible ) {
//			CategoryItemLabelGenerator labelGen = new StandardCategoryItemLabelGenerator();

			// 4.1.2.0 (2008/03/12) 桁区切り文字を使用するかどうか
			NumberFormat numForm = NumberFormat.getInstance();
			numForm.setGroupingUsed( useItemLabelSep );

			CategoryItemLabelGenerator labelGen
				= new StandardCategoryItemLabelGenerator(
							StandardCategoryItemLabelGenerator.DEFAULT_LABEL_FORMAT_STRING,
							numForm
				);
			rend.setBaseItemLabelGenerator( labelGen );

			if( serNo >= 0 ) {
				rend.setSeriesItemLabelsVisible( serNo,true );
			}
			else {
				rend.setBaseItemLabelsVisible( true );
			}
			// 4.1.1.0 (2008/02/04) 動的なマーカーラインを使用する場合、
			// 指定の番号は閾値なので、表示しません。
			if( dynamicOCNo >= 0 ) {
				rend.setSeriesItemLabelsVisible( dynamicOCNo,false );		// 閾値
			}

			// 4.1.2.0 (2008/03/12) isItemLastVisible の設定
			if( isItemLastVisible && rend instanceof HybsDrawItem ) {
				((HybsDrawItem)rend).setItemLabelLastVisible( true ) ;
			}
		}

		// クリッカブル・マップ
		if( urlGen != null ) {
			if( serNo >= 0 ) {
				rend.setSeriesItemURLGenerator( serNo,urlGen );
			}
			else {
				rend.setBaseItemURLGenerator( urlGen );
			}
		}

		return rend;
	}

	/**
	 * 縦軸の表示名称を指定します。
	 *
	 * @param   valLbl 縦軸の表示名称
	 */
	public void setValueLabel( final String valLbl ) {
		if( valLbl != null && valLbl.length() > 0 ) {
			valueLabel = valLbl;
		}
	}

	/**
	 * 縦軸の表示名称を取得します。
	 *
	 * @return	縦軸の表示名称
	 */
	protected String getValueLabel() { return valueLabel; }

	/**
	 * マーカーラインの設定値配列を指定します。
	 *
	 * 指定の値にマーカーラインを引きます。
	 * 色は、setMarkColor( String ) で指定します。
	 * markColors と、あわせて設定する必要があります。
	 * markColorsが、指定されていない場合は、Color.RED で マーカーされます。
	 * dynamicOCNo を使用する場合は、そのValues の並びの箇所に、
	 * "G"  という文字列を設定して下さい。
	 * 例：2000,G  基準１は、2000の直線、基準２は、グラフ という意味。
	 *
	 * @og.rev 4.1.1.0 (2008/02/21) 基準ラインにグラフを使用する場合の位置指定
	 *
	 * @param	mkVal	マーカーラインの設定値配列
	 * @see		#setMarkColors( String[] )
	 */
	public void setMarkValues( final String[] mkVal ) {
		if( mkVal != null && mkVal.length > 0 ) {
			int size = mkVal.length;
			markValues = new double[size];
			for( int i=0; i<size; i++ ) {
				if( "G".equals( mkVal[i] ) ) {
					dynamicOCadrs = i;
					markValues[i] = 0.0;
				}
				else {
					markValues[i] = Double.parseDouble( mkVal[i] );
				}
			}
		}
	}

	/**
	 * マーカーラインの色を配列指定します。
	 *
	 * マーカーラインとは、縦軸設定のラインの事で、縦軸の設定値(markValues)に
	 * 横方向にラインを付与します。このラインの色を、カンマ区切り形式で
	 * 指定します。
	 * markValues と、あわせて設定する必要があります。
	 * 個数が異なる場合は、markValues が優先され、無指定の場合は、Color.REDで
	 * 表示されます。
	 *
	 * <pre>
	 *      BLACK   , BLUE   , CYAN , DARK_GRAY , GRAY  , GREEN , LIGHT_GRAY ,
	 *      MAGENTA , ORANGE , PINK , RED       , WHITE , YELLOW , (PURPLE)
	 *      #XXXXXX形式の16bitRGB表記 でも指定可能です。
	 * </pre>
	 *
	 * @param   mkCol マーカーラインの色配列
	 * @see		java.awt.Color#BLACK
	 * @see		#setMarkValues( String[] )
	 */
	public void setMarkColors( final String[] mkCol ) {
		if( mkCol != null && mkCol.length > 0 ) {
			int size = mkCol.length;
			markColors = new Color[size];
			for( int i=0; i<size; i++ ) {
				markColors[i] = StringUtil.getColorInstance( mkCol[i] );
			}
		}
	}

	/**
	 * マーカーライン描画時に、その設定値を表示します。
	 *
	 * マーカーラインの縦軸の設定値(markValues)に、設定値をラベル表示します。
	 * 位置は、TextAnchor.BASELINE_LEFT 固定です。
	 * 初期値は、表示する(true)です。
	 *
	 * @og.rev 4.1.1.0 (2008/02/04) 新規追加
	 *
	 * @param   useAnchor true:設定値を表示する。
	 */
	public void setUseMarkAnchor( final boolean useAnchor ) {
		useMarkAnchor = useAnchor ;
	}

	/**
	 * マーカーラインの超過時のShape色をカンマ区切り形式で複数指定します。
	 *
	 * HybsLine でのみ使用可能です。
	 * マーカーラインを使用する場合に、そのラインを超えない色、超える色を指定します。
	 * この色の指定は、マーカーラインの色やマーカーラインの設定値の個数＋１に
	 * なります。つまり、色１：設定値１：色２：設定値２：色３ となります。
	 * 色１は、設定値１より小さい場合に使用され、色２は、設定値１より大きく、
	 * 設定値２より小さい場合、色３は、設定値２より大きい場合になります。
	 *
	 * 指定文字列は、java.awt.Color クラスのstatic フィールド名で指定します。
	 * BLACK , BLUE , CYAN , DARK_GRAY , GRAY , GREEN , LIGHT_GRAY ,
	 * MAGENTA , ORANGE , PINK , RED , WHITE , YELLOW , (PURPLE) が指定できます。
	 * また、#XXXXXX形式の16bitRGB表記 でも指定可能です。
	 * markValues と、あわせて設定する必要があります。
	 * どちらかが、指定されていない場合は、マーカー表示されません。
	 * <pre>
	 *      BLACK   , BLUE   , CYAN , DARK_GRAY , GRAY  , GREEN , LIGHT_GRAY ,
	 *      MAGENTA , ORANGE , PINK , RED       , WHITE , YELLOW , (PURPLE)
	 *      #XXXXXX形式の16bitRGB表記 でも指定可能です。
	 * </pre>
	 *
	 * @og.rev 4.0.3.0 (2008/01/07) 新規追加
	 *
	 * @param	ovCol	マーカーラインの超過時のShape色
	 * @see		java.awt.Color#BLACK
	 */
	public void setMarkOverColors( final String[] ovCol ) {
		if( ovCol != null && ovCol.length > 0 ) {
			int size = ovCol.length;
			markOverColors = new Color[size];
			for( int i=0; i<size; i++ ) {
				markOverColors[i] = StringUtil.getColorInstance( ovCol[i] );
			}
		}
	}

	/**
	 * マーカーラインの超過時のShape色を規定するオブジェクトを返します。
	 *
	 * HybsLine でのみ使用可能です。
	 * 設定情報が存在しない場合は、null を返します。
	 *
	 * @og.rev 4.0.3.0 (2008/01/07) 新規追加
	 * @og.rev 4.1.1.0 (2008/02/04) markColors 属性追加
	 * @og.rev 4.1.1.1 (2008/02/25) markValues での "G" 指定を考慮
	 *
	 * @return	マーカーラインの超過時のShape色管理オブジェクト
	 */
	private ValueMarkOverColors getValueMarkOverColors() {
		ValueMarkOverColors vmoc = null;

		if( markValues != null && markOverColors != null &&
			markValues.length + 1 == markOverColors.length &&
			markValues.length >= 1 ) {
				int dynAdrs = (dynamicOCadrs >= 0) ? dynamicOCadrs : 0 ;

				vmoc = new ValueMarkOverColors(
								markValues,
								markOverColors,
								markColors[dynAdrs],
								dynamicOCadrs
						);
		}

		return vmoc ;
	}

	/**
	 * 動的なマーカーラインの基準シリーズ番号を設定します(初期値:-1)。
	 *
	 * 動的なマーカーラインを使用する場合は、基準値となるシリーズ番号を指定します。
	 *
	 * マーカーラインの最下位閾値に相当します。これは、グラフ化されますが、
	 * Shape は自動的に削除されます。
	 * この設定を使用する場合、最初のデータには、必ずShapeが付きます。それ以外の
	 * データに、Shape を付けるかどうかは、shapesVisible 属性で指定します。
	 * この線の色は、markColors で指定した、最下位の色になります。また、
	 * markValues で指定した、最下位の値は、使用されません。ただし、色指定の
	 * 関係上、設定しておく必要があります。
	 * また、isValueVisible == true で、設定値の値表示を行う場合も、最下位の
	 * 値は表示しないようにします。
	 * 初期値は、使用しない(-1)です。
	 *
	 * @og.rev 4.1.1.0 (2008/02/04) 新規追加
	 * @og.rev 4.1.1.0 (2008/02/19) setUseDynamicOC ⇒ setDynamicOCNo
	 *
	 * @param	ocNo	動的なマーカーラインの基準シリーズ番号
	 */
	public void setDynamicOCNo( final int ocNo ) {
		dynamicOCNo = ocNo;
	}

	/**
	 * マーカーラインの ValueMarker オブジェクト配列を返します。
	 *
	 * マーカーライン は、markValues(double配列) と markColors(Color配列)
	 * より、ValueMarker オブジェクトを、順次配列化していきます。
	 * useMarkAnchor == true の場合は、設定値(markValues) の Anchor を
	 * 表示します。位置は、TextAnchor.BASELINE_LEFT 固定です。
	 * dynamicOCNo が指定された場合は、最下位の閾値は使用しません。
	 *
	 * @og.rev 4.1.1.0 (2008/02/04) useMarkAnchor 属性追加
	 * @og.rev 4.1.1.0 (2008/02/04) 動的なオーバーカラー useDynamicOC 属性追加
	 *
	 * @return	マーカーラインのオブジェクト配列(not null)
	 * @see		java.awt.Color#BLACK
	 * @see		#setMarkValues( String[] )
	 * @see		#setMarkColors( String[] )
	 * @see		#setUseMarkAnchor( boolean )
	 */
	protected ValueMarker[] getValueMarkers() {
		List<ValueMarker> markerList = new ArrayList<ValueMarker>();
		int size = (markValues == null) ? 0 : markValues.length;
		for( int i=0; i<size; i++ ) {
			if( i != dynamicOCadrs ) {
				ValueMarker marker = new ValueMarker( markValues[i] );
				marker.setPaint( markColors[i]  );
				if( useMarkAnchor ) {
					marker.setLabel( String.valueOf( markValues[i] ) );
					marker.setLabelTextAnchor( TextAnchor.BASELINE_LEFT );
				}
				markerList.add( marker );
			}
		}

		return markerList.toArray( new ValueMarker[markerList.size()] );
	}

	/**
	 * チャートの縦軸の最小値をセットします(初期値:自動計算)。
	 *
	 * 何も指定しない場合は、データの最小値から、自動的に計算します。
	 *
	 * @param   lowBnd 縦軸の最小値
	 * @see		#setUpperBound( String )
	 */
	public void setLowerBound( final String lowBnd ) {
		if( lowBnd != null && lowBnd.length() > 0 ) {
			lowerBound = lowBnd;
		}
	}

	/**
	 * チャートの縦軸の最大値をセットします(初期値:自動計算)。
	 *
	 * 何も指定しない場合は、データの最大値から、自動的に計算します。
	 *
	 * @param   upBnd 縦軸の最大値
	 * @see		#setLowerBound( String )
	 */
	public void setUpperBound( final String upBnd ) {
		if( upBnd != null && upBnd.length() > 0 ) {
			upperBound = upBnd;
		}
	}

	/**
	 * チャートの縦軸の目盛の幅をセットします(初期値:自動計算)。
	 *
	 * 何も指定しない場合は、データの範囲から、自動的に計算します。
	 *
	 * @og.rev 4.0.3.0 (2008/01/07) 新規追加
	 *
	 * @param   size 縦軸のチック(目盛)の幅
	 */
	public void setTickSize( final String size ) {
		if( size != null && size.length() > 0 ) {
			tickSize = size;
		}
	}

	/**
	 * バーチャートのグラデーション処理を行うかどうか[true/false]をセットします(初期値:true)。
	 *
	 * 通常のバーチャートは、単一色表示で表されますが、これにグラデーション効果を
	 * 付加するかどうかを指定します。
	 * 通常のバーチャートが、少しきれいに見えます。
	 * 初期値は、true(グラデーション効果を付与する)です。
	 *
	 * @param   useGrad グラデーション処理 [true:する/false:しない]
	 */
	public void setUseGradient( final boolean useGrad ) {
		useGradient = useGrad;
	}

	/**
	 * バーチャートのグラデーション処理を行うかどうかを取得します。
	 *
	 * 通常のバーチャートは、単一色表示で表されますが、これにグラデーション効果を
	 * 付加するかどうかを指定します。
	 * 通常のバーチャートが、少しきれいに見えます。
	 * 初期値は、false(使用しない)です。
	 *
	 * @return	グラデーション処理(する:true/しない:false)
	 * @see     #setUseGradient( boolean )
	 */
	protected boolean isUseGradient() { return useGradient; }

	/**
	 * ラインチャートのポイントを四角表示するかどうかを指定します(初期値:false)。
	 *
	 * ラインチャートは、通常、線分で表され、各ポイントについても、線分で接続されます。
	 * shapesVisible を true に設定すると、各ポイントが、線上に四角く表示され、
	 * そのポイントの位置を、容易に判断出来るようになります。
	 * 初期値は、マーカーを入れません。
	 *
	 * @param   shVisible ポイントを四角表示するかどうか
	 */
	public void setShapesVisible( final boolean shVisible ) {
		shapesVisible = shVisible;
	}

	/**
	 * ラインチャートのポイントを四角表示するかどうかを取得します。
	 *
	 * ラインチャートは、通常、線分で表され、各ポイントについても、線分で接続されます。
	 * shapesVisible を true に設定すると、各ポイントが、線上に四角く表示され、
	 * そのポイントの位置を、容易に判断出来るようになります。
	 * 初期値は、マーカーを入れません。
	 *
	 * @return	ポイントを四角表示するかどうか
	 * @see     #setShapesVisible( boolean )
	 */
	protected boolean isShapesVisible() { return shapesVisible; }

	/**
	 * データ毎にShapeを切り替える時の色の繰返しパターンを文字列配列で指定します。
	 *
	 * HybsLine でのみ使用可能です。
	 * これは、データそのものが、繰返し性のある場合に、その繰返し性に対応した
	 * 形状のShape を表示させる場合に使用します。
	 * 繰返しShapeの形状は、JFreeChart のシリーズ毎の繰返し標準形状を使用します。
	 * 現在のバージョンでは、１０個までの繰返しに対応可能です。
	 * 繰返し色を、指定した分だけ、順に使用されていきます。
	 *
	 * 指定文字列は、java.awt.Color クラスのstatic フィールド名で指定します。
	 * BLACK , BLUE , CYAN , DARK_GRAY , GRAY , GREEN , LIGHT_GRAY ,
	 * MAGENTA , ORANGE , PINK , RED , WHITE , YELLOW , (PURPLE) が指定できます。
	 * また、#XXXXXX形式の16bitRGB表記 でも指定可能です。
	 *
	 * @og.rev 4.0.3.0 (2008/01/07) 新規追加
	 *
	 * @param	colors	データ毎の色の繰返しパターン(文字列配列)
	 * @see		java.awt.Color#BLACK
	 */
	public void setShapeColors( final String[] colors ) {
		if( colors != null && colors.length > 0 ) {
			int size = colors.length;
			shapeColors = new Color[size];
			for( int i=0; i<size; i++ ) {
				shapeColors[i] = StringUtil.getColorInstance( colors[i] );
			}
		}
	}

	/**
	 * shapeの大きさを倍率指定で変更します(初期値:null)。
	 *
	 * ラインチャートのShape(各グラフのポイントのマーカー)の大きさは、通常は、
	 * 自動設定されます。
	 * この大きさを、倍率指定で、変更可能です。
	 * 指定は、double 型です。
	 * 初期値は、null は、スケール変更しません(自動設定のままの大きさ)
	 *
	 * @og.rev 4.1.1.0 (2008/02/04) 新規追加
	 *
	 * @param	scale	shapeの大きさの倍率
	 */
	public void setShapeScale( final String scale ) {
		shapeScale = scale;
	}

	/**
	 * ラインチャートの線をドットラインにするかどうかを指定します(初期値:false)。
	 *
	 * ラインチャートは、通常、線分で表されます。
	 * これに、点線で表すことで、グラフの違いを、色だけでなく形状でも識別
	 * しやすくすることが可能です。
	 * 初期値は、線分です。
	 *
	 * @param   useDotLine ドットラインにするかどうか
	 */
	public void setUseDottedLine( final boolean useDotLine ) {
		useDottedLine = useDotLine;
	}

	/**
	 * ラインチャートの線をドットラインにするかどうかを取得します。
	 *
	 * ラインチャートは、通常、線分で表されます。
	 * これに、点線で表すことで、グラフの違いを、色だけでなく形状でも識別
	 * しやすくすることが可能です。
	 * 初期値は、線分です。
	 *
	 * @return	ドットラインにするかどうか
	 * @see     #setUseDottedLine( boolean )
	 */
	protected boolean isUseDottedLine() { return useDottedLine; }

	/**
	 * 複数チャート描画時のチャート色の繰返しパターンを文字列配列で指定します。
	 *
	 * 通常、複数のチャートを同時に表示させる場合は、縦軸が共通であれば、
	 * １回のSelect分で複数データを取得します。
	 * この、データをシリーズと呼んでおり、これを区別する為に、色を分けます。
	 * 初期値は、JFreeChart が自動で割り振ります。
	 * これを、外部からCVS形式で、カンマ区切りで色コードを指定します。
	 * 指定データが多い場合は、多い分の色は使用されません。少ない場合は、
	 * 順番に繰り返して使用されます。
	 * 例えば、１色だけ指定した場合は、すべてのシリーズが同じ色で表されます。
	 *
	 * 指定文字列は、java.awt.Color クラスのstatic フィールド名で指定します。
	 * BLACK , BLUE , CYAN , DARK_GRAY , GRAY , GREEN , LIGHT_GRAY ,
	 * MAGENTA , ORANGE , PINK , RED , WHITE , YELLOW , (PURPLE) が指定できます。
	 * また、#XXXXXX形式の16bitRGB表記 でも指定可能です。
	 *
	 * @param	colors	チャート色の繰返しパターン(文字列配列)
	 * @see		java.awt.Color#BLACK
	 */
	public void setSeriesColors( final String[] colors ) {
		if( colors != null && colors.length > 0 ) {
			int size = colors.length;
			seriesColors = new Color[size];
			for( int i=0; i<size; i++ ) {
				seriesColors[i] = StringUtil.getColorInstance( colors[i] );
			}
		}
	}

	/**
	 * 複数チャート描画時のチャート色の繰返しパターンをColor配列で指定します。
	 *
	 * @return	チャート色の繰返しパターン(Color配列)
	 * @see		#setSeriesColors( String[] )
	 */
	protected Color[] getSeriesColors() {
		return seriesColors ;
	}

	/**
	 * Value(縦軸)のラベルを表示するかどうかを指定します(初期値:true[表示する])。
	 *
	 * ValueAxis にて設定される、縦軸情報の、ラベルを表示するかどうか指定します。
	 * 初期値は、true(表示する)です。
	 *
	 * @og.rev 4.0.2.0 (2007/12/20) 新規追加
	 *
	 * @param	labelsVisible	Value(縦軸)のラベルを表示するかどうか
	 */
	public void setValueLabelsVisible( final boolean labelsVisible ) {
		isLabelsVisible = labelsVisible;
	}

	/**
	 * Value(縦軸)のマーカーを表示するかどうかを指定します(初期値:true[表示する])。
	 *
	 * ValueAxis にて設定される、縦軸情報の、マーカーを表示するかどうか指定します。
	 * 初期値は、true(表示する)です。
	 *
	 * @og.rev 4.0.2.0 (2007/12/20) 新規追加
	 *
	 * @param	marksVisible	Value(縦軸)のマーカーを表示するかどうか
	 */
	public void setValueMarksVisible( final boolean marksVisible ) {
		isMarksVisible = marksVisible;
	}

	/**
	 * データの値(itemText)を表示するかどうか[true/false/last]を指定します(初期値:false[表示しない])。
	 *
	 * CategoryItemRenderer 関連のグラフの設定値をグラフ上に表示するかどうかを
	 * 指定します。
	 * true に設定した場合、通常の場合は、すべてのシリーズにラベル表示されます。
	 * false に設定すると、表示されません。
	 * last を設定すると、各シリーズの最後の値のみ表示されます。
	 * ChartCreate クラスに、seriesPikup が設定されている場合は、指定のシリーズ
	 * のみの設定値を表示し、他の値は、表示しません。
	 * 同様に、dynamicOCNo が指定されている場合(動的なマーカーライン)
	 * 指定のシリーズは、閾値として使用されるため、設定値は表示されません。
	 * ラベルの表示位置は、表示する線グラフの傾きに応じてラベルの表示場所を
	 * 変えます。山形、右坂、谷形、左坂 に応じて、上中、下右、下中、上右 に
	 * 位置を設定します。右にずらすのは、10 ピクセル固定です。
	 * 初期値は、false(表示しない)です。
	 *
	 * @og.rev 4.1.1.0 (2008/02/04) 新規追加
	 * @og.rev 4.1.2.0 (2008/03/12) 文字列のまま、受け取りるように変更
	 *
	 * @param	visible	データの値の表示のさせ方 [true/false/last]
	 * @see     ChartCreate#setSeriesPikup( int )
	 */
	public void setItemLabelVisible( final String visible ) {
		if( "true".equalsIgnoreCase( visible ) ) {
			isItemValVisible = true;
		}
		else if( "last".equalsIgnoreCase( visible ) ) {
			isItemValVisible  = true;
			isItemLastVisible = true;
		}
		else if( visible != null && !"false".equalsIgnoreCase( visible ) ) {
			String errMsg = "itemLabelVisible は、「true,false,last」から指定してください。["
							+ visible + "]";
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * データの値(itemText)の表示に桁区切り文字を使用するかどうかを指定します。
	 *
	 * itemLabelVisible=true 時に、表示されるデータ値ラベルで、
	 * NumberFormat していますが、３桁区切り文字(123,456,789.0) の
	 * 区切り記号を表示するかどうかを指定します。
	 * true を指定すると、表示します。false では、表示しません。
	 * 初期値は、true(使用する)です。
	 *
	 * @og.rev 4.1.2.0 (2008/03/12) 新規追加
	 *
	 * @param	used	桁区切り文字を使用するかどうか
	 */
	public void setUseItemLabelSep( final boolean used ) {
		useItemLabelSep = used;
	}

	/**
	 * 縦軸の表示領域INSET値(double)を指定します。
	 *
	 * 縦軸文字表示領域(NumberAxis)の幅の追加値を設定します。
	 * これは、通常の表示領域の左側にスペースを挿入します。
	 * あくまで、追加する値なので、文字の長さは含まれません。
	 * 何も指定しない場合は、設定しません。
	 *
	 * @og.rev 4.0.3.0 (2008/01/07) 新規追加
	 *
	 * @param	inset	Value(縦軸)の表示領域INSET値
	 */
	public void setValueInset( final String inset ) {
		valueInset = inset;
	}

	/**
	 * BOXチャートのバー幅(double)を指定します。
	 *
	 * BOXチャートのバー幅(double)を指定します。
	 * 設定されるのは、org.jfree.chart.renderer.category.BarRenderer を
	 * 使用した描画のみです。
	 * 棒グラフのバー幅指定は、直接、CategoryItemRendererState に設定しています。
	 * 通常は、barMaxWidth(0.0～1.0)とbarItemMargin(0.0～1.0)を用いて比率で指定します。
	 * 何も指定しない場合は、設定しません。
	 *
	 * @og.rev 4.0.3.0 (2008/01/07) 新規追加
	 *
	 * @param	width	BOXチャートのバー幅(double)
	 */
	public void setBarWidth( final String width ) {
		barWidth = width;
	}

	/**
	 * BOXチャートのバー幅(double)を取得します。
	 *
	 * 設定されるのは、org.jfree.chart.renderer.category.BarRenderer を
	 * 使用した描画のみです。
	 * 棒グラフのバー幅指定は、直接、CategoryItemRendererState に設定しています。
	 * 通常は、barMaxWidth(0.0～1.0)とbarItemMargin(0.0～1.0)を用いて比率で指定します。
	 * 何も指定しない場合は、設定しません。
	 *
	 * @og.rev 4.0.3.0 (2008/01/07) 新規追加
	 *
	 * @return	BOXチャートのバー幅(double)の文字列
	 */
	protected String getBarWidth() {
		return barWidth ;
	}

	/**
	 * BOXチャートのバー幅の最大値(0.0～1.0)を指定します。
	 *
	 * BOXチャートのバー幅の比率の最大値を指定します。
	 * 表示領域を１として小数点以下の数値で棒の幅を設定します。
	 * 設定した幅に無理がある時は適当なサイズに調整されます。
	 * (小さくしたときには棒が線のようになる)
	 * 設定されるのは、org.jfree.chart.renderer.category.BarRenderer を
	 * 使用した描画のみです。
	 * 何も指定しない場合は、設定しません。
	 *
	 * @og.rev 4.0.2.0 (2007/12/20) 新規追加
	 *
	 * @param	barWidth	BOXチャートのバー幅の最大値(0.0～1.0)
	 */
	public void setBarMaxWidth( final String barWidth ) {
		maxBarWidth = barWidth;
	}

	/**
	 * BOXチャートのバーアイテムのマージン(0.0～1.0)を指定します。
	 *
	 * BOXチャートのバーアイテムのマージンの比率を指定します。
	 * 棒の間を表示領域を１として小数点以下の数値で幅を設定します。
	 * 無理がある時は適当なサイズに調整されます。
	 * barMaxWidth より優先されます。
	 * 設定されるのは、org.jfree.chart.renderer.category.BarRenderer を
	 * 使用した描画のみです。
	 * 何も指定しない場合は、設定しません。
	 *
	 * @og.rev 4.0.2.0 (2007/12/20) 新規追加
	 *
	 * @param	margin	BOXチャートのバーアイテムのマージン(0.0～1.0)
	 */
	public void setBarItemMargin( final String margin ) {
		itemMargin = margin;
	}

	/**
	 * 表示下限値(これ以下のデータは未表示)の値(double)を指定します。
	 *
	 * HybsLine でのみ使用可能です。
	 * この設定値以下のデータは、存在しない扱いとします。
	 * Lineを引くとき、このデータと、存在しているデータ間にラインは引かれません。
	 * 何も指定しない場合は、設定しません。
	 *
	 * @og.rev 4.0.3.0 (2008/01/07) 新規追加
	 *
	 * @param	limit	表示下限値(これ以下のデータは未表示)
	 */
	public void setVisibleLimit( final String limit ) {
		visibleLimit = limit;
	}

	/**
	 * グラフの書き出し位置の調整比率を指定します。
	 *
	 * グラフを描画する場合の、書き出し位置を少しずらします。
	 * これは、グラフの幅に対して、比率で指定します。
	 * 0.0(初期値)の場合は、初期描画位置である、CategoryAnchor.Middle と
	 * 同じ箇所から、書き出されます。
	 * 1.0 の場合、中心から、グラフ幅の半分が加算され、END位置に寄ります。
	 * 同様に、-1.0 の場合は、グラフ幅の半分が減算され、START 位置になります。
	 * つまり、中心から、グラフ幅の半分単位で、前方/後方にずらす事が出来ます。
	 *   書き出し位置 ＝ 中心(Middle) + (domainMargin)＊幅/２
	 * 初期値は、0.0(真ん中:MIDDLE)です。
	 *
	 * @og.rev 4.1.1.0 (2008/02/14) 新規追加
	 *
	 * @param	margin	グラフの書き出し位置の調整比率
	 */
	public void setDomainMargin( final double margin ) {
		domainMargin = margin;
	}

	/**
	 * 時刻を表す場合の表現の仕方を指定します(初期値:null)。
	 *
	 * HybsNumberAxis にオーバーライドする 時間を表示する DecimalFormat の内部クラスを利用するに当たり、
	 * 時刻の表示方法を指定します。
	 * 外部から与える数字は、連続している必要があるため、１０進数です。
	 * たとえば、1700 → 17:00 , 2150 → 21:30 という感じです。
	 * 2400 を超えると日付違いになります。
	 *
	 * 英語表記(:)と日本語表記(時)の区別と、24時間を超える場合の表示方法によって、６種類のパターンが存在します。
	 * 　E1:そのまま、24:00 となり、加算されていく。
	 * 　E2:そのまま、0:00 に戻る。(日付は無視)
	 * 　E3:そのまま、1 00:00 と日付が付与される。
	 * 　H1:そのまま、24時00分 となり、加算されていく。
	 * 　H2:そのまま、00時00分 に戻る。(日付は無視)
	 * 　H3:そのまま、1日 00時00分 と日付が付与される。
	 * 初期値は、使用しない(-1)です。
	 *
	 * @og.rev 5.5.2.1 (2012/05/07) 新規追加
	 *
	 * @param	type	時刻の表示方法 [E1/E2/E3/H1/H2/H3]
	 */
	public void setTimeFormatType( final String type ) {
		timeFormatType = type;
	}

	/**
	 * ラベルの表示向きを縦にするかどうか[true/false]を指定します(初期値:false)。
	 *
	 * ChartCreate の rotationLabel は、角度を指定できましたが、NumberAxis では、
	 * 縦にするかどうかの指定しかできません。
	 * ここでは、true を指定するとラベルは、縦書きになります。
	 * 初期値は、false(横書き)です。
	 *
	 * @og.rev 5.5.2.1 (2012/05/07) 新規追加
	 *
	 * @param   useVLavels ラベルの表示向き [false:横書き/:true:縦書き]
	 */
	public void setUseVerticalLabels( final boolean useVLavels ) {
		useVerticalLabels = useVLavels;
	}

	/**
	 * HybsNumberAxis にオーバーライドする 時間を表示する DecimalFormat の内部クラスを定義します。
	 *
	 * これは、ChartPlot_XY で使用する、HybsNumberAxis に対して、拡張します。
	 * データが、時刻を表す場合、Date型のオブジェクトを利用して、org.jfree.chart.axis.PeriodAxis
	 * を利用できればよいのですが、単なる数字で扱っているケースがあり、時刻に内部的に
	 * 変換するケースで使います。
	 * 外部から与える数字は、連続している必要があるため、１０進数です。
	 * たとえば、1700 → 17:00 , 2150 → 21:30 という感じです。
	 * 24時間を超える場合は、
	 * Type1:そのまま、0:00 に戻る。
	 * Type2:そのまま、24:00 と、そのまま、加算されていく。
	 * の２種類が用意されています。
	 *
	 * @og.rev 5.5.2.1 (2012/05/07) 新規追加
	 *
	 */
	private static final class HybsTimeFormat extends DecimalFormat {
		/** For serialization. */
		private static final long serialVersionUID = 201205075512L;

		private final boolean is24repeat ;
		private final boolean useDay ;

		private final int hsCode ;

		/**
		 * コンストラクター
		 *
		 * 引数に、時刻を表す場合の表現の仕方を指定します。
		 * 　E1:そのまま、24:00 となり、加算されていく。
		 * 　E2:そのまま、0:00 に戻る。(日付は無視)
		 * 　E3:そのまま、1 00:00 と日付が付与される。
		 * 　H1:そのまま、24時00分 となり、加算されていく。
		 * 　H2:そのまま、00時00分 に戻る。(日付は無視)
		 * 　H3:そのまま、1日 00時00分 と日付が付与される。
		 *
		 * @param	type	時刻の表示方法 [E1/E2/E3/H1/H2/H3]
		 */
		public HybsTimeFormat( final String inType ) {
			super();

			DecimalFormatSymbols sym = new DecimalFormatSymbols();
			if( "E1".equals( inType ) ) {
				applyPattern( "#0.00" );
				sym.setDecimalSeparator( ':' );
				is24repeat = false;		useDay = false;
			}
			else if( "E2".equals( inType ) ) {
				applyPattern( "#0.00" );
				sym.setDecimalSeparator( ':' );
				is24repeat = true;		useDay = false;
			}
			else if( "E3".equals( inType ) ) {
				applyPattern( "#,#0.00" );
				sym.setDecimalSeparator( ':' );
				sym.setGroupingSeparator( ' ' );
				is24repeat = true;		useDay = true;
			}
			else if( "H1".equals( inType ) ) {
				applyPattern( "#0.00分" );
				sym.setDecimalSeparator( '時' );
				is24repeat = false;		useDay = false;
			}
			else if( "H2".equals( inType ) ) {
				applyPattern( "#0.00分" );
				sym.setDecimalSeparator( '時' );
				is24repeat = true;		useDay = false;
			}
			else if( "H3".equals( inType ) ) {
				applyPattern( "#,#0.00分" );
				sym.setDecimalSeparator( '時' );
				sym.setGroupingSeparator( '日' );
				is24repeat = true;		useDay = true;
			}
			else {
				String errMsg = "timeFormatType は、(E1/E2/E3/H1/H2/H3) から指定してください。timeFormatType=[" + inType + "]" ;
				throw new HybsSystemException( errMsg );
			}
			setDecimalFormatSymbols( sym );

			hsCode = super.hashCode() + Boolean.valueOf(is24repeat).hashCode() + Boolean.valueOf(useDay).hashCode();
		}

		/**
		 * double をフォーマットして文字列を作成します( format(double,StringBuffer,FieldPosition) のオーバーライド )
		 *
		 * timeFormatType によって、データの処理が変わります。
		 * 　E1:そのまま、24:00 となり、加算されていく。
		 * 　E2:そのまま、0:00 に戻る。(日付は無視)
		 * 　E3:そのまま、1 00:00 と日付が付与される。
		 * 　H1:そのまま、24時00分 となり、加算されていく。
		 * 　H2:そのまま、00時00分 に戻る。(日付は無視)
		 * 　H3:そのまま、1日 00時00分 と日付が付与される。
		 *
		 * @param	number	フォーマットする double
		 * @param	result	テキストを追加する位置
		 * @param	fieldPos	入力では、必要であれば位置合わせフィールド。 出力では、その位置合わせフィールドのオフセット
		 * @return	フォーマットされた数値文字列
		 */
		public StringBuffer format( final double number, final StringBuffer result, final FieldPosition fieldPos ) {
			int day    = (int)(number/100/24);			// 日付。端数は切り捨てる。
			int hour   = (int)(number/100);				// 時間のみ取り出す。
			if( is24repeat ) { hour = hour%24; }		// ２４時間を超えた場合、余りのみ使用する。

			int minute = (int)((number%100)*0.6);		// 0.6 = 60/100 の事。100 が 60分 となる。

			double timeVal = hour + minute/100.0 ;
			if( useDay ) { timeVal = timeVal + day*100 ; }	// 端数を切り捨てた日付に対して、100倍しておく。
			return super.format( timeVal, result, fieldPos );
		}

		/**
		 * このオブジェクトと指定されたオブジェクトを比較します。
		 *
		 * @og.rev 5.5.2.4 (2012/05/16) 新規追加 findbug対応
		 *
		 * @param	anObject	比較されるオブジェクト
		 * @return	指定されたオブジェクトが等しい場合は true、そうでない場合は false
		 */
		public boolean equals( final Object anObject ) {
			return this == anObject;
		}

		/**
		 * このオブジェクトのハッシュコードを取得します。
		 *
		 * @og.rev 5.5.2.4 (2012/05/16) 新規追加 findbug対応
		 *
		 * @return	ハッシュコード
		 */
		public int hashCode() {
			return hsCode;
		}
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	@Override
	public String toString() {
		StringBuilder rtn = new StringBuilder( HybsSystem.BUFFER_MIDDLE );

		rtn.append( "[" ).append( this.getClass().getName() ).append( "]" ).append( HybsSystem.CR );
		rtn.append( "chartType     [" ).append( chartType     ).append( "]" ).append( HybsSystem.CR );
		rtn.append( "valueLabel    [" ).append( valueLabel    ).append( "]" ).append( HybsSystem.CR );
		rtn.append( "markValues    [" ).append( Arrays.toString( markValues ) ).append( "]" ).append( HybsSystem.CR );
		rtn.append( "markColors    [" ).append( Arrays.toString( markColors ) ).append( "]" ).append( HybsSystem.CR );
		rtn.append( "lowerBound    [" ).append( lowerBound    ).append( "]" ).append( HybsSystem.CR );
		rtn.append( "upperBound    [" ).append( upperBound    ).append( "]" ).append( HybsSystem.CR );
		rtn.append( "useGradient   [" ).append( useGradient   ).append( "]" ).append( HybsSystem.CR );
		rtn.append( "shapesVisible [" ).append( shapesVisible ).append( "]" ).append( HybsSystem.CR );
		rtn.append( "useDottedLine [" ).append( useDottedLine ).append( "]" ).append( HybsSystem.CR );

		return rtn.toString();
	}
}
