/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.io;

import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;

/**
 * POI による、EXCELバイナリファイルに対する、ユーティリティクラスです。
 *
 * ここでは、アクティブセル領域になるように、不要なセルや行を削除します。
 *
 * 入力形式は、openXML形式にも対応しています。
 * ファイルの内容に応じて、.xlsと.xlsxのどちらで読み取るかは、内部的に
 * 自動判定されます。
 *
 * @og.rev 5.5.7.2 (2012/10/09) 新規作成
 * @og.group その他
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class ExcelUtil {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.5.7.2 (2012/10/09)" ;

	/**
	 * オブジェクトを作らせない為の、private コンストラクタ
	 */
	private ExcelUtil() {}

	/**
	 * DBTableModel から 各形式のデータを作成して,BufferedReader より読み取ります。
	 * コメント/空行を除き、最初の行は、必ず項目名が必要です。
	 * それ以降は、コメント/空行を除き、データとして読み込んでいきます。
	 * このメソッドは、EXCEL 読み込み時に使用します。
	 *
	 * @og.rev 5.5.7.2 (2012/10/09) 新規追加
	 *
	 * @param	wb	処理対象のワークブック
	 * @return	アクティブ領域のみに再設定された、ワークブック
	 */
	public static Workbook activeWorkbook( final Workbook wb ) {
		int sheetSu = wb.getNumberOfSheets();
		for( int shno = 0; shno<sheetSu; shno++ ) {
			Sheet sheet = wb.getSheetAt(shno);

			int nFirstRow = sheet.getFirstRowNum();
			int nLastRow  = sheet.getLastRowNum();

			for( int nIndexRow = nFirstRow; nIndexRow <= nLastRow; nIndexRow++) {
				Row oRow = sheet.getRow(nIndexRow);
				if( oRow != null ) {
					boolean isAllBrank = true;
					int nFirstCell = oRow.getFirstCellNum();
					int nLastCell  = oRow.getLastCellNum();
					for( int nIndexCell = nFirstCell; nIndexCell <= nLastCell; nIndexCell++) {
						Cell oCell = oRow.getCell(nIndexCell);
						if( oCell != null && oCell.getCellType() != Cell.CELL_TYPE_BLANK ) { 
							isAllBrank = false;	
							break;
						}
					}
					if( isAllBrank ) { sheet.removeRow( oRow ); }
				}
			}
		}
		return wb ;
	}
}
