/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.mail;

/**
 * unicode と、JIS との文字コードの関係で、変換しています。
 * <pre>
 * http://www.ingrid.org/java/i18n/encoding/ja-conv.html
 *
 * 0x00a2 ⇒ 0xffe0	 ￠ (1-81, CENT SIGN)
 * 0x00a3 ⇒ 0xffe1	 ￡ (1-82, POUND SIGN)
 * 0x00a5 ⇒ 0x005c	 \ (D/12, YEN SIGN)
 * 0x00ac ⇒ 0xffe2	 ￢ (2-44, NOT SIGN)
 * 0x2016 ⇒ 0x2225	 ∥ (1-34, DOUBLE VERTICAL LINE)
 * 0x203e ⇒ 0x007e	 ~ (F/14, OVERLINE)
 * 0x2212 ⇒ 0xff0d	 － (1-61, MINUS SIGN)
 * 0x301c ⇒ 0xff5e	 ～ (1-33, WAVE DASH)
 *
 * それぞれコード変換します。
 * </pre>
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
final class UnicodeCorrecter {

	/**
	 * インスタンスの生成を抑止します。
	 */
	private UnicodeCorrecter() {
		// 何もありません。(PMD エラー回避)
	}

	/**
	 * Unicode 文字列の補正を行います。
	 * "MS932" コンバータでエンコードしようとした際に
	 * 正常に変換できない部分を補正します。
	 *
	 * @param	str 入力文字列
	 * @return	Unicode 文字列の補正結果
	 */
	public static String correctToCP932( final String str ) {
		String rtn = "";

		if( str != null ) {
			final int cnt = str.length();
			final StringBuilder buf = new StringBuilder( cnt );
			for(int i=0; i<cnt; i++) {
				buf.append(correctToCP932(str.charAt(i)));
			}
			rtn = buf.toString() ;
		}
		return rtn ;
	}

	/**
	 * キャラクタ単位に、Unicode 文字の補正を行います。
	 *
	 * 風間殿のページを参考にしています。
	 * @see <a href="http://www.ingrid.org/java/i18n/encoding/ja-conv.html" target="_blank">
	 * http://www.ingrid.org/java/i18n/encoding/ja-conv.html</a>
	 *
	 * @param	ch 入力文字
	 * @return	Unicode 文字の補正結果
	 */
	public static char correctToCP932( final char ch ) {
		char rtn = ch;

		switch (ch) {
	//		case 0x00a2:	return 0xffe0;		// ≪
	//		case 0x00a3:	return 0xffe1;		// ￣
	//		case 0x00ac:	return 0xffe2;		// μ
	//		case 0x03bc:	return 0x00b5;		// ・
	//		case 0x2014:	return 0x2015;		// ，
	//		case 0x2016:	return 0x2225;		// ≫
	//		case 0x2212:	return 0xff0d;		// ―
	//		case 0x226a:	return 0x00ab;		// ∥
	//		case 0x226b:	return 0x00bb;		// ヴ
	//		case 0x301c:	return 0xff5e;		// －
	//		case 0x30f4:	return 0x3094;		// ～
	//		case 0x30fb:	return 0x00b7;		// ￠
	//		case 0xff0c:	return 0x00b8;		// ￡
	//		case 0xffe3:	return 0x00af;		// ￢

			case 0x00a2:	rtn = 0xffe0; break;		// ￠ (1-81, CENT SIGN)
			case 0x00a3:	rtn = 0xffe1; break;		// ￡ (1-82, POUND SIGN)
			case 0x00a5:	rtn = 0x005c; break;		// \ (D/12, YEN SIGN)
			case 0x00ac:	rtn = 0xffe2; break;		// ￢ (2-44, NOT SIGN)
			case 0x2016:	rtn = 0x2225; break;		// ∥ (1-34, DOUBLE VERTICAL LINE)
			case 0x203e:	rtn = 0x007e; break;		// ~ (F/14, OVERLINE)
			case 0x2212:	rtn = 0xff0d; break;		// － (1-61, MINUS SIGN)
			case 0x301c:	rtn = 0xff5e; break;		// ～ (1-33, WAVE DASH)

	//		case 0x301c:	return 0xff5e;
	//		case 0x2016:	return 0x2225;
	//		case 0x2212:	return 0xff0d;
			default:		break;			// 4.0.0 (2005/01/31)
		}
		return rtn;
	}
}
