/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

// import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.fukurou.util.LogWriter;
import static org.opengion.fukurou.util.HybsConst.CR ;				// 6.1.0.0 (2014/12/26)
import static org.opengion.fukurou.util.HybsConst.BUFFER_MIDDLE;	// 6.1.0.0 (2014/12/26) refactoring

/**
 * データのコード情報を取り扱うクラスです。
 *
 * 開始、終了、ステップの情報から、HTMLのメニューやリストを作成するための オプション
 * タグを作成したり、与えられたキーをもとに、チェック済みのオプションタグを作成したりします。
 * 
 * ここでは、数字（連番）の自動生成を行います。パラメータで、開始、終了、ステップを指定します。
 * パラメータの初期値は、開始(1)、終了(10)、ステップ(1) です。
 * 
 * 例：1,10,1    → 1,2,3,4,5,6,7,8,9,10 のプルダウン
 * 例：10,100,10 → 10,20,30,40,50,60,70,80,90,100 のプルダウン
 * 例：-5,5,1    → -5,-4,-3,-2,-1,0,1,2,3,4,5 のプルダウン
 * 例：5,-5,-2   → 5,3,1,-1,-3,-5 のプルダウン
 *
 * @og.group 選択データ制御
 * @og.rev 5.6.1.1 (2013/02/08) 新規追加
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
// public class Selection_NUM implements Selection {
public class Selection_NUM extends Selection_NULL {
	private final String   CACHE ;
	private final String   ST_ED_STEP ;

	/**
	 * コンストラクター
	 *
	 * 引数は、開始、終了、ステップです。
	 * パラメータの初期値は、開始(1)、終了(10)、ステップ(1) です。
	 *
	 * @param	editPrm	開始、終了、[ステップ]を表す引数(例：1,10,1)
	 */
	public Selection_NUM( final String editPrm ) {
	//	if( param.length < 2 ) {
	//	final String errMsg = "引数は、開始、終了、[ステップ] です。最低でも２個必要です。";
	//		throw new IllegalArgumentException( errMsg );
	//	}

		final String[] param = (editPrm == null) ? new String[0] : editPrm.split( "," ) ;

		final int start = (param.length > 0) ? Integer.parseInt( param[0].trim() ) : 1;
		final int end   = (param.length > 1) ? Integer.parseInt( param[1].trim() ) : 10 ;
		final int step  = (param.length > 2) ? Integer.parseInt( param[2].trim() ) : 1;

		if( step == 0 ) {
			final String errMsg = "ステップ に ０ は指定できません。無限ループします。";
			throw new IllegalArgumentException( errMsg );
		}

		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );

		// ステップの正負による判定の違い。while( Math.signum( end-start ) * step >= 0.0 ) で、判る？
		// 終了条件は、含む(val<=end)
		int val  = start;
		final int sign = ( step > 0 ) ? 1 : -1 ;	// ステップの符号。
		while( (end - val) * sign >= 0 ) {
			buf.append( "<option value=\"" ).append( val )
				.append( "\">" ).append( val ).append( "</option>" );		// 6.0.2.5 (2014/10/31) char を append する。
			val += step;
		}

		CACHE = buf.toString();
		ST_ED_STEP = "Start=" + start + " , End=" + end + " , Step=" + step ;
	}

//	/**
//	 * 初期値が選択済みの 選択肢(オプション)を返します。
//	 * このオプションは、引数の値を初期値とするオプションタグを返します。
//	 * このメソッドでは、ラベル(短)が設定されている場合でも、これを使用せずに必ずラベル(長)を使用します。
//	 *
//	 * @og.rev 5.7.7.1 (2014/06/13) Selection_NULL を 継承するため、削除
//	 *
//	 * @param   selectValue  選択されている値
//	 * @param   seqFlag  シーケンスアクセス機能 [true:ON/false:OFF]
//	 *
//	 * @return  オプションタグ
//	 * @see     #getOption( String, boolean, boolean )
//	 */
//	public String getOption( final String selectValue,final boolean seqFlag ) {
//		return getOption( selectValue, seqFlag, false );
//	}

	/**
	 * 初期値が選択済みの 選択肢(オプション)を返します。
	 * このオプションは、引数の値を初期値とするオプションタグを返します。
	 * このメソッドでは、引数のuseShortLabelがtrueに指定された場合に、ラベル(短)をベースとした
	 * ツールチップ表示を行います。
	 *
	 * @param   selectValue  選択されている値
	 * @param   seqFlag  シーケンスアクセス機能 [true:ON/false:OFF]
	 * @param   useShortLabel ラベル(短)をベースとしたオプション表示を行うかどうか。(未使用)
	 *
	 * @return  オプションタグ
	 * @og.rtnNotNull
	 */
	@Override
	public String getOption( final String selectValue,final boolean seqFlag, final boolean useShortLabel ) {
		// マッチするアドレスを探す。
		final int selected = CACHE.indexOf( "\"" + selectValue + "\"" );

		if( selected < 0 ) {
			if( selectValue != null && selectValue.length() > 0 ) {
				final String errMsg = "数字範囲に存在しない値が指定されました。"
							+ " value=[" + selectValue + "]"
							+ CR + ST_ED_STEP ;
				LogWriter.log( errMsg );
			}
			return CACHE;
		}
		else {
			// "値" 文字列の位置が、selected なので、値の文字数＋２までが、前半部分になる。
			final int indx = selected + selectValue.length() + 2 ;

			final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
			// 3.6.0.6 (2004/10/22) シーケンスアクセス機能を指定する seqFlag を導入
			if( seqFlag ) {
				buf.append( "<option value=\"" ).append( selectValue ).append( '"' );		// 6.0.2.5 (2014/10/31) char を append する。
			}
			else {
				buf.append( CACHE.substring( 0,indx ) );
			}
			buf.append( " selected=\"selected\"" )
				.append( CACHE.substring( indx ) );
			return buf.toString() ;
		}
	}

//	/**
//	 * 初期値が選択済みの 選択肢(オプション)を返します。
//	 * このオプションは、引数の値を初期値とするオプションタグを返します。
//	 * ※ このクラスでは実装されていません。
//	 *
//	 * @og.rev 2.1.0.1 (2002/10/17) 選択リストを、正方向にしか選べないようにする sequenceFlag を導入する
//	 * @og.rev 3.8.6.0 (2006/09/29) useLabel 属性 追加
//	 * @og.rev 5.7.7.1 (2014/06/13) Selection_NULL を 継承するため、削除
//	 *
//	 * @param   name         ラジオの name
//	 * @param   selectValue  選択されている値
//	 * @param   useLabel     ラベル表示の有無 [true:有/false:無]
//	 *
//	 * @return  オプションタグ
//	 */
//	public String getRadio( final String name,final String selectValue,final boolean useLabel ) {
//	final String errMsg = "このクラスでは実装されていません。";
//		throw new UnsupportedOperationException( errMsg );
//	}

//	/**
//	 * 初期値が選択済みの 選択肢(オプション)を返します。
//	 * このオプションは、引数の値を初期値とするオプションタグを返します。
//	 * ※ このクラスでは実装されていません。
//	 *
//	 * @og.rev 5.7.7.1 (2014/06/13) Selection_NULL を 継承するため、削除
//	 *
//	 * @param   selectValue  選択されている値
//	 *
//	 * @return  オプションタグ
//	 */
//	public String getRadioLabel( final String selectValue ) {
//	final String errMsg = "このクラスでは実装されていません。";
//		throw new UnsupportedOperationException( errMsg );
//	}

//	/**
//	 * 選択肢(value)に対するラベルを返します。
//	 * 選択肢(value)が、存在しなかった場合は、選択肢そのものを返します。
//	 * getValueLabel( XX ) は、getValueLabel( XX,false ) と同じです。
//	 *
//	 * @og.rev 5.7.7.1 (2014/06/13) Selection_NULL を 継承するため、削除
//	 *
//	 * @param   selectValue 選択肢の値
//	 *
//	 * @return  選択肢のラベル
//	 * @see     #getValueLabel( String,boolean )
//	 */
//	public String getValueLabel( final String selectValue ) {
//		return getValueLabel( selectValue,false );
//	}

	/**
	 * 選択肢(value)に対するラベルを返します。
	 * 選択肢(value)が、存在しなかった場合は、選択肢そのものを返します。
	 * このメソッドでは、短縮ラベルを返すかどうかを指定するフラグを指定します。
	 * getValueLabel( XX,false ) は、getValueLabel( XX ) と同じです。
	 *
	 * @og.rev 4.0.0.0 (2005/11/30) を追加
	 *
	 * @param	selectValue	選択肢の値
	 * @param	flag	短縮ラベルを [true:使用する/false:しない] (未使用)
	 *
	 * @return  選択肢のラベル
	 * @see     #getValueLabel( String )
	 */
	@Override
	public String getValueLabel( final String selectValue,final boolean flag ) {
		// あろうがなかろうが、選択肢そのものを返します。
		return selectValue;
	}

//	/**
//	 * マルチ・キーセレクトを使用するかどうかを返します(false固定)。
//	 * true：使用する。false:使用しない です。
//	 * ただし、実際に使用するかどうかは、HTML出力時に決めることが出来ます。
//	 * ここでは、USE_MULTI_KEY_SELECT が true で、USE_SIZE(=20)以上の場合に
//	 * true を返します。
//	 *
//	 * @og.rev 3.5.5.7 (2004/05/10) 新規作成
//	 * @og.rev 6.0.4.0 (2014/11/28) Selection_NULL を 継承するため、削除
//	 *
//	 * @return  選択リストで、マルチ・キーセレクトを使用するかどうか(true:使用する) (false固定)
//	 */
//	@Override
//	public boolean useMultiSelect() {
//		return true;
//	}

//	/**
//	 * オブジェクトのキャッシュが時間切れかどうかを返します。
//	 * キャッシュが時間切れ(無効)であれば、true を、有効であれば、
//	 * false を返します。
//	 *
//	 * @og.rev 4.0.0.0 (2005/01/31) 新規作成
//	 * @og.rev 5.7.7.1 (2014/06/13) Selection_NULL を 継承するため、削除
//	 *
//	 * @return  キャッシュが時間切れなら true
//	 */
//	public boolean isTimeOver() {
//		return false;
//	}
}
